"use strict";
var util = require('util');
var utils = require('../utils');
/** @module types */

/**
 * Creates a new instance of ResultSet
 * @class
 * @classdesc Represents the result of a query
 * @param {Object} response
 * @param {String} host
 * @param {Object} triedHosts
 * @param {Number} consistency
 * @constructor
 */
function ResultSet(response, host, triedHosts, consistency) {
  /**
   * Information on the execution of a successful query:
   * @member {Object}
   * @property {Number} achievedConsistency The consistency level that has been actually achieved by the query.
   * @property {String} queriedHost The Cassandra host that coordinated this query.
   * @property {Object} triedHosts Gets the associative array of host that were queried before getting a valid response,
   * being the last host the one that replied correctly.
   * @property {Uuid} traceId Identifier of the trace session.
   * @property {Array.<string>} warnings Warning messages generated by the server when executing the query.
   */
  this.info = {
    queriedHost: host,
    triedHosts: triedHosts,
    achievedConsistency: consistency,
    traceId: null,
    warnings: null,
    customPayload: null
  };
  if (response.flags) {
    this.info.traceId = response.flags.traceId;
    this.info.warnings = response.flags.warnings;
    this.info.customPayload = response.flags.customPayload;
  }
  var self = this;
  /**
   * Gets an array rows returned by the query, in case the result was buffered.
   * @member {Array.<Row>}
   */
  this.rows = response.rows;
  /**
   * Gets the row length of the result, regardless if the result has been buffered or not
   * @member {Number}
   */
  this.rowLength = this.rows ? this.rows.length : response.rowLength;
  /**
   * Gets the columns returned in this ResultSet.
   * @member Array
   */
  this.columns = !response.meta ? null : response.meta.columns;
  //Define meta as a private (not enumerable) property, for backward compatibility
  Object.defineProperty(this, 'meta', { value: response.meta, enumerable: false, writable: false });
  /**
   * A string token representing the current page state of query. It can be used in the following executions to
   * continue paging and retrieve the remained of the result for the query.
   * @name pageState
   * @type String
   * @memberof module:types~ResultSet#
   * @default null
   */
  Object.defineProperty(this, 'pageState', { get: function () { return self.getPageState(); }, enumerable: true });
  //private properties
  Object.defineProperty(this, 'id', { value: response.id, enumerable: false});
  Object.defineProperty(this, '_columns', { value: undefined, enumerable: false, writable: true});
  /**
   * Method used to manually fetch the next page of results.
   * This method is only exposed when using the {@link Client#eachRow} method and there are more rows available in following pages.
   * @name nextPage
   * @type Function
   * @memberof module:types~ResultSet#
   */
  this.nextPage = undefined;
}

/**
 * Returns the first row or null if the result rows are empty
 */
ResultSet.prototype.first = function () {
  if (this.rows && this.rows.length) {
    return this.rows[0];
  }
  return null;
};

ResultSet.prototype.getPageState = function () {
  if (this.meta && this.meta.pageState) {
    return this.meta.pageState.toString('hex');
  }
  return null;
};

ResultSet.prototype.getColumns = function () {
  if (!this.meta) {
    return null;
  }
  return this.meta.columns;
};

module.exports = ResultSet;