import time
import logging

try:
    import os_client_config
except ImportError:
    os_client_config = None
from salt import exceptions

log = logging.getLogger(__name__)


class CinderException(Exception):

    _msg = "Cinder module exception occured."

    def __init__(self, message=None, **kwargs):
        super(CinderException, self).__init__(message or self._msg)


class NoCinderEndpoint(CinderException):
    _msg = "Cinder endpoint not found in keystone catalog."


class NoAuthPluginConfigured(CinderException):
    _msg = ("You are using keystoneauth auth plugin that does not support "
            "fetching endpoint list from token (noauth or admin_token).")


class NoCredentials(CinderException):
    _msg = "Please provide cloud name present in clouds.yaml."


class ResourceNotFound(CinderException):
    _msg = "Uniq resource: {resource} with name: {name} not found."

    def __init__(self, resource, name, **kwargs):
        super(CinderException, self).__init__(
            self._msg.format(resource=resource, name=name))


class MultipleResourcesFound(CinderException):
    _msg = "Multiple resource: {resource} with name: {name} found."

    def __init__(self, resource, name, **kwargs):
        super(CinderException, self).__init__(
            self._msg.format(resource=resource, name=name))


def _get_raw_client(cloud_name):
    if not os_client_config:
        raise exceptions.SaltInvocationError(
            "Cannot load os-client-config. Please check your environment "
            "configuration.")
    service_type = 'volumev3'
    config = os_client_config.OpenStackConfig()
    cloud = config.get_one_cloud(cloud_name)
    adapter = cloud.get_session_client(service_type)
    try:
        access_info = adapter.session.auth.get_access(adapter.session)
        endpoints = access_info.service_catalog.get_endpoints()
    except (AttributeError, ValueError):
        e = NoAuthPluginConfigured()
        log.exception('%s' % e)
        raise e
    if service_type not in endpoints:
        if not service_type:
            e = NoCinderEndpoint()
            log.error('%s' % e)
            raise e
    return adapter


def send(method):
    def wrap(func):
        def wrapped_f(*args, **kwargs):
            cloud_name = kwargs.pop('cloud_name')
            connection_params = kwargs.pop('connection_params', {}) or {}
            connect_retries = connection_params.get('connect_retries', 60)
            connect_retry_delay = connection_params.get('connect_retry_delay',
                                                        1)
            if not cloud_name:
                e = NoCredentials()
                log.error('%s' % e)
                raise e
            adapter = _get_raw_client(cloud_name)
            # Remove salt internal kwargs
            kwarg_keys = list(kwargs.keys())
            for k in kwarg_keys:
                if k.startswith('__'):
                    kwargs.pop(k)
            url, json = func(*args, **kwargs)
            response = None
            for i in range(connect_retries):
                try:
                    if json:
                        response = getattr(adapter, method)(
                            url, json=json, connect_retries=connect_retries)
                    else:
                        response = getattr(adapter, method)(url)
                except Exception as e:
                    if not hasattr(e, 'http_status') or (e.http_status >= 500
                                                      or e.http_status == 0):
                        msg = ("Got retriable exception when contacting "
                               "Cinder API. Sleeping for %ss. Attepmpts "
                               "%s of %s")
                        log.error(
                            msg % (connect_retry_delay, i, connect_retries))
                        time.sleep(connect_retry_delay)
                        continue
                break
            if not response or not response.content:
                return {}
            try:
                resp = response.json()
            except:
                resp = response.content
            return resp
        return wrapped_f
    return wrap
