# Copyright 2012 OpenStack Foundation
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from re import search

import testtools

from tempest.api.compute import base
from tempest.common.utils.linux import remote_client
from tempest.common import waiters
from tempest import config
from tempest.lib.common.utils import data_utils
from tempest.lib.common.utils import test_utils
from tempest.lib import decorators
from tempest.lib import exceptions as lib_exc

CONF = config.CONF


class ServersWithSpecificFlavorTestJSON(base.BaseV2ComputeAdminTest):
    """Test creating servers with specific flavor"""

    @classmethod
    def setup_credentials(cls):
        cls.prepare_instance_network()
        super(ServersWithSpecificFlavorTestJSON, cls).setup_credentials()

    @classmethod
    def setup_clients(cls):
        super(ServersWithSpecificFlavorTestJSON, cls).setup_clients()
        cls.client = cls.servers_client

    @decorators.idempotent_id('b3c7bcfc-bb5b-4e22-b517-c7f686b802ca')
    @testtools.skipUnless(CONF.validation.run_validation,
                          'Instance validation tests are disabled.')
    @testtools.skipIf("aarch64" in CONF.scenario.img_file,
                      "Aarch64 does not support ephemeral disk test")
    def test_verify_created_server_ephemeral_disk(self):
        """Verify that the ephemeral disk is created when creating server"""
        flavor_base = self.flavors_client.show_flavor(
            self.flavor_ref)['flavor']

        def create_flavor_with_ephemeral(ephem_disk):
            name = 'flavor_with_ephemeral_%s' % ephem_disk
            flavor_name = data_utils.rand_name(
                prefix=CONF.resource_name_prefix, name=name)

            ram = flavor_base['ram']
            vcpus = flavor_base['vcpus']
            disk = flavor_base['disk']

            # Create a flavor with ephemeral disk
            flavor = self.create_flavor(name=flavor_name, ram=ram, vcpus=vcpus,
                                        disk=disk, ephemeral=ephem_disk)

            # Set extra specs same as self.flavor_ref for the created flavor,
            # because the environment may need some special extra specs to
            # create server which should have been contained in
            # self.flavor_ref.
            extra_spec_keys = \
                self.admin_flavors_client.list_flavor_extra_specs(
                    self.flavor_ref)['extra_specs']
            if extra_spec_keys:
                self.admin_flavors_client.set_flavor_extra_spec(
                    flavor['id'], **extra_spec_keys)

            return flavor['id']

        flavor_with_eph_disk_id = create_flavor_with_ephemeral(ephem_disk=1)
        flavor_no_eph_disk_id = create_flavor_with_ephemeral(ephem_disk=0)

        admin_pass = self.image_ssh_password

        validation_resources = self.get_test_validation_resources(
            self.os_primary)
        server_no_eph_disk = self.create_test_server(
            validatable=True,
            validation_resources=validation_resources,
            wait_until='ACTIVE',
            adminPass=admin_pass,
            flavor=flavor_no_eph_disk_id)

        self.addCleanup(waiters.wait_for_server_termination,
                        self.servers_client, server_no_eph_disk['id'])
        self.addCleanup(test_utils.call_and_ignore_notfound_exc,
                        self.servers_client.delete_server,
                        server_no_eph_disk['id'])

        # Get partition number of server without ephemeral disk.
        server_no_eph_disk = self.client.show_server(
            server_no_eph_disk['id'])['server']
        linux_client = remote_client.RemoteClient(
            self.get_server_ip(server_no_eph_disk,
                               validation_resources),
            self.ssh_user,
            admin_pass,
            validation_resources['keypair']['private_key'],
            server=server_no_eph_disk,
            servers_client=self.client)
        disks_num = len(linux_client.get_disks().split('\n'))

        # Explicit server deletion necessary for Juno compatibility
        self.client.delete_server(server_no_eph_disk['id'])

        server_with_eph_disk = self.create_test_server(
            validatable=True,
            validation_resources=validation_resources,
            wait_until='ACTIVE',
            adminPass=admin_pass,
            flavor=flavor_with_eph_disk_id)

        self.addCleanup(waiters.wait_for_server_termination,
                        self.servers_client, server_with_eph_disk['id'])
        self.addCleanup(test_utils.call_and_ignore_notfound_exc,
                        self.servers_client.delete_server,
                        server_with_eph_disk['id'])

        server_with_eph_disk = self.client.show_server(
            server_with_eph_disk['id'])['server']
        linux_client = remote_client.RemoteClient(
            self.get_server_ip(server_with_eph_disk,
                               validation_resources),
            self.ssh_user,
            admin_pass,
            validation_resources['keypair']['private_key'],
            server=server_with_eph_disk,
            servers_client=self.client)
        disks_num_eph = len(linux_client.get_disks().split('\n'))
        self.assertEqual(disks_num + 1, disks_num_eph)


class ServersTestUEFI(base.BaseV2ComputeAdminTest):
    """Test creating server with UEFI firmware type"""

    @classmethod
    def setup_credentials(cls):
        cls.prepare_instance_network()
        super(ServersTestUEFI, cls).setup_credentials()

    @classmethod
    def setup_clients(cls):
        super(ServersTestUEFI, cls).setup_clients()
        cls.client = cls.servers_client

    @testtools.skipIf(
        CONF.compute_feature_enabled.barbican_integration_enabled,
        "Not supported when barbican integration enabled.")
    @decorators.idempotent_id('94feb6c3-d07e-b3b9-def8-64fd082d9b21')
    def test_created_server_uefi(self):
        # create custom image with uefi type
        custom_img = self.create_image_with_custom_property(
            hw_machine_type='q35',
            hw_firmware_type='uefi',
            )
        # create the server and wait for it to become ready
        validation_resources = self.get_class_validation_resources(
            self.os_primary)
        server = self.create_test_server(
            image_id=custom_img, validatable=True,
            validation_resources=validation_resources, wait_until='SSHABLE')
        # check UEFI boot loader in console log server
        uefi_boot_loader = "UEFI Misc Device"
        console_log = self.client.get_console_output(server['id'])['output']
        self.assertTrue(console_log, "Console output was empty.")
        self.assertIn(uefi_boot_loader, console_log)


class WindowsServersBaseTest(base.BaseV2ComputeAdminTest):
    """Test Windows OS guest servers"""

    image_id = None
    flavor_id = None

    @classmethod
    def skip_checks(cls):
        super(WindowsServersBaseTest, cls).skip_checks()

        if not (cls.image_id and cls.flavor_id):
            skip_msg = ("Environment is not prepared for testing "
                        "Windows servers")
            raise cls.skipException(skip_msg)

    @classmethod
    def setup_credentials(cls):
        cls.prepare_instance_network()
        super(WindowsServersBaseTest, cls).setup_credentials()

    @classmethod
    def setup_clients(cls):
        super(WindowsServersBaseTest, cls).setup_clients()
        cls.client = cls.servers_client

    def _test_create_server(self):
        # Create the server and wait for it to become ready
        validation_resources = self.get_class_validation_resources(
            self.os_primary)
        self.create_test_server(
            image_id=self.image_id,
            flavor=self.flavor_id,
            validatable=True,
            validation_resources=validation_resources,
            wait_until='PINGABLE')

    def _test_live_migration(self):
        block_migration = (CONF.compute_feature_enabled.
                           block_migration_for_live_migration)
        disk_over_commit = False
        validation_resources = self.get_class_validation_resources(
            self.os_primary)
        server_id = self.create_test_server(
            image_id=self.image_id,
            flavor=self.flavor_id,
            validatable=True,
            validation_resources=validation_resources,
            wait_until='PINGABLE')['id']
        source_host = self.get_host_for_server(server_id)
        if not CONF.compute_feature_enabled.can_migrate_between_any_hosts:
            # not to specify a host so that the scheduler will pick one
            destination_host = None
        else:
            destination_host = self.get_host_other_than(server_id)
        self.admin_servers_client.live_migrate_server(
            server_id,
            host=destination_host,
            block_migration=block_migration,
            disk_over_commit=disk_over_commit)
        waiters.wait_for_server_status(self.client,
                                       server_id, 'ACTIVE')
        destination_host = self.get_host_for_server(server_id)
        self.assertNotEqual(source_host, destination_host)

    def _test_cold_migration(self):
        # Run as admin to allow migrate tpm secret
        validation_resources = self.get_class_validation_resources(
            self.os_admin)
        server_id = self.create_test_server(
            clients=self.os_admin,
            image_id=self.image_id,
            flavor=self.flavor_id,
            validatable=True,
            validation_resources=validation_resources,
            wait_until='PINGABLE')['id']
        source_host = self.get_host_for_server(server_id)
        self.admin_servers_client.migrate_server(server_id)
        waiters.wait_for_server_status(self.admin_servers_client,
                                       server_id, 'VERIFY_RESIZE')
        self.admin_servers_client.confirm_resize_server(server_id)
        waiters.wait_for_server_status(self.admin_servers_client,
                                       server_id, 'ACTIVE')
        destination_host = self.get_host_for_server(server_id)
        self.assertNotEqual(source_host, destination_host)

    def _test_create_server_from_volume(self):
        size = self._get_flavor_disc_size()
        volume_bootable = self.create_volume(
            image_ref=self.image_id, size=size)
        bdm = [{
            'uuid': volume_bootable['id'],
            'source_type': 'volume',
            'destination_type': 'volume',
            'boot_index': 0}]
        validation_resources = self.get_class_validation_resources(
            self.os_admin)
        self.create_test_server(
            image_id='',
            flavor=self.flavor_id,
            clients=self.os_admin,
            validatable=True,
            validation_resources=validation_resources,
            wait_until='PINGABLE',
            block_device_mapping_v2=bdm
        )

    def _get_flavor_disc_size(self):
        # Returns a disc size of a flavor
        flavor = self.admin_flavors_client.show_flavor(self.flavor_id)[
            'flavor']
        return flavor['disk']


class WindowsServers10Test(WindowsServersBaseTest):

    image_id = CONF.compute.windows10_image_ref
    flavor_id = CONF.compute.windows10_flavor_ref

    @decorators.idempotent_id('4d54bcfa-08d3-48eb-b7a1-3568db4fc607')
    def test_create_server(self):
        self._test_create_server()

    @decorators.attr(type='multinode')
    @decorators.idempotent_id('6c22fcb1-4c3e-4bf6-b8c7-c3e2322cf5ff')
    @testtools.skipUnless(CONF.compute_feature_enabled.live_migration,
                          'Live migration is not available.')
    def test_live_migration(self):
        self._test_live_migration()

    @decorators.attr(type='multinode')
    @decorators.idempotent_id('96d67c40-fd4d-4286-a3c7-880d9eb77a95')
    @testtools.skipUnless(CONF.compute_feature_enabled.cold_migration,
                          'Cold migration is not available.')
    def test_cold_migration(self):
        self._test_cold_migration()

    @decorators.idempotent_id('8b7ddf91-34fe-4538-8d08-2443d9968771')
    def test_create_server_from_volume(self):
        self._test_create_server_from_volume()


class WindowsServers11Test(WindowsServersBaseTest):

    image_id = CONF.compute.windows11_image_ref
    flavor_id = CONF.compute.windows11_flavor_ref

    @decorators.idempotent_id('1cff7fea-f251-4a05-a667-9b946913a3c5')
    def test_create_server(self):
        self._test_create_server()

    @decorators.attr(type=['negative'])
    @decorators.idempotent_id('9afd991e-0478-41ca-b5cf-bf32b10ae5a7')
    @testtools.skipUnless(CONF.compute_feature_enabled.live_migration,
                          'Live migration is not available.')
    def test_live_migration_with_vtpm_negative(self):
        """Test live migrating instance with vTPM should not be supported"""
        self.assertRaises(lib_exc.BadRequest, self._test_live_migration)

    @decorators.attr(type='multinode')
    @decorators.idempotent_id('7da88453-cc6d-4fef-b893-b4ae8f40767d')
    @testtools.skipUnless(CONF.compute_feature_enabled.cold_migration,
                          'Cold migration is not available.')
    def test_cold_migration(self):
        self._test_cold_migration()

    @decorators.idempotent_id('1e8a241e-29ff-4627-abd0-5be74251838b')
    def test_create_server_from_volume(self):
        self._test_create_server_from_volume()


class ServersTestVGPU(base.BaseV2ComputeAdminTest):
    """Test creating server with vGPU flavor"""

    @classmethod
    def setup_credentials(cls):
        cls.prepare_instance_network()
        super(ServersTestVGPU, cls).setup_credentials()

    @classmethod
    def setup_clients(cls):
        super(ServersTestVGPU, cls).setup_clients()
        cls.client = cls.servers_client

    @testtools.skipUnless(CONF.compute_feature_enabled.vgpu_flavor_ref,
                          'vGPU flavor is not available.')
    @testtools.skipUnless(CONF.compute.image_full_ref,
                          'Current test requires full OS to be used.')
    @decorators.idempotent_id('5c06d62b-d9c9-4cef-8b56-ef003af03519')
    def test_create_server_vgpu(self):
        # create the server and wait for it to become ready
        validation_resources = self.get_class_validation_resources(
            self.os_primary)
        server = self.create_test_server(
            image_id=CONF.compute.image_full_ref,
            validatable=True,
            validation_resources=validation_resources,
            flavor=CONF.compute_feature_enabled.vgpu_flavor_ref)
        # check that vgpu_pci_name is in console log server
        linux_client = remote_client.RemoteClient(
            self.get_server_ip(server, validation_resources),
            CONF.compute.image_full_username,
            pkey=validation_resources['keypair']['private_key'],
            server=server,
            servers_client=self.client)
        output = linux_client.exec_command('lspci')
        self.assertTrue(search(
            CONF.compute_feature_enabled.vgpu_lspci_pattern, output))
