# Copyright 2018 Red Hat, Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import time

from tempest.api.image import base
from tempest import config
from tempest.lib.common.utils import data_utils
from tempest.lib import decorators
from tempest.lib import exceptions as lib_exc

CONF = config.CONF


class BasicOperationsImagesAdminTest(base.BaseV2ImageAdminTest):
    """"Test image operations about image owner"""

    @decorators.related_bug('1420008')
    @decorators.idempotent_id('646a6eaa-135f-4493-a0af-12583021224e')
    def test_create_image_owner_param(self):
        """Test creating image with specified owner"""
        # NOTE: Create image with owner different from tenant owner by
        # using "owner" parameter requires an admin privileges.
        random_id = data_utils.rand_uuid_hex()
        image = self.admin_client.create_image(
            container_format='bare', disk_format='raw', owner=random_id)
        self.addCleanup(self.admin_client.delete_image, image['id'])
        image_info = self.admin_client.show_image(image['id'])
        self.assertEqual(random_id, image_info['owner'])

    @decorators.related_bug('1420008')
    @decorators.idempotent_id('525ba546-10ef-4aad-bba1-1858095ce553')
    def test_update_image_owner_param(self):
        """Test updating image owner"""
        random_id_1 = data_utils.rand_uuid_hex()
        image = self.admin_client.create_image(
            container_format='bare', disk_format='raw', owner=random_id_1)
        self.addCleanup(self.admin_client.delete_image, image['id'])
        created_image_info = self.admin_client.show_image(image['id'])

        random_id_2 = data_utils.rand_uuid_hex()
        self.admin_client.update_image(
            image['id'], [dict(replace="/owner", value=random_id_2)])
        updated_image_info = self.admin_client.show_image(image['id'])

        self.assertEqual(random_id_2, updated_image_info['owner'])
        self.assertNotEqual(created_image_info['owner'],
                            updated_image_info['owner'])


class ImageWebUploadAdminTest(base.BaseV2ImageAdminTest):
    @classmethod
    def skip_checks(cls):
        super(ImageWebUploadAdminTest, cls).skip_checks()
        enabled_methods = CONF.image_feature_enabled.enabled_import_methods
        if "web-download" not in enabled_methods:
            raise cls.skipException(
                "Glance image upload via url feature disabled")

    @decorators.idempotent_id('5b2ce43c-924c-4bae-bac0-f5d6ed69d72e')
    def test_image_upload_via_url(self):
        # Create image
        image_name = data_utils.rand_name("image")
        container_format = CONF.image.container_formats[0]
        disk_format = CONF.image.disk_formats[0]
        image = self.create_image(name=image_name,
                                  container_format=container_format,
                                  disk_format=disk_format,
                                  visibility='private')
        self.assertEqual('queued', image['status'])

        # Upload image via url
        image_uri = CONF.image.http_image
        method = {"name": "web-download", "uri": image_uri}
        self.admin_client.import_image(image_id=image["id"], method=method)

        timeout = CONF.image.build_timeout
        interval = CONF.image.build_interval

        start_time = int(time.time())
        while True:
            body = self.admin_client.show_image(image['id'])
            if body["status"] == "active":
                break
            if int(time.time()) - start_time >= timeout:
                message = ('Image %(id)s failed to become active within '
                           'the required time (%(timeout)s s).' %
                           {'id': image['id'], 'timeout': timeout})
                raise lib_exc.TimeoutException(message)
            time.sleep(interval)
