# Copyright (C) 2016 VA Linux Systems Japan K.K.
# Copyright (C) 2016 Fumihiko Kakuma <kakuma at valinux co jp>
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from tempest import config
from tempest.lib import decorators

from neutron_tempest_plugin.neutron_dynamic_routing import frr
from neutron_tempest_plugin.neutron_dynamic_routing.scenario import base
from neutron_tempest_plugin.neutron_dynamic_routing.scenario\
    import base_test_proto as test_base


CONF = config.CONF


class BgpSpeakerIpv6Test(test_base.BgpSpeakerProtoTestBase):

    RAS_MAX = 3
    IP_ALLOCATION_MULTIPLIER = 0
    ip_version = 6
    PNet = base.Net(name='', net='2001:db8::', mask=64,
                    cidr='2001:db8::/64', router=None)
    PPool = base.Pool(name='test-pool-ext', prefixlen=PNet.mask,
                      prefixes=[PNet.net + '/8'])
    PSubNet = base.SubNet(name='', cidr=PNet.cidr, mask=PNet.mask)
    TPool = base.Pool(name='tenant-test-pool', prefixlen=64,
                      prefixes=['2001:db8:8000::/48'])
    L_AS = base.AS(asn='64512', router_id='192.168.0.2', adv_net='')
    ras_l = [
        base.AS(asn='64522', router_id='192.168.0.12',
                adv_net='2001:db8:9002::/48'),
        base.AS(asn='64523', router_id='192.168.0.13',
                adv_net='2001:db8:9003::/48'),
        base.AS(asn='64524', router_id='192.168.0.14',
                adv_net='2001:db8:9004::/48')
    ]

    bgp_speaker_args = {
        'local_as': L_AS.asn,
        'ip_version': ip_version,
        'name': 'my-bgp-speaker1',
        'advertise_floating_ip_host_routes': True,
        'advertise_tenant_networks': True
    }
    bgp_peer_args = [
        {'remote_as': ras_l[0].asn,
         'name': 'my-bgp-peer1',
         'peer_ip': None,
         'auth_type': 'none'},
        {'remote_as': ras_l[1].asn,
         'name': 'my-bgp-peer2',
         'peer_ip': None,
         'auth_type': 'none'},
        {'remote_as': ras_l[2].asn,
         'name': 'my-bgp-peer3',
         'peer_ip': None,
         'auth_type': 'none'}
    ]

    def setUp(self):
        super(BgpSpeakerIpv6Test, self).setUp()

    @classmethod
    def skip_checks(cls):
        super().skip_checks()
        if not CONF.dynamic_routing.frr_bgp_ipv6_enabled:
            raise cls.skipException("Dynamic routing IPv6 tests not enabled.")
        if CONF.production:
            raise cls.skipException("Skip on production environment.")

    @classmethod
    def resource_setup_container(cls):
        # frr container
        for i in range(cls.RAS_MAX):
            container_number = i + 1
            bgpd = {
                'bgp': {
                    'as_number': int(cls.ras_l[i].asn),
                    'router_id': cls.ras_l[i].router_id,
                    'neighbors': [
                        {
                            'peer_group': True,
                            'address': 'dnr',
                            'as_number': cls.L_AS.asn,
                            'passive': True,
                        }
                    ],
                    'listen': {
                        'ranges': [
                            {'cidr': CONF.dynamic_routing.
                             frr_bgp_ipv6_control_cidr,
                             'peer_group': 'dnr'}
                        ],
                        'limit': 10,
                    },
                    'ebgp_requires_policy': False,
                }
            }
            qg = frr.FrrBGPContainer(
                "qv6" + str(container_number),
                CONF.dynamic_routing.frr_docker_image,
                bgpd=bgpd,
            )
            qg.run(wait=True)
            cls.containers.append(qg.ctn)
            cls.r_ass.append(qg)
            cls.r_as_ip.append(
                CONF.dynamic_routing.frr_provider_ips[
                    cls.IP_ALLOCATION_MULTIPLIER + container_number - 1]
            )
        cls.tnet_gen = cls.get_subnet(start='2001:db8:8000:1::',
                                      end='2001:db8:8000:ffff::',
                                      step=65536 * 65536 * 65536 * 65536)

    @decorators.idempotent_id('5194a8e2-95bd-49f0-872d-1e3e875ede32')
    def test_check_neighbor_established(self):
        self._test_check_neighbor_established(self.ip_version)

    @decorators.idempotent_id('6a3483fc-8c8a-4387-bda6-c7061410e04b')
    def test_check_advertised_tenant_network(self):
        self._test_check_advertised_tenant_network(self.ip_version)

    @decorators.idempotent_id('aca5d678-c249-4de5-921b-6b6ba621e4f7')
    def test_check_advertised_multiple_tenant_network(self):
        self._test_check_advertised_multiple_tenant_network(self.ip_version)

    @decorators.idempotent_id('f81012f3-2f7e-4b3c-8c1d-b1778146d712')
    def test_check_neighbor_established_with_multiple_peers(self):
        self._test_check_neighbor_established_with_multiple_peers(
            self.ip_version)

    @decorators.idempotent_id('be710ec1-a338-44c9-8b89-31c3532aae65')
    def test_check_advertised_tenant_network_with_multiple_peers(self):
        self._test_check_advertised_tenant_network_with_multiple_peers(
            self.ip_version)
