# Copyright (C) 2016 VA Linux Systems Japan K.K.
# Copyright (C) 2016 Fumihiko Kakuma <kakuma at valinux co jp>
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from tempest.common import utils
from tempest import config
from tempest.lib import decorators
from tempest.lib import exceptions
import tenacity

from neutron_tempest_plugin.neutron_dynamic_routing import frr
from neutron_tempest_plugin.neutron_dynamic_routing.frr import constants
from neutron_tempest_plugin.neutron_dynamic_routing.scenario import base
from neutron_tempest_plugin.neutron_dynamic_routing.scenario\
    import base_test_proto as test_base

CONF = config.CONF


class BgpSpeakerIpv4Test(test_base.BgpSpeakerProtoTestBase):

    RAS_MAX = 3
    IP_ALLOCATION_MULTIPLIER = 0
    ip_version = 4
    MyScope = base.Scope(name='my-scope')
    PNet = base.Net(name='', net='172.24.6.0', mask=24,
                    cidr='172.24.6.0/24', router=None)
    PPool = base.Pool(name='test-pool-ext', prefixlen=PNet.mask,
                      prefixes=[PNet.net + '/8'])
    PSubNet = base.SubNet(name='', cidr=PNet.cidr, mask=PNet.mask)
    TPool = base.Pool(name='tenant-test-pool', prefixlen=28,
                      prefixes=['10.10.0.0/16'])
    L_AS = base.AS(asn='64512', router_id='192.168.0.2', adv_net='')
    ras_l = [
        base.AS(asn='64522', router_id='192.168.0.12',
                adv_net='192.168.162.0/24'),
        base.AS(asn='64523', router_id='192.168.0.13',
                adv_net='192.168.163.0/24'),
        base.AS(asn='64524', router_id='192.168.0.14',
                adv_net='192.168.164.0/24')
    ]

    bgp_speaker_args = {
        'local_as': L_AS.asn,
        'ip_version': ip_version,
        'name': 'my-bgp-speaker1',
        'advertise_floating_ip_host_routes': True,
        'advertise_tenant_networks': True
    }
    bgp_peer_args = [
        {'remote_as': ras_l[0].asn,
         'name': 'my-bgp-peer1',
         'peer_ip': None,
         'auth_type': 'none'},
        {'remote_as': ras_l[1].asn,
         'name': 'my-bgp-peer2',
         'peer_ip': None,
         'auth_type': 'none'},
        {'remote_as': ras_l[2].asn,
         'name': 'my-bgp-peer3',
         'peer_ip': None,
         'auth_type': 'none'}
    ]

    def setUp(self):
        for ctn in self.containers:
            ctn.restart()
        super(BgpSpeakerIpv4Test, self).setUp()

    @classmethod
    def skip_checks(cls):
        super().skip_checks()
        if CONF.production:
            raise cls.skipException('Skip on production environment.')

    @classmethod
    def resource_setup_container(cls):
        # frr container
        for i in range(cls.RAS_MAX):
            container_number = i + 1
            bgp_listenon = CONF.dynamic_routing.frr_provider_ipv4_ips[
                cls.IP_ALLOCATION_MULTIPLIER + container_number - 1
            ].split('/')[0]
            bgpd = {
                'bgp': {
                    'as_number': int(cls.ras_l[i].asn),
                    'router_id': bgp_listenon,
                    'neighbors': [
                        {
                            'peer_group': True,
                            'address': 'dnr',
                            'as_number': cls.L_AS.asn,
                            'passive': True,
                        }
                    ],
                    'listen': {
                        'ranges': [
                            {'cidr': CONF.dynamic_routing.
                             frr_bgp_ipv4_control_cidr,
                             'peer_group': 'dnr'}
                        ],
                        'limit': 10,
                    },
                    'ebgp_requires_policy': False,
                }
            }
            daemons = {
                "bgpd": {
                    "enabled": "yes",
                    "listenon": bgp_listenon,
                },
                "vtysh": {"enabled": "yes"},
            }
            qg = frr.FrrBGPContainer(
                "q" + str(container_number),
                CONF.dynamic_routing.frr_docker_image,
                bgpd=bgpd,
                daemons=daemons,
            )
            qg.run(wait=True)
            cls.containers.append(qg.ctn)
            cls.r_ass.append(qg)
            cls.r_as_ip.append(
                CONF.dynamic_routing.frr_provider_ipv4_ips[
                    cls.IP_ALLOCATION_MULTIPLIER + container_number - 1]
            )
        cls.tnet_gen = cls.get_subnet(start='10.10.1.0', end='10.10.255.0',
                                      step=256)

    @tenacity.retry(
        stop=tenacity.stop_after_attempt(10),
        wait=tenacity.wait_fixed(2),
        retry=tenacity.retry_if_exception_type(exceptions.Conflict),
        reraise=True)
    def _add_bgp_speaker_to_agent(self, agent_id, speaker_id):
        self.bgp_client.add_bgp_speaker_to_dragent(agent_id, speaker_id)

    @decorators.attr(type='smoke')
    @decorators.idempotent_id('7f2acbc2-ff88-4a63-aa02-a2f9feb3f5b0')
    def test_check_neighbor_established(self):
        self._test_check_neighbor_established(self.ip_version)

    @decorators.idempotent_id('f32245fc-aeab-4244-acfa-3af9dd662e8d')
    def test_check_advertised_tenant_network(self):
        self._test_check_advertised_tenant_network(self.ip_version)

    @decorators.idempotent_id('a5c238de-b750-499c-aaa2-b44a057e9ed3')
    def test_check_advertised_multiple_tenant_network(self):
        self._test_check_advertised_multiple_tenant_network(self.ip_version)

    @decorators.idempotent_id('e4961cc1-0c47-4081-a896-caaa9342ca75')
    def test_check_neighbor_established_with_multiple_peers(self):
        self._test_check_neighbor_established_with_multiple_peers(
            self.ip_version)

    @decorators.idempotent_id('91971dfb-c129-4744-9fbb-ac4f9e8d56c0')
    def test_check_advertised_tenant_network_with_multiple_peers(self):
        self._test_check_advertised_tenant_network_with_multiple_peers(
            self.ip_version)

    @decorators.idempotent_id('cc615252-c6cb-4d75-a70e-608fb2c3736a')
    def test_schedule_added_speaker(self):
        self.bgp_peer_args[0]['peer_ip'] = self.r_as_ip[0].split('/')[0]
        num, subnet = next(self.tnet_gen)
        mask = '/' + str(self.TPool.prefixlen)
        TNet = base.Net(
            name='',
            net=subnet,
            mask=self.TPool.prefixlen,
            cidr=subnet + mask,
            router=None,
        )
        TSubNet = base.SubNet(name='', cidr=TNet.cidr, mask=TNet.mask)
        MyRouter = base.Router(name='my-router' + str(num), gw='')
        ext_net_id = self.create_bgp_network(
            4,
            self.MyScope,
            self.PNet,
            self.PPool,
            self.PSubNet,
            self.TPool,
            [[TNet, TSubNet, MyRouter]],
        )
        speaker_id, peers_ids = self.create_and_add_peers_to_speaker(
            ext_net_id, self.bgp_speaker_args, [self.bgp_peer_args[0]]
        )
        self.r_ass[0].bgp_check_neighbor_state(
            'dnr', constants.BGP_FSM_ESTABLISHED
        )

    @decorators.idempotent_id('ce98c33c-0ffa-49ae-b365-da836406793b')
    def test_unschedule_deleted_speaker(self):
        self.bgp_peer_args[0]['peer_ip'] = self.r_as_ip[0].split('/')[0]
        num, subnet = next(self.tnet_gen)
        mask = '/' + str(self.TPool.prefixlen)
        TNet = base.Net(
            name='',
            net=subnet,
            mask=self.TPool.prefixlen,
            cidr=subnet + mask,
            router=None,
        )
        TSubNet = base.SubNet(name='', cidr=TNet.cidr, mask=TNet.mask)
        MyRouter = base.Router(name='my-router' + str(num), gw='')
        ext_net_id = self.create_bgp_network(
            4,
            self.MyScope,
            self.PNet,
            self.PPool,
            self.PSubNet,
            self.TPool,
            [[TNet, TSubNet, MyRouter]],
        )
        speaker_id, peers_ids = self.create_and_add_peers_to_speaker(
            ext_net_id,
            self.bgp_speaker_args,
            [self.bgp_peer_args[0]],
            auto_delete=False,
        )
        self.r_ass[0].bgp_check_neighbor_state(
            'dnr', constants.BGP_FSM_ESTABLISHED
        )
        self.delete_bgp_speaker(speaker_id)
        self.delete_bgp_peer(peers_ids[0])
        self.r_ass[0].bgp_check_neighbor_absent('dnr')

    @decorators.idempotent_id('aa6c565c-ded3-413b-8dc9-3928b3b0e38f')
    def test_remove_add_speaker_agent(self):
        self.bgp_peer_args[0]['peer_ip'] = self.r_as_ip[0].split('/')[0]
        num, subnet = next(self.tnet_gen)
        mask = '/' + str(self.TPool.prefixlen)
        TNet = base.Net(
            name='',
            net=subnet,
            mask=self.TPool.prefixlen,
            cidr=subnet + mask,
            router=None,
        )
        TSubNet = base.SubNet(name='', cidr=TNet.cidr, mask=TNet.mask)
        MyRouter = base.Router(name='my-router' + str(num), gw='')
        ext_net_id = self.create_bgp_network(
            4,
            self.MyScope,
            self.PNet,
            self.PPool,
            self.PSubNet,
            self.TPool,
            [[TNet, TSubNet, MyRouter]],
        )
        speaker_id, peers_ids = self.create_and_add_peers_to_speaker(
            ext_net_id, self.bgp_speaker_args, [self.bgp_peer_args[0]]
        )
        self.r_ass[0].bgp_check_neighbor_state(
            'dnr', constants.BGP_FSM_ESTABLISHED
        )
        agent_list = self.bgp_client.list_dragents_for_bgp_speaker(speaker_id)[
            'agents'
        ]
        self.assertEqual(1, len(agent_list))
        agent_id = agent_list[0]['id']
        self.bgp_client.remove_bgp_speaker_from_dragent(agent_id, speaker_id)
        self.r_ass[0].bgp_check_neighbor_absent('dnr')
        self._add_bgp_speaker_to_agent(agent_id, speaker_id)
        self.r_ass[0].bgp_check_neighbor_state(
            'dnr', constants.BGP_FSM_ESTABLISHED
        )


class BgpSpeaker4byteASNTest(test_base.BgpSpeakerProtoTestBase):

    RAS_MAX = 3
    IP_ALLOCATION_MULTIPLIER = 3
    ip_version = 4
    MyScope = base.Scope(name='my-scope')
    PNet = base.Net(name='', net='172.24.6.0', mask=24, cidr='172.24.6.0/24',
                    router=None)
    PPool = base.Pool(name='test-pool-ext', prefixlen=PNet.mask,
                      prefixes=[PNet.net + '/8'])
    PSubNet = base.SubNet(name='', cidr=PNet.cidr, mask=PNet.mask)
    TPool = base.Pool(name='tenant-test-pool', prefixlen=28,
                      prefixes=['10.10.0.0/16'])
    L_AS = base.AS(asn='4200000000', router_id='192.168.0.3', adv_net='')
    ras_l = [
        base.AS(asn='4210000000', router_id='192.168.0.12',
                adv_net='192.168.162.0/24'),
        base.AS(asn='64522', router_id='192.168.0.13',
                adv_net='192.168.163.0/24'),
        base.AS(asn='4230000000', router_id='192.168.0.14',
                adv_net='192.168.164.0/24'),
    ]

    bgp_speaker_args = {
        'local_as': L_AS.asn,
        'ip_version': ip_version,
        'name': 'my-bgp-speaker1',
        'advertise_floating_ip_host_routes': True,
        'advertise_tenant_networks': True,
    }
    bgp_peer_args = [
        {'remote_as': ras_l[0].asn,
         'name': 'my-bgp-peer1',
         'peer_ip': None,
         'auth_type': 'none'},
        {'remote_as': ras_l[1].asn,
         'name': 'my-bgp-peer2',
         'peer_ip': None,
         'auth_type': 'none'},
        {'remote_as': ras_l[2].asn,
         'name': 'my-bgp-peer3',
         'peer_ip': None,
         'auth_type': 'none'},
    ]

    @classmethod
    @utils.requires_ext(extension='bgp_4byte_asn', service='network')
    def resource_setup(cls):
        super(BgpSpeaker4byteASNTest, cls).resource_setup()

    @classmethod
    def skip_checks(cls):
        super().skip_checks()
        if CONF.production:
            raise cls.skipException('Skip on production environment.')

    @classmethod
    def resource_setup_container(cls):
        # frr container
        for i in range(cls.RAS_MAX):
            container_number = i + 1
            bgp_listenon = CONF.dynamic_routing.frr_provider_ipv4_ips[
                cls.IP_ALLOCATION_MULTIPLIER + container_number - 1
            ].split('/')[0]
            bgpd = {
                'bgp': {
                    'as_number': int(cls.ras_l[i].asn),
                    'router_id': bgp_listenon,
                    'neighbors': [
                        {
                            'peer_group': True,
                            'address': 'dnr',
                            'as_number': cls.L_AS.asn,
                            'passive': True,
                        }
                    ],
                    'listen': {
                        'ranges': [
                            {'cidr': CONF.dynamic_routing.
                             frr_bgp_ipv4_control_cidr,
                             'peer_group': 'dnr'}
                        ],
                        'limit': 10,
                    },
                    'ebgp_requires_policy': False,
                }
            }
            daemons = {
                "bgpd": {
                    "enabled": "yes",
                    "listenon": bgp_listenon,
                },
                "vtysh": {"enabled": "yes"},
            }
            qg = frr.FrrBGPContainer(
                'q-4byte-asn-' + str(container_number),
                CONF.dynamic_routing.frr_docker_image,
                bgpd=bgpd,
                daemons=daemons,
            )
            qg.run(wait=True)
            cls.containers.append(qg.ctn)
            cls.r_ass.append(qg)
            cls.r_as_ip.append(
                CONF.dynamic_routing.frr_provider_ipv4_ips[
                    cls.IP_ALLOCATION_MULTIPLIER + container_number - 1]
            )
        cls.tnet_gen = cls.get_subnet(
            start='10.10.1.0', end='10.10.255.0', step=256
        )

    @decorators.idempotent_id('9f18c931-a59e-4a27-939b-21124995ffe2')
    def test_check_neighbor_established(self):
        self._test_check_neighbor_established(self.ip_version)

    @decorators.idempotent_id('73466aa5-7d1d-4f9f-8fb4-4100fad2dffe')
    def test_check_advertised_tenant_network(self):
        self._test_check_advertised_tenant_network(self.ip_version)

    @decorators.idempotent_id('c3158328-2f69-4aa2-b1b7-5a06ab58afaf')
    def test_check_advertised_multiple_tenant_network(self):
        self._test_check_advertised_multiple_tenant_network(self.ip_version)

    @decorators.idempotent_id('212a3d82-ac50-43dc-b657-030b1133643e')
    def test_check_neighbor_established_with_multiple_peers(self):
        self._test_check_neighbor_established_with_multiple_peers(
            self.ip_version)

    @decorators.idempotent_id('c72411c8-ea79-495d-bdbd-a10159642676')
    def test_check_advertised_tenant_network_with_multiple_peers(self):
        self._test_check_advertised_tenant_network_with_multiple_peers(
            self.ip_version)
