# Copyright 2016 Cisco Systems, Inc.
# All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import os

import netaddr
from oslo_concurrency import lockutils
from oslo_log import log as logging
import tempest.api.compute.base as base2
from tempest.common import utils
from tempest import config
from tempest.lib import decorators

from neutron_tempest_plugin.bgpvpn import base
from neutron_tempest_plugin.bgpvpn.scenario import manager


CONF = config.CONF
LOG = logging.getLogger(__name__)

NET_D = 'D'
NET_E = 'E'
NET_F = 'F'


if "SUBNETPOOL_PREFIX_V4" in os.environ:
    subnet_base = netaddr.IPNetwork(os.environ['SUBNETPOOL_PREFIX_V4'])
    if subnet_base.prefixlen > 21:
        raise Exception("if SUBNETPOOL_PREFIX_V4 is set, it needs to offer "
                        "space for at least 8 /24 subnets")
else:
    subnet_base = netaddr.IPNetwork("10.200.0.0/16")


def assign_24(idx):
    # how many addresses in a /24:
    range_size = 2 ** (32 - 24)
    return netaddr.cidr_merge(
        subnet_base[range_size * idx:range_size * (idx + 1)])[0]


S1D = assign_24(1)
S1E = assign_24(4)
S1F = assign_24(6)
NET_D_S1 = str(S1D)
NET_E_S1 = str(S1E)
NET_F_S1 = str(S1F)
IP_D_S1_1 = str(S1D[10])
IP_E_S1_1 = str(S1E[20])
IP_F_S1_1 = str(S1F[30])


class TestBGPVPNMigration(base.BaseBgpvpnTest, manager.NetworkScenarioTest,
                          base2.BaseV2ComputeAdminTest):
    block_migration = None

    @classmethod
    def setUpClass(cls):
        super(TestBGPVPNMigration, cls).setUpClass()
        cls._rt_index = 0
        cls.admin_migration_client = cls.os_admin.migrations_client

    @classmethod
    @lockutils.synchronized('bgpvpn')
    def new_rt(cls):
        cls._rt_index += 1
        return "64513:%d" % cls._rt_index

    def setUp(self):
        super(TestBGPVPNMigration, self).setUp()
        self.servers_keypairs = {}
        self.servers = []
        self.server_fixed_ips = {}
        self.ports = {}
        self.networks = {}
        self.subnets = {}
        self.server_fips = {}
        self._create_security_group_for_test()
        self.RT1 = self.new_rt()

    @decorators.idempotent_id('8a5a6fac-313c-464b-9c5c-77eca377ff1a')
    @utils.services('compute', 'network')
    def test_bgpvpn_livemigration(self):
        """This test checks connection between VMs with BGPVPN

         after live migration

        1. Create networks D, E and F with their respective subnets
        2. Associate network D, E and network F to a given L3 BGPVPN
        3. Start up server 1 in network D
        4. Start up server 2 in network E
        5. Create router and connect it to network D
        6. Give a FIP to server 1
        7. Check that server 1 can ping server 2
        8. Live migrate server 1
        9. Check that server 1 can ping server 2
        10. Live migrate server 2
        11. Check that server 1 can ping server 2
        """
        self._create_networks_and_subnets(names=[NET_D, NET_E, NET_F],
                                          subnet_cidrs=[[NET_D_S1], [NET_E_S1],
                                                        [NET_F_S1]])
        self._create_l3_bgpvpn()
        self._associate_all_nets_to_bgpvpn()
        self._create_servers(ports_config=[
            [self.networks[NET_D], IP_D_S1_1],
            [self.networks[NET_E], IP_E_S1_1],
            [self.networks[NET_F], IP_F_S1_1]
        ])
        self._associate_fip_and_check_l3_bgpvpn(subnet=NET_D)
        # Migrate both VMs one by one and check connection
        for i in range(2):
            source_host = self.get_host_for_server(self.servers[i]['id'])
            if not CONF.compute_feature_enabled.can_migrate_between_any_hosts:
                # not to specify a host so that the scheduler will pick one
                destination_host = None
            else:
                destination_host = \
                    self.get_host_other_than(self.servers[i]['id'])
            LOG.info("Live migrate from source %s to destination %s",
                     source_host, destination_host)
            self._live_migrate(self.servers[i]['id'],
                               destination_host, 'ACTIVE')
            self._check_l3_bgpvpn()
