#!/bin/bash
# Redirect all outputs
exec > >(tee -i /tmp/bootstart-trymos-output.log) 2>&1

set -x
# allow access to the local variables from prepare-metadata.py
set -a

export NODE_TYPE='trymos'
export NODE_METADATA='{"labels": {"local-volume-provisioner": "enabled", "openstack-compute-node": "enabled", "openstack-control-plane": "enabled", "openstack-gateway": "enabled", "openvswitch": "enabled", "role": "ceph-osd-node"}}'

_ENVIRONMENT_FOLDER="/usr/share/trymos/environment"
_DEPLOY_FINISHED_LOCK="/var/tmp/trymos-deploy-finished.lock"
_TRYMOS_INFO_FILE="/etc/trymos-info"


export RELEASE_OPENSTACK_K8S=/srv/release-openstack-k8s
export CERT_REPO_DIR=/srv/certs
export CSR_CONF="${CERT_REPO_DIR}/api-csr.yaml"
export SSL_BARE_NAME='api'

function get_instance_type {
    if grep "OpenStack" -i /sys/devices/virtual/dmi/id/product_name  -q; then
        echo "openstack"
    elif grep -iq 'amazon' /sys/devices/virtual/dmi/id/{product_version,sys_vendor}; then
        echo "amazon"
    elif grep "virtualbox" -i /sys/devices/virtual/dmi/id/product_name  -q; then
        echo "virtualbox"
    else
        echo "UNKNOWN"
    fi
}
export _INSTANCE_TYPE=$(get_instance_type)

if [[ -f "$_ENVIRONMENT_FOLDER/${_INSTANCE_TYPE}" ]]; then
    source $_ENVIRONMENT_FOLDER/${_INSTANCE_TYPE}
else
    echo "Unknown instance type $_INSTANCE_TYPE"
    exit 1
fi

source $_ENVIRONMENT_FOLDER/common
source /usr/share/trymos/functions

export STORAGE_CLUSTER_CIDR=${CONTROL_NETWORK_CIDR}
export STORAGE_PUBLIC_CIDR=${CONTROL_NETWORK_CIDR}
export TUNNEL_INTERFACE=${TUNNEL_INTERFACE:-${DEFAULT_INTERFACE}}
export LIVE_MIGRATION_INTERFACE=${LIVE_MIGRATION_INTERFACE:-${DEFAULT_INTERFACE}}

if [[ -f ${_DEPLOY_FINISHED_LOCK} ]]; then
    echo "TryMOS deploy has been performed on this node."
    echo "If you want to redeploy please delete lock file"
    echo "${_DEPLOY_FINISHED_LOCK}"
    exit 1
fi

function info {
    local msg="[INFO]: $1"
    echo "$msg"
}

function generate_ceph_metadata {
    mkdir -p /usr/share/metadata
    cat << EOF > /usr/share/metadata/ceph.yaml
storageDevices:
  - name: ${CEPH_STORAGE_OSD_DEVICE}
    role: hdd
    sizeGb: 2
ramGb: 8
cores: 2
EOF

}


function configure_virt_public_iface {

cat << EOF > /etc/systemd/network/20-public-int.netdev
[NetDev]
Name=ens4
Kind=dummy

[Match]
Name=ens4
EOF

cat << EOF > /etc/systemd/network/20-public-int.network
[Match]
Name=ens4

[Network]
Address=10.11.12.23/24
Broadcast=10.11.12.255
EOF

    systemctl restart systemd-networkd
    sleep 15

}

function network_config {
    PUBLIC_NODE_IP_ADDRESS=${PUBLIC_INTERFACE_IP:-$(ip addr show dev ${PUBLIC_INTERFACE} | grep -Po 'inet \K[\d.]+' | egrep -v "127.0.|172.17")}
    PUBLIC_NODE_IP_NETMASK=${PUBLIC_INTERFACE_NETMASK:-$(ip addr show dev ${PUBLIC_INTERFACE} | grep -Po 'inet \K[\d.]+\/[\d]+' | egrep -v "127.0.|172.17" | cut -d'/' -f2)}

    local public_interface=${1:-${PUBLIC_INTERFACE}}
    local cloud_netplan_cfg="/etc/netplan/50-cloud-init.yaml"
    local match_ip_line

    DEBIAN_FRONTEND=noninteractive apt -y install bridge-utils

cat << EOF > /etc/systemd/network/10-veth-phy-br.netdev
[NetDev]
Name=veth-phy
Kind=veth
[Peer]
Name=veth-br
EOF

    sed -i 's/.*ethernets:.*/&\n        veth-phy: {}/' ${cloud_netplan_cfg}
    sed -i 's/.*ethernets:.*/&\n        veth-br: {}/' ${cloud_netplan_cfg}
    # NOTE(ohryhorov): have to be disabled if PUBLIC_INTERFACE is defined by
    # cloud-init.
    sed -i "s/.*ethernets:.*/&\n        ${PUBLIC_INTERFACE}: {}/" ${cloud_netplan_cfg}
    sed -i "s/${DEFAULT_INTERFACE}:/&\n            critical: true/" ${cloud_netplan_cfg}

    public_address_match_ip_line=$(grep -nm1 "${PUBLIC_NODE_IP_ADDRESS}/${PUBLIC_NODE_IP_NETMASK}" ${cloud_netplan_cfg} | cut -d: -f1)
    if [ -n "${public_address_match_ip_line}" ] ; then
        sed -i "$((${public_address_match_ip_line}-1)),$((${public_address_match_ip_line}))d" ${cloud_netplan_cfg}
    fi

cat << EOF >> ${cloud_netplan_cfg}
    bridges:
        br-public:
            dhcp4: false
            interfaces:
            - ${PUBLIC_INTERFACE}
            - veth-br
            addresses:
            - ${PUBLIC_NODE_IP_ADDRESS}/${PUBLIC_NODE_IP_NETMASK}
EOF

# Remove Tunnel interface from netplan
if [[ $TUNNEL_INTERFACE_NETPLAN_MANAGE == false ]]; then
    sed -i "/        ${TUNNEL_INTERFACE}/,/            set-name: ${TUNNEL_INTERFACE}/d" ${cloud_netplan_cfg}
fi

    netplan --debug apply

    # NOTE(vsaienko): give some time to apply changes
    sleep 15

# Remove Tunnel interface from netplan
if [[ $TUNNEL_INTERFACE_NETPLAN_MANAGE == false ]]; then
    ip addr flush ${TUNNEL_INTERFACE}
    ip link set ${TUNNEL_INTERFACE} up
fi

}

function collect_ceph_metadata {
    local ceph_osd_node
    ceph_osd_node=$(kubectl get nodes -l role=ceph-osd-node -o jsonpath={.items[?\(@.metadata.name==\"$(hostname)\"\)].metadata.name})

    if [[ -f /usr/share/metadata/ceph.yaml && ${ceph_osd_node} ]]; then
        HW_METADATA="{\"ceph\": {\"$(hostname)\": \"$(base64 -w 0 /usr/share/metadata/ceph.yaml)\"}}"
        ceph_store_drive=$(cat /usr/share/metadata/ceph.yaml | egrep '\- name\: ' | awk '{print $3}')
        if [[ -b /dev/${ceph_store_drive} ]]; then
            sgdisk --zap-all /dev/${ceph_store_drive}
        fi
    fi
}

function wait_for_pods {
    local namespace=$1
    local component_filter=${2:-""}
    local timeout=${3:-900}
    local delay=${4:-30}


    end=$(date +%s)
    end=$((end + timeout))
    while true; do
        sleep $delay
        if kubectl get pods --namespace=${namespace} $component_filter 2>&1 |grep -q 'No resources found'; then
            continue
        fi

        kubectl get pods --namespace=${namespace} $component_filter -o json | jq -r \
            '.items[].status.phase' | grep Pending > /dev/null && \
            PENDING="True" || PENDING="False"

        query='.items[]|select(.status.phase=="Running")'
        query="$query|.status.containerStatuses[].ready"
        kubectl get pods --namespace=${namespace} $component_filter -o json | jq -r "$query" | \
            grep false > /dev/null && READY="False" || READY="True"

        kubectl get jobs --namespace=${namespace} $component_filter -o json | jq -r \
            '.items[] | .spec.completions == .status.succeeded' | \
            grep false > /dev/null && JOBR="False" || JOBR="True"
        [ $PENDING == "False" -a $READY == "True" -a $JOBR == "True" ] && \
            break || true
        sleep 5
        now=$(date +%s)
        if [ $now -gt $end ] ; then
            echo "Containers failed to start after $timeout seconds"
            echo
            kubectl get pods --namespace ${namespace} $component_filter -o wide
            echo
            if [ $PENDING == "True" ] ; then
                echo "Some pods are in pending state:"
                kubectl get pods $component_filter --field-selector=status.phase=Pending -n ${namespace} -o wide
            fi
            [ $READY == "False" ] && echo "Some pods are not ready"
            [ $JOBR == "False" ] && echo "Some jobs have not succeeded"
            exit -1
        fi
    done
}

function wait_for_os_component {
    local component=$1
    echo "Waiting for: $component"
    wait_for_pods openstack "-l application=$component"
    echo "All resources for component $component Ready"
}

function install_release_controllers {
    if [[ -f "${RELEASE_OPENSTACK_K8S}/release/ci/50-nodemaintenance.yaml" ]]; then
        kubectl apply -f ${RELEASE_OPENSTACK_K8S}/release/ci/50-nodemaintenance.yaml
    fi

    for release_object in $(ls ${RELEASE_OPENSTACK_K8S}/release/*.yaml -1); do
        info "Installing ${release_object}"
        function apply_retry {
            kubectl apply -f ${release_object}
        }
        retry 3 "Can't apply ${release_object}" apply_retry
    done
}

function prepare_dns_and_ssl {

    local tmpd
    tmpd=$(mktemp -d)

    info "Preparing DNS and SSL configuration."

    kubectl apply -f ${RELEASE_OPENSTACK_K8S}/release/ci/30-coredns.yaml

    wait_for_pods coredns "" 300

    EXTERNAL_DNS_IP=$(kubectl -n coredns get services coredns-coredns -o jsonpath='{.status.loadBalancer.ingress[].ip}')
    if [ -z ${EXTERNAL_DNS_IP} ]; then
        EXTERNAL_DNS_IP=$(kubectl -n coredns get services coredns-coredns -o jsonpath='{.spec.clusterIP}')
    fi

    LOCAL_DOMAIN=$(kubectl get configmap -n kube-system coredns -o jsonpath='{.data.Corefile}' | awk '/in-addr.arpa.*{/ {print $2}')

    pushd $tmpd
    info "Generating SSL certificates."

    for ssl_app in cfssl cfssljson; do
        curl --retry 5 --retry-delay 10 -L ${MIRANTIS_ARTIFACTORY_URL}/openstack/bin/utils/cfssl/${ssl_app} -o ${tmpd}/${ssl_app} && chmod +x ${tmpd}/${ssl_app}
    done

    # Generate SSL certs
    yq w ${CSR_CONF} "CN" "*.openstack.svc.${LOCAL_DOMAIN}" | \
    yq w -j - "hosts[+]" "*.openstack.svc.${LOCAL_DOMAIN}" | \
    ${tmpd}/cfssl gencert -ca=${CERT_REPO_DIR}/ca.crt -ca-key=${CERT_REPO_DIR}/ca.key - | ${tmpd}/cfssljson -bare ${SSL_BARE_NAME}
    mv ${tmpd}/${SSL_BARE_NAME}* ${CERT_REPO_DIR}
    popd
}

function install_3rd_party {

    info "Installing metallb."
    kubectl apply -f ${RELEASE_OPENSTACK_K8S}/release/3rd-party/30-metallb.yaml
    # NOTE(vsaienko): unless PRODX-7154 is resolved, update calico daemonset to satisfy metallb podsecuritypolicy.
    kubectl -n kube-system patch ds calico-node --type='json' -p='[{"op": "replace", "path": "/spec/template/spec/containers/0/securityContext/allowPrivilegeEscalation", "value": true}]'
}

function install_ceph {

    info "Installing Ceph."
    ceph_osd_name=$(kubectl get nodes -l role=ceph-osd-node -o=jsonpath='{.items[*].metadata.name}')
    sed -i "s/Values.CHANGE_ME_node1_name/${ceph_osd_name}/" ${CEPH_CLUSTER_FILE}
    yq w -i ${CEPH_CLUSTER_FILE} "spec.network.clusterNet" ${STORAGE_CLUSTER_CIDR}
    yq w -i ${CEPH_CLUSTER_FILE} "spec.network.publicNet" ${STORAGE_PUBLIC_CIDR}
    yq w -i ${CEPH_CLUSTER_FILE} "spec.nodes[0].devices[0].name" $(cat /usr/share/metadata/ceph.yaml | egrep '\- name\: ' | awk '{print $3}')
    kubectl apply -f ${CEPH_CLUSTER_FILE}

    function get_ceph_secret_retry {
        kubectl -n openstack-ceph-shared get secrets openstack-ceph-keys
    }
    retry 30 "Get secret openstack-ceph-keys failed" get_ceph_secret_retry

# NOTE(okononenko): do scale deploy for ceph-controller here because it not relised as config yet
# TODO(okononenko): drop this when number of replicas for ceph controller might be specified via chart values. Related Task:PRODX-18984
    kubectl scale --replicas=1 deployments/ceph-controller -n ceph-lcm-mirantis
}

function install_openstack {
    gateway_ip=${PUBLIC_INTERFACE_IP:-$(ip addr show dev br-public | grep -Po 'inet \K[\d.]+' | egrep -v "127.0.|172.17")}

    sed -i "s/cluster.local/${LOCAL_DOMAIN}/g" ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml

    yq w -i ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.neutron.dns_servers[+]" ${EXTERNAL_DNS_IP}
    yq w -i -- ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.ssl.public_endpoints.ca_cert" "$(cat ${CERT_REPO_DIR}/ca.crt)"
    yq w -i -- ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.ssl.public_endpoints.api_cert" "$(cat ${CERT_REPO_DIR}/api.pem)"
    yq w -i -- ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.ssl.public_endpoints.api_key" "$(cat ${CERT_REPO_DIR}/api-key.pem)"
    yq w -i ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.neutron.floating_network.subnet.gateway" ${gateway_ip}
    yq w -i ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.neutron.tunnel_interface" ${TUNNEL_INTERFACE}
    yq w -i ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.nova.live_migration_interface" ${LIVE_MIGRATION_INTERFACE}
    yq w -i ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.features.services[+]" "object-storage"
    #NOTE (ohryhorov): libvirt_type is set to qemu because kvm is not supported by average flavors in AWS
    yq w -i ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml "spec.services.compute.nova.values.conf.nova.libvirt.virt_type" "qemu"

    kubectl apply -f ${RELEASE_OPENSTACK_K8S}/examples/osdpl/${OPENSTACK_CONTEXT_NAME}.yaml

    wait_for_pods openstack "-l app.kubernetes.io/name=cache" 1800

    for component in mariadb rabbitmq memcached openvswitch libvirt keystone glance cinder nova neutron barbican octavia cinder designate heat; do
        wait_for_os_component $component
    done
    wait_for_pods openstack

    kubectl -n openstack create job --from=cronjob/nova-cell-setup  nova-cell-setup-pd01-$(cat /dev/urandom | tr -dc a-z | head -c3)
    info "Openstack was deployed successfully..."
}

function configure_public_resolve {
    local tmpd
    tmpd=$(mktemp -d)

    openstack_ingress_ip=$(kubectl get services ingress -n openstack -o jsonpath='{.status.loadBalancer.ingress[0].ip}')
    if [ -z ${openstack_ingress_ip} ]; then
        openstack_ingress_ip=$(kubectl get services ingress -n openstack -o jsonpath='{.spec.clusterIP}')
    fi

    kubectl -n kube-system get configmap coredns -o yaml > ${tmpd}/coredns-config.yaml
    yq w -i ${tmpd}/coredns-config.yaml "data.Corefile" "$(kubectl -n kube-system get configmap coredns -ojsonpath='{.data.Corefile}')
it.just.works:53 {
    errors
    cache 30
    forward . ${EXTERNAL_DNS_IP}
}"
    kubectl -n kube-system apply -f ${tmpd}/coredns-config.yaml
    kubectl -n kube-system delete pod -l k8s-app=kube-dns
    sed -i "s/1.2.3.4/${openstack_ingress_ip}/g" -i ${RELEASE_OPENSTACK_K8S}/release/ci/30-coredns.yaml
    kubectl apply -f ${RELEASE_OPENSTACK_K8S}/release/ci/30-coredns.yaml
}

function deploy_finished {
    touch ${_DEPLOY_FINISHED_LOCK}
}

function write_trymos_info {
    local distro_info
    local os_controller_version
    local ceph_controller_version
    local os_version
    local keystone_client_pod
    local os_admin_username
    local os_admin_password

    distro_info=$(lsb_release -sd)
    os_controller_version=$(kubectl -n osh-system get helmbundles openstack-operator -o jsonpath='{.status.releaseStatuses.openstack-operator.version}')
    ceph_controller_version=$(kubectl -n osh-system get helmbundles ceph-operator -o jsonpath='{.status.releaseStatuses.ceph-operator.version}')
    os_version=$(kubectl -n openstack get osdpl osh-dev -o jsonpath='{.spec.openstack_version}')
    keystone_client_pod=$(kubectl -n openstack get pods -l application=keystone,component=client -o jsonpath='{.items[*].metadata.name}')
    os_admin_username=$(kubectl -n openstack exec -it $keystone_client_pod -- bash -c "echo \$OS_USERNAME")
    os_admin_password=$(kubectl -n openstack exec -it $keystone_client_pod -- bash -c "echo \$OS_PASSWORD")

    cat << EOF > ${_TRYMOS_INFO_FILE}
####################################################
#################### TryMOS ########################
####################################################

############### Components Versions ################

OS:                       ${distro_info}
TryMOS Version:           ${TRYMOS_VERSION}
OpenStack Controller:     ${os_controller_version}
Ceph Controller:          ${ceph_controller_version}
OpenStack Version:        ${os_version}

############# UCP access information ###############
UCP Admin Username:       ${UCP_USERNAME}
UCP Admin Password:       ${UCP_PASSWORD}

########### OpenStack access information ###########
OpenStack Admin Username: ${os_admin_username}
OpenStack Admin Password: ${os_admin_password}

####################################################
EOF

    cat <<EOF > /etc/update-motd.d/99-trymos-info
#!/bin/sh
cat ${_TRYMOS_INFO_FILE}
EOF

    chmod +x /etc/update-motd.d/99-trymos-info

    cat ${_TRYMOS_INFO_FILE}
}

case "$NODE_TYPE" in
    # Please keep the "prepare_metadata_files", "disable-rp-filter", "network_config" and "prepare_network" functions
    # at the very beginning in the same order.
    trymos)
        disable_rp_filter
        configure_virt_public_iface
        network_config
        prepare_network
        prepare_docker_config
        install_required_packages
        configure_ntp
        configure_atop
        workaround_default_forward_policy
        install_docker
        swarm_init
        create_ucp_config
        cache_images
        install_ucp
        download_bundles
        rm_ucp_config
        install_kubectl
        wait_for_node
        set_node_labels
        generate_ceph_metadata
        collect_ceph_metadata
        configure_contrack
        disable_iptables_for_bridges
        nested_virt_config
        disable_master_taint
        install_release_controllers
        prepare_dns_and_ssl
        install_3rd_party
        install_ceph
        install_openstack
        configure_public_resolve
        write_trymos_info
        deploy_finished
        ;;
    *)
        echo "Usage: $0 {trymos}"
        exit 1
esac

