#!/bin/bash
set -x
# allow access to the local variables from prepare-metadata.py
set -a

#
# Variables in this block are passed from heat template
#
CONTROL_NETWORK_CIDR=$control_network_cidr
PUBLIC_INTERFACE=$private_floating_interface
PUBLIC_INTERFACE_IP=$private_floating_interface_ip
PUBLIC_INTERFACE_CIDR=$private_floating_network_cidr
PUBLIC_INTERFACE_NETMASK=$(echo ${PUBLIC_INTERFACE_CIDR} | cut -d'/' -f2)
STORAGE_BACKEND_INTERFACE=$storage_backend_interface
STORAGE_BACKEND_INTERFACE_IP=$storage_backend_network_interface_ip
STORAGE_BACKEND_NETWORK=$storage_backend_network_cidr
STORAGE_BACKEND_NETWORK_NETMASK=$(echo ${STORAGE_BACKEND_NETWORK} | cut -d'/' -f2)
STORAGE_FRONTEND_INTERFACE=$storage_frontend_interface
STORAGE_FRONTEND_INTERFACE_IP=$storage_frontend_network_interface_ip
STORAGE_FRONTEND_NETWORK=$storage_frontend_network_cidr
STORAGE_FRONTEND_NETWORK_NETMASK=$(echo ${STORAGE_FRONTEND_NETWORK} | cut -d'/' -f2)
IRONIC_BAREMETAL_NETWORK=$ironic_baremetal_network_cidr
IRONIC_BAREMETAL_INTERFACE_IP=$ironic_baremetal_interface_ip
IRONIC_BAREMETAL_NETWORK_NETMASK=$(echo ${IRONIC_BAREMETAL_NETWORK} | cut -d'/' -f2)
IRONIC_BAREMETAL_INTERFACE=$(ip route get ${IRONIC_BAREMETAL_NETWORK%/*} | awk '/^broadcast / {print $4}')
IRONIC_BAREMETAL_TUNNEL_NETWORK=$ironic_baremetal_tunnel_cidr
TUNNEL_INTERFACE_IP=$tunnel_interface_ip
TUNNEL_INTERFACE_NETWORK_NETMASK=$(ip -o addr show |grep -w ${TUNNEL_INTERFACE_IP} | awk '{print $4}' |awk -F '/' '{print $2}')
TUNNEL_INTERFACE=$(ip -o addr show |grep -w ${TUNNEL_INTERFACE_IP}/${TUNNEL_INTERFACE_NETWORK_NETMASK} | awk '{print $2}')
NODE_TYPE=$node_type
UCP_MASTER_HOST=$ucp_master_host
NODE_METADATA='$node_metadata'
DOCKER_EE_URL='$docker_ee_url'
DOCKER_EE_RELEASE='$docker_ee_release'
DOCKER_UCP_IMAGE='$docker_ucp_image'
FLOATING_NETWORK_PREFIXES=$private_floating_network_cidr
IRONIC_MT_ENABLED=$ironic_mt_enabled
IRONIC_BAREMETAL_NETWORK_PREFIX=$(sed 's/[0-9]*\/[0-9]*$//' <<< $IRONIC_BAREMETAL_NETWORK)
IRONIC_BAREMETAL_TUNNEL_NETWORK_PREFIX=$(sed 's/[0-9]*\/[0-9]*$//' <<< $IRONIC_BAREMETAL_TUNNEL_NETWORK)
HUGE_PAGES=$huge_pages
#
# End of block
#

DOCKER_DEFAULT_ADDRESS_POOL=${DOCKER_DEFAULT_ADDRESS_POOL:-10.10.1.0/16}
# DOCKER_DEFAULT_ADDRESS_SIZE have to be less then netmask in DOCKER_DEFAULT_ADDRESS_POOL because
# to the fact that actual netmask for docker_gwbridge is given from it
DOCKER_DEFAULT_ADDRESS_SIZE=${DOCKER_DEFAULT_ADDRESS_SIZE:-24}
DOCKER_EE_RELEASE=${DOCKER_EE_RELEASE:-stable-19.03}
DOCKER_EE_PACKAGES='docker-ee'
DOCKER_RELEASE='stable'
DOCKER_PACKAGES='docker-ce'
if [ -n "${DOCKER_EE_URL}" ]; then
    DOCKER_URL="${DOCKER_EE_URL}"
    DOCKER_RELEASE="${DOCKER_EE_RELEASE}"
    DOCKER_PACKAGES="${DOCKER_EE_PACKAGES}"
fi
CONTROL_NETWORK_CIDR=${CONTROL_NETWORK_CIDR:-"10.10.0.0/24"}
CONTROL_IP_ADDRESS=$(ip route get ${CONTROL_NETWORK_CIDR%/*} | head -n1 | fgrep -v ' via ' | awk '/ src / {print $6}')
PUBLIC_INTERFACE=${PUBLIC_INTERFACE:-ens4}
UCP_USERNAME=${UCP_USERNAME:-admin}
UCP_PASSWORD=${UCP_PASSWORD:-administrator}
OS_CODENAME=$(lsb_release -c -s)
KUBECTL_VERSION=${KUBECTL_VERSION:-1.18.8}
NODE_DEPLOYMENT_RETRIES=${NODE_DEPLOYMENT_RETRIES:-15}
FLOATING_NETWORK_PREFIXES=${FLOATING_NETWORK_PREFIXES:-10.11.12.0/24}
PUBLIC_INTERFACE=${PUBLIC_INTERFACE:-ens4}
UCP_MASTER_HOST=${UCP_MASTER_HOST:-${CONTROL_IP_ADDRESS}}
UCP_IP_ADDRESS=${UCP_IP_ADDRESS:-$CONTROL_IP_ADDRESS}
NTP_SERVERS=${NTP_SERVERS:-"ldap.scc.mirantis.net ldap.bud.mirantis.net"}
DEFAULT_INTERFACE=$(ip route show default | awk '{print $5}')
IRONIC_BAREMETAL_VXLAN_INTERFACE='vxlan10'
HUGE_PAGES_1G_SIZE=$(echo "${HUGE_PAGES}" | awk -F ',' '{print $1}')
HUGE_PAGES_2Mb_SIZE=$(echo "${HUGE_PAGES}" | awk -F ',' '{print $2}')
CONFIGURE_HUGE_PAGES=false
TUNNEL_INTERFACE_NETPLAN_MANAGE=true
if [[ "${HUGE_PAGES_1G_SIZE}" -gt 0 ]] || [[ "${HUGE_PAGES_2Mb_SIZE}" -gt 0 ]]; then
    CONFIGURE_HUGE_PAGES=true
    TUNNEL_INTERFACE_NETPLAN_MANAGE=false
fi

function retry {
    local retries=$1
    shift
    local msg="$1"
    shift

    local count=0
    until "$@"; do
        exit=$?
        wait=$((2 ** $count))
        count=$(($count + 1))
        if [ $count -lt $retries ]; then
            echo "Retry $count/$retries exited $exit, retrying in $wait seconds..."
            sleep $wait
        else
            echo "Retry $count/$retries exited $exit, no more retries left."
            echo "$msg"
            return $exit
        fi
    done
    return 0
}

function wait_condition_send {
    local status=${1:-SUCCESS}
    local reason=${2:-\"empty\"}
    local data=${3:-\"empty\"}
    local data_binary="{\"status\": \"$status\", \"reason\": \"$reason\", \"data\": $data}"
    echo "Trying to send signal to wait condition 5 times: $data_binary"
    WAIT_CONDITION_NOTIFY_EXIT_CODE=2
    i=0
    while (( ${WAIT_CONDITION_NOTIFY_EXIT_CODE} != 0 && ${i} < 5 )); do
        $wait_condition_notify -k --data-binary "$data_binary" && WAIT_CONDITION_NOTIFY_EXIT_CODE=0 || WAIT_CONDITION_NOTIFY_EXIT_CODE=2
        i=$((i + 1))
        sleep 1
    done
    if (( ${WAIT_CONDITION_NOTIFY_EXIT_CODE} !=0 && "${status}" == "SUCCESS" ))
    then
        status="FAILURE"
        reason="Can't reach metadata service to report about SUCCESS."
    fi
    if [ "$status" == "FAILURE" ]; then
        exit 1
    fi
}


function configure_atop {
    sed -i 's/INTERVAL=600/INTERVAL=60/' /usr/share/atop/atop.daily
    systemctl restart atop
}

function install_required_packages {
    # Disable unattended upgrades
    sed -i 's/1/0/' /etc/apt/apt.conf.d/20auto-upgrades
    local pkg_list="apt-transport-https ca-certificates curl software-properties-common jq unzip atop iptables-persistent socat ntp"

    if [[ "${CONFIGURE_HUGE_PAGES}" == true ]]; then
        pkg_list="$pkg_list dpdk-igb-uio-dkms"
    fi
    function install_retry {
        apt update
        export DEBIAN_FRONTEND=noninteractive; apt install -y $pkg_list
    }
    retry 10 "Failed to install required packages" install_retry
}


function install_docker {
    function install_retry {
        curl --retry 6 --retry-delay 5 -fsSL "${DOCKER_URL}/gpg" | sudo apt-key add -
        add-apt-repository "deb [arch=amd64] ${DOCKER_URL}/ ${OS_CODENAME} ${DOCKER_RELEASE}"
        apt update
        apt install -y ${DOCKER_PACKAGES}
    }
    retry 10 "Failed to install docker" install_retry
}

function prepare_docker_config {
    mkdir -p /etc/docker
    cat <<EOF > /etc/docker/daemon.json
{
  "default-address-pools": [
    { "base": "${DOCKER_DEFAULT_ADDRESS_POOL}", "size": ${DOCKER_DEFAULT_ADDRESS_SIZE} }
  ],
  "log-driver": "json-file",
  "log-opts": {
    "max-size": "1g",
    "max-file": "3"
  }
}
EOF

}

function pull_image {
    local image="$1"
    function pull_image_retry {
        docker pull ${image}
    }
    retry 10 "Can't pool docker image $image" pull_image_retry
}

function cache_images {

    # Cache UCP image with retries
    pull_image "${DOCKER_UCP_IMAGE}"

    local image_prefix=${DOCKER_UCP_IMAGE%/*}
    local image_name=${DOCKER_UCP_IMAGE##*/}
    if [[ "$image_name" =~ .*3.3.* ]]; then
        for image in $(docker container run --rm --name ucp -v /var/run/docker.sock:/var/run/docker.sock ${DOCKER_UCP_IMAGE} images --list); do
            pull_image "${image_prefix}/${image##*/}"
            docker tag ${image_prefix}/${image##*/} ${image}
        done
    fi
}

function install_ucp {
    local tmpd
    tmpd=$(mktemp -d)
    function docker_run_retry {
        docker container run --rm --name ucp \
        -v /var/run/docker.sock:/var/run/docker.sock \
        ${DOCKER_UCP_IMAGE} install \
        --host-address $UCP_IP_ADDRESS \
        --admin-username $UCP_USERNAME \
        --admin-password $UCP_PASSWORD \
        --existing-config
    }

    retry 10 "Can't bring up docker UCP container" docker_run_retry
}

function download_bundles {
    local tmpd
    tmpd=$(mktemp -d)

    function download_bundles_retry {
    # Download the client certificate bundle
        curl --retry 6 --retry-delay 5 -k -H "Authorization: Bearer $AUTHTOKEN" https://${UCP_MASTER_HOST}/api/clientbundle -o ${tmpd}/bundle.zip
    }

    function get_authtoken_retry {
    # Download the bundle https://docs.docker.com/ee/ucp/user-access/cli/
    # Create an environment variable with the user security token
        AUTHTOKEN=$(curl --retry 6 --retry-delay 5 -sk -d '{"username":"'$UCP_USERNAME'","password":"'$UCP_PASSWORD'"}' https://${UCP_MASTER_HOST}/auth/login | jq -r .auth_token)
        if [ -z ${AUTHTOKEN} ]; then
            return -1
        fi
    }

    retry 10 "Can't get AUTHTOKEN from master." get_authtoken_retry
    retry 10 "Can't download bundle file from master." download_bundles_retry

    pushd $tmpd
    # Unzip the bundle.
    unzip bundle.zip

    # Run the utility script.
    eval "$(<env.sh)"
    mkdir -p /etc/kubernetes /root/.kube/
    cp kube.yml /etc/kubernetes/admin.conf
    cp kube.yml /root/.kube/config
    popd
}

function wait_for_node {
    function retry_wait {
        kubectl --kubeconfig /etc/kubernetes/admin.conf get nodes |grep -w Ready |awk '{print $1}' |grep -q $(hostname)
    }
    retry $NODE_DEPLOYMENT_RETRIES "The node didn't come up." retry_wait
}

function join_node {
    local type=${1}
    function retry_join_node {
        env -i $(docker swarm join-token $type |grep 'docker swarm join' | xargs)
    }
    retry 10 "Failed to join node to swarm" retry_join_node
}

function create_ucp_config {
    echo "
[scheduling_configuration]
    enable_admin_ucp_scheduling = true
    default_node_orchestrator = \"kubernetes\"
[cluster_config]
    dns = [\"172.18.208.44\"]
" | docker config create com.docker.ucp.config -
}

function swarm_init {
    docker swarm init --advertise-addr ${UCP_IP_ADDRESS}
}

function rm_ucp_config {
    docker config rm com.docker.ucp.config
}

function install_kubectl {
    local kubectl=/usr/local/bin/kubectl
    curl --retry 6 --retry-delay 5 -L https://artifactory.mcp.mirantis.net/artifactory/binary-dev-kaas-local/core/bin/mirror/kubectl-${KUBECTL_VERSION}-linux -o ${kubectl}
    chmod +x ${kubectl}
cat << EOF >> ~/.bashrc
source /usr/share/bash-completion/bash_completion
source <(kubectl completion bash)
EOF
}

function configure_ntp {
    cat << EOF > /etc/ntp.conf
# ntpd will only synchronize your clock.

# For details, see:
# - the ntp.conf man page
# - http://support.ntp.org/bin/view/Support/GettingStarted

# Specify one or more NTP servers.
EOF

    for server in $NTP_SERVERS; do
cat << EOF >> /etc/ntp.conf
server ${server} iburst
EOF
    done

    cat << EOF >> /etc/ntp.conf
# Set general access to this service
restrict -4 default kod nomodify notrap nopeer noquery
restrict -6 default kod nomodify notrap nopeer noquery

# Allow access from localhost
restrict 127.0.0.1
restrict ::1

# Do not listen on any interface address by default
interface ignore wildcard
interface listen lo
interface listen ${DEFAULT_INTERFACE}

# Location of drift file
driftfile /var/lib/ntp/ntp.drift
EOF

    systemctl restart ntp
}

function prepare_network {
    if [ -z "${CONTROL_IP_ADDRESS}" ]; then
        wait_condition_send "FAILURE" "CONTROL_IP_ADDRESS is not found for the network ${CONTROL_NETWORK_CIDR}"
        exit 1
    fi

    systemctl restart systemd-resolved
    # Make sure local hostname is present in /etc/hosts
    sed -i "s/127.0.0.1 localhost/127.0.0.1 localhost\n${CONTROL_IP_ADDRESS} $(hostname)/" /etc/hosts
}

function workaround_default_forward_policy {
    cat << EOF > /etc/iptables/rules.v4
*filter
:DOCKER-USER - [0:0]
EOF
    for net in $FLOATING_NETWORK_PREFIXES; do
cat << EOF >> /etc/iptables/rules.v4
-A DOCKER-USER -d ${net} -j ACCEPT
-A DOCKER-USER -s ${net} -j ACCEPT
EOF
    done

cat << EOF >> /etc/iptables/rules.v4
-A DOCKER-USER -j RETURN
COMMIT
EOF

#Allow access to Internet from VMs for virtualized environment.
cat << EOF >> /etc/iptables/rules.v4
*nat
:POSTROUTING ACCEPT - [0:0]
EOF
    for net in $FLOATING_NETWORK_PREFIXES; do
cat << EOF >> /etc/iptables/rules.v4
-A POSTROUTING -s ${net} -o ${DEFAULT_INTERFACE} -j MASQUERADE
EOF
    done

cat << EOF >> /etc/iptables/rules.v4
COMMIT
EOF
    sudo netfilter-persistent reload
}

function disable_rp_filter {
    # Run this func before "network_config" to create new interfaces with the default rp_filter value
    cat << EOF > /etc/sysctl.d/99-disable-rp-filter.conf
net.ipv4.conf.all.rp_filter=0
net.ipv4.conf.default.rp_filter=0
EOF
    sysctl -p /etc/sysctl.d/99-disable-rp-filter.conf
}

function configure_contrack {
    cat << EOF > /etc/sysctl.d/100-contrackd.conf
net.netfilter.nf_conntrack_log_invalid=255
net.netfilter.nf_conntrack_tcp_be_liberal=1
EOF

    sysctl -p /etc/sysctl.d/100-contrackd.conf
}

# NOTE(vsaienko): disable calling iptables when sending packets via bridge.
function disable_iptables_for_bridges {
    cat << EOF > /etc/sysctl.d/101-bridge-nf-call-iptables.conf
net.bridge.bridge-nf-call-ip6tables=0
net.bridge.bridge-nf-call-iptables=0
net.bridge.bridge-nf-call-arptables=0
EOF

    sysctl -p /etc/sysctl.d/101-bridge-nf-call-iptables.conf
}

function network_config {
    PUBLIC_NODE_IP_ADDRESS=${PUBLIC_INTERFACE_IP:-$(ip addr show dev ${PUBLIC_INTERFACE} | grep -Po 'inet \K[\d.]+' | egrep -v "127.0.|172.17")}
    PUBLIC_NODE_IP_NETMASK=${PUBLIC_INTERFACE_NETMASK:-$(ip addr show dev ${PUBLIC_INTERFACE} | grep -Po 'inet \K[\d.]+\/[\d]+' | egrep -v "127.0.|172.17" | cut -d'/' -f2)}

    local public_interface=${1:-${PUBLIC_INTERFACE}}
    local cloud_netplan_cfg="/etc/netplan/50-cloud-init.yaml"
    local match_ip_line

    DEBIAN_FRONTEND=noninteractive apt -y install bridge-utils

cat << EOF > /etc/systemd/network/10-veth-phy-br.netdev
[NetDev]
Name=veth-phy
Kind=veth
[Peer]
Name=veth-br
EOF

if [[ -n ${IRONIC_BAREMETAL_INTERFACE} ]]; then

cat << EOF > /etc/systemd/network/11-veth-bm.netdev
[NetDev]
Name=veth-bm
Kind=veth
[Peer]
Name=veth-bm-br
EOF

cat << EOF > /etc/systemd/network/12-veth-vbmc.netdev
[NetDev]
Name=veth-vbmc
Kind=veth
[Peer]
Name=veth-vbmc-br
EOF
fi

    sed -i 's/.*ethernets:.*/&\n        veth-phy: {}/' ${cloud_netplan_cfg}
    sed -i 's/.*ethernets:.*/&\n        veth-br: {}/' ${cloud_netplan_cfg}
if [[ -n ${IRONIC_BAREMETAL_INTERFACE} ]]; then
    sed -i 's/.*ethernets:.*/&\n        veth-bm: {}/' ${cloud_netplan_cfg}
    sed -i 's/.*ethernets:.*/&\n        veth-bm-br: {}/' ${cloud_netplan_cfg}
    sed -i 's/.*ethernets:.*/&\n        veth-vbmc: {}/' ${cloud_netplan_cfg}
    sed -i 's/.*ethernets:.*/&\n        veth-vbmc-br: {}/' ${cloud_netplan_cfg}

    if [[ "${IRONIC_MT_ENABLED,,}" == true ]] ; then
        sed -i "s/.*ethernets:.*/&\n        ${IRONIC_BAREMETAL_VXLAN_INTERFACE}: {}/" ${cloud_netplan_cfg}
        sed -i 's/.*ethernets:.*/&\n        veth-pxe: {}/' ${cloud_netplan_cfg}
        sed -i 's/.*ethernets:.*/&\n        veth-pxe-br: {}/' ${cloud_netplan_cfg}
cat << EOF > /etc/systemd/network/13-veth-pxe.netdev
[NetDev]
Name=veth-pxe
Kind=veth
[Peer]
Name=veth-pxe-br
EOF
        sed -i "s/- ${IRONIC_BAREMETAL_NETWORK_PREFIX}\([0-9]*\)/- ${IRONIC_BAREMETAL_TUNNEL_NETWORK_PREFIX}\1/" ${cloud_netplan_cfg}
cat << EOF >> ${cloud_netplan_cfg}
    vlans:
        pxe.1000:
            id: 1000
            link: veth-pxe
            addresses:
            - ${IRONIC_BAREMETAL_INTERFACE_IP}/${IRONIC_BAREMETAL_NETWORK_NETMASK}
EOF
    else
        ironic_baremetal_address_match_ip_line=$(grep -nm1 "${IRONIC_BAREMETAL_INTERFACE_IP}/${IRONIC_BAREMETAL_NETWORK_NETMASK}" ${cloud_netplan_cfg} | cut -d: -f1)
        sed -i "$((${ironic_baremetal_address_match_ip_line}-1)),$((${ironic_baremetal_address_match_ip_line}))d" ${cloud_netplan_cfg}
    fi
fi

    public_address_match_ip_line=$(grep -nm1 "${PUBLIC_NODE_IP_ADDRESS}/${PUBLIC_NODE_IP_NETMASK}" ${cloud_netplan_cfg} | cut -d: -f1)
    sed -i "$((${public_address_match_ip_line}-1)),$((${public_address_match_ip_line}))d" ${cloud_netplan_cfg}

cat << EOF >> ${cloud_netplan_cfg}
    bridges:
        br-public:
            dhcp4: false
            interfaces:
            - ${PUBLIC_INTERFACE}
            - veth-br
            addresses:
            - ${PUBLIC_NODE_IP_ADDRESS}/${PUBLIC_NODE_IP_NETMASK}
EOF

if [[ -n ${IRONIC_BAREMETAL_INTERFACE} ]]; then
cat << EOF >> ${cloud_netplan_cfg}
        br-baremetal:
            dhcp4: false
            interfaces:
            - veth-bm-br
            - veth-vbmc-br
EOF
    if [[ "${IRONIC_MT_ENABLED,,}" != true ]] ; then
cat << EOF >> ${cloud_netplan_cfg}
            - ${IRONIC_BAREMETAL_INTERFACE}
            addresses:
            - ${IRONIC_BAREMETAL_INTERFACE_IP}/${IRONIC_BAREMETAL_NETWORK_NETMASK}
EOF
    else
cat << EOF >> ${cloud_netplan_cfg}
            - ${IRONIC_BAREMETAL_VXLAN_INTERFACE}
            - veth-pxe-br
EOF

# Remove Tunnel interface from netplan
if [[ $TUNNEL_INTERFACE_NETPLAN_MANAGE == false ]]; then
    sed -i "/        ${TUNNEL_INTERFACE}/,/            set-name: ${TUNNEL_INTERFACE}/d" ${cloud_netplan_cfg}
fi

cat << EOF > /etc/systemd/system/ironic-vxlan-tunnel.service
[Unit]
Description=Ironic VXLAN tunnel
After=network.target

[Service]
Type=oneshot
RemainAfterExit=true
ExecStart=/sbin/ip link add ${IRONIC_BAREMETAL_VXLAN_INTERFACE} type vxlan id 10 group 239.1.1.10 dstport 0 dev ${IRONIC_BAREMETAL_INTERFACE}
ExecStart=/sbin/ip link set ${IRONIC_BAREMETAL_VXLAN_INTERFACE} up

[Install]
WantedBy=multi-user.target
EOF

        systemctl enable ironic-vxlan-tunnel.service
        systemctl start ironic-vxlan-tunnel.service
    fi
fi
    netplan --debug apply

    # NOTE(vsaienko): give some time to apply changes
    sleep 15
}

$functions_override

function set_node_labels {
    function set_node_labels_retry {
        kubectl patch node $(hostname) -p "{\"metadata\": ${NODE_METADATA}}"
    }
    retry 10 "Labeling node failed" set_node_labels_retry
}

HW_METADATA='{}'
# Place files specified in metadata to system.
# For example netplan.io metadata, the restart of services
# is not covered by script.
function prepare_metadata_files {
    /usr/sbin/prepare-metadata.py  --metadata-file /usr/share/metadata/lab-metadata.yaml
}

function collect_ceph_metadata {
    local ceph_osd_node
    ceph_osd_node=$(kubectl get nodes -l role=ceph-osd-node -o jsonpath={.items[?\(@.metadata.name==\"$(hostname)\"\)].metadata.name})

    if [[ -f /usr/share/metadata/ceph.yaml && ${ceph_osd_node} ]]; then
        HW_METADATA="{\"ceph\": {\"$(hostname)\": \"$(base64 -w 0 /usr/share/metadata/ceph.yaml)\"}}"
        ceph_store_drive=$(cat /usr/share/metadata/ceph.yaml | egrep '\- name\: vd?' | awk '{print $3}')
        if [[ -b /dev/${ceph_store_drive} ]]; then
            sgdisk --zap-all /dev/${ceph_store_drive}
        fi
    fi
}

function collect_interfaces_metadata {

    local if_metadata_file="/usr/share/metadata/interfaces.yaml"

    pci_id=$(ethtool -i ${TUNNEL_INTERFACE} |grep bus-info | awk '{print $2}')
    mac=$(cat $(find /sys/bus/pci/devices/${pci_id}/ -name net)/${TUNNEL_INTERFACE}/address)
    ip=${TUNNEL_INTERFACE_IP}/${TUNNEL_INTERFACE_NETWORK_NETMASK}

    cat << EOF > ${if_metadata_file}
${TUNNEL_INTERFACE}:
  pci_id: "${pci_id}"
  ip_addr: "${ip}"
  mac: "${mac}"
EOF

    HW_METADATA=$(echo ${HW_METADATA} | jq -cr ". += {\"interfaces\": {\"$(hostname)\": \"$(base64 -w 0 ${if_metadata_file})\"}}")

}

function nested_virt_config {

cat << EOF > /etc/modprobe.d/qemu-system-x86.conf
options kvm_intel nested=1
options kvm_intel enable_apicv=1
options kvm_intel ept=1
EOF

    modprobe -r kvm_intel || true
    modprobe kvm_intel nested=1

}

function configure_huge_pages {
    local nr_2Mb
    local nr_1G
    local apply_grub=false

    if [[ ! $(cat /etc/default/grub | grep -E ^GRUB_CMDLINE_LINUX=".*hugepagesz=2M.*") ]] && [[ "${HUGE_PAGES_2Mb_SIZE}" -gt 0 ]]; then
        nr_2Mb=$(( "${HUGE_PAGES_2Mb_SIZE}" / 2 ))
        echo "Configure GRUB kernel cmd to allocate ${nr_2Mb} 2Mb huge pages"
        echo "GRUB_CMDLINE_LINUX=\"\$GRUB_CMDLINE_LINUX hugepagesz=2M hugepages=$nr_2Mb\"" >> /etc/default/grub
        apply_grub=true
    fi

    if [[ ! $(cat /etc/default/grub | grep -E ^GRUB_CMDLINE_LINUX=".*hugepagesz=1G.*") ]] && [[ "${HUGE_PAGES_1G_SIZE}" -gt 0 ]]; then
        echo "Configure GRUB kernel cmd to allocate ${HUGE_PAGES_1G_SIZE} 1G huge pages"
        echo "GRUB_CMDLINE_LINUX=\"\$GRUB_CMDLINE_LINUX hugepagesz=1G hugepages=${HUGE_PAGES_1G_SIZE}\"" >> /etc/default/grub
        apply_grub=true
    fi

    if [[ "${apply_grub}" == true ]]; then
        update-grub
        cloud-init clean -r
    else
        echo "GRUB already configured with huge pages"
    fi
}

case "$NODE_TYPE" in
    # Please keep the "prepare_metadata_files", "disable-rp-filter", "network_config" and "prepare_network" functions
    # at the very beginning in the same order.
    ucp)
        prepare_metadata_files
        disable_rp_filter
        network_config
        prepare_network
        prepare_docker_config
        install_required_packages
        configure_ntp
        configure_atop
        workaround_default_forward_policy
        install_docker
        swarm_init
        create_ucp_config
        cache_images
        install_ucp
        download_bundles
        rm_ucp_config
        install_kubectl
        wait_for_node
        set_node_labels
        collect_ceph_metadata
        configure_contrack
        disable_iptables_for_bridges
        ;;
    master)
        nested_virt_config
        prepare_metadata_files
        disable_rp_filter
        network_config
        prepare_network
        prepare_docker_config
        install_required_packages
        configure_ntp
        configure_atop
        workaround_default_forward_policy
        install_docker
        cache_images
        download_bundles
        join_node manager
        install_kubectl
        wait_for_node
        set_node_labels
        collect_ceph_metadata
        configure_contrack
        disable_iptables_for_bridges
        collect_interfaces_metadata
        ;;
    worker)
        if [[ "${CONFIGURE_HUGE_PAGES}" == true ]]; then
            configure_huge_pages
        fi
        nested_virt_config
        prepare_metadata_files
        disable_rp_filter
        network_config
        prepare_network
        prepare_docker_config
        install_required_packages
        configure_ntp
        configure_atop
        workaround_default_forward_policy
        install_docker
        cache_images
        download_bundles
        join_node worker
        install_kubectl
        wait_for_node
        set_node_labels
        collect_ceph_metadata
        configure_contrack
        disable_iptables_for_bridges
        collect_interfaces_metadata
        ;;
    spare)
        prepare_metadata_files
        disable_rp_filter
        network_config
        prepare_network
        prepare_docker_config
        install_required_packages
        configure_ntp
        configure_atop
        install_docker
        cache_images
        download_bundles
        workaround_default_forward_policy
        configure_contrack
        disable_iptables_for_bridges
        ;;
    *)
        echo "Usage: $0 {ucp|master|worker}"
        exit 1
esac

wait_condition_send "SUCCESS" "Instance successfuly started." "${HW_METADATA}"
