import pytest

from devops.helpers.proc_enums import ExitCodes
from tcp_tests import logger
from tcp_tests import settings

LOG = logger.logger


class TestBackupRestoreGalera(object):
    """
    Created by https://mirantis.jira.com/browse/PROD-32674
    """
    @pytest.fixture()
    def handle_restore_params(self, reclass_actions):
        reclass_actions.add_key(
            'parameters.xtrabackup.client.restore_full_latest',
            '1',
            'cluster/*/openstack/database/init.yml')
        reclass_actions.add_bool_key(
            'parameters.xtrabackup.client.enabled',
            'True',
            'cluster/*/openstack/database/init.yml')
        reclass_actions.add_key(
            'parameters.xtrabackup.client.restore_from',
            'remote',
            'cluster/*/openstack/database/init.yml')
        yield
        reclass_actions.delete_key(
            'parameters.xtrabackup.client.restore_full_latest',
            'cluster/*/openstack/database/init.yml')
        reclass_actions.delete_key(
            'parameters.xtrabackup.client.enabled',
            'cluster/*/openstack/database/init.yml')
        reclass_actions.delete_key(
            'parameters.xtrabackup.client.restore_from',
            'cluster/*/openstack/database/init.yml')

    def _get_cfg_fqn(self, salt):
        salt_master = salt.local("I@salt:master", "network.get_fqdn")
        return salt_master['return'][0].keys()[0]

    def _perform_action_on_flavor(self, underlay_actions,
                                  name, action, cfg_node):
        xxx = underlay_actions.check_call(
            'source /root/keystonercv3 && '
            'openstack flavor {} {}'.format(action, name),
            node_name=cfg_node)
        LOG.info(xxx)

    def create_flavor(self, underlay_actions, name, cfg_node):
        self._perform_action_on_flavor(underlay_actions, name,
                                       'create', cfg_node)

    def delete_flavor(self, underlay_actions, name, cfg_node):
        self._perform_action_on_flavor(underlay_actions, name,
                                       'delete', cfg_node)

    def is_flavor_restored(self, underlay_actions, name, cfg_node):
        get_by_name = underlay_actions.check_call(
            'source /root/keystonercv3 && ' +
            'openstack flavor show {}'.format(name),
            node_name=cfg_node,
            raise_on_err=False,
            expected=[ExitCodes.EX_OK, ExitCodes.EX_ERROR]
        )
        return len(get_by_name['stderr']) == 0

    @pytest.mark.grab_versions
    @pytest.mark.parametrize("_", [settings.ENV_NAME])
    @pytest.mark.run_galera_backup_restore
    def test_backup_restore_galera(self, salt_actions, drivetrain_actions,
                                   show_step, underlay_actions,
                                   handle_restore_params,  _):
        """Execute backup/restore for galera

        Scenario:
            1. Create flavor to be backuped
            2. Run job galera_backup_database
            3. Create flavor not to be restored
            4. Run job galera_verify_restore

        """
        dt = drivetrain_actions
        show_step(1)
        ctl_node = underlay_actions.get_target_node_names(target='ctl')[0]
        print(ctl_node)
        fixture_flavor1 = 'test1'
        fixture_flavor2 = 'test2'
        self.create_flavor(underlay_actions, fixture_flavor1, ctl_node)
        # ################## Run backup job #########################
        show_step(2)
        job_name = 'galera_backup_database'
        job_parameters = {
            'ASK_CONFIRMATION': False
        }
        job_result, job_description = dt.start_job_on_jenkins(
            job_name=job_name,
            job_parameters=job_parameters)

        assert job_result == 'SUCCESS', job_description

        show_step(3)
        self.create_flavor(underlay_actions, fixture_flavor2, ctl_node)
        # ######################## Run Restore ###########################
        show_step(4)
        job_name = 'galera_verify_restore'
        job_parameters = {
             'RESTORE_TYPE': 'ONLY_RESTORE',
             'ASK_CONFIRMATION': False
        }
        job_result, job_description = dt.start_job_on_jenkins(
            job_name=job_name,
            job_parameters=job_parameters)

        assert job_result == 'SUCCESS', job_description

        assert self.is_flavor_restored(underlay_actions,
                                       fixture_flavor1,
                                       ctl_node)
        assert self.is_flavor_restored(underlay_actions,
                                       fixture_flavor2,
                                       ctl_node) is False
        self.delete_flavor(underlay_actions, fixture_flavor1, ctl_node)
