/**
 *
 * Deploy the product cluster using Jenkins master on CICD cluster
 *
 * Expected parameters:

 *   ENV_NAME                      Fuel-devops environment name
 *   MCP_VERSION                   MCP version, like 2018.4 or proposed
 *   PASSED_STEPS                  Steps passed to install components using Jenkins on CICD cluster: "salt,core,cicd,openstack:3200,stacklight:2400",
                                   where 3200 and 2400 might be timeouts (not used in the testing pipeline)
 *   PARENT_NODE_NAME              Name of the jenkins slave to create the environment
 *   PARENT_WORKSPACE              Path to the workspace of the parent job to use tcp-qa repo
 *   TEMPEST_TEST_SUITE_NAME       Name of tempest suite
 *   TCP_QA_REFS                   Reference to the tcp-qa change on Gerrit, like refs/changes/46/418546/41
 */

@Library('tcp-qa')_

def common = new com.mirantis.mk.Common()
def shared = new com.mirantis.system_qa.SharedPipeline()
def stacks = shared.get_steps_list(PASSED_STEPS)
def testPlanNamePrefix = env.TEST_PLAN_NAME_PREFIX ?: "[2019.2.0-update]System"
def testPlanName = env.TEST_PLAN_NAME ?: "${testPlanNamePrefix}-${MCP_VERSION}-${new Date().format('yyyy-MM-dd')}"

if (! env.PARENT_NODE_NAME) {
    error "'PARENT_NODE_NAME' must be set from the parent deployment job!"
}

currentBuild.description = "${PARENT_NODE_NAME}:${ENV_NAME}"

def find_latest(search_pattern) {
    return "find ${PARENT_WORKSPACE} -name \"${search_pattern}\" -printf \"'%T@ %p \\n'\" | sort -r |head -n1|awk -F' ' '{ print \$2 }'"
}

timeout(time: 2, unit: 'HOURS') {
node ("${PARENT_NODE_NAME}") {
    if (! fileExists("${PARENT_WORKSPACE}")) {
        error "'PARENT_WORKSPACE' contains path to non-existing directory ${PARENT_WORKSPACE} on the node '${PARENT_NODE_NAME}'."
    }
    dir("${PARENT_WORKSPACE}") {
        if (env.REPORT_FILE_URL) {
            print "Download report from '${env.REPORT_FILE_URL}'"
            file_text = new URL ("${env.REPORT_FILE_URL}").getText()
            print "Copy downloaded file to workspace as ${REPORT_FILE_NAME}"
            writeFile file: "${REPORT_FILE_NAME}", text: file_text
        }
        def description = ''
        def exception_message = ''
        try {

            if (env.TCP_QA_REFS) {
                stage("Update working dir to patch ${TCP_QA_REFS}") {
                    shared.update_working_dir()
                }
            }

            def report_name = ''
            def testSuiteName = ''
            def methodname = ''
            def testrail_name_template = ''
            def reporter_extra_options = []

            def report_url = ''

            //  deployment_report_name = "deployment_${ENV_NAME}.xml"
            def deployment_report_name = sh(script: find_latest("deployment_${ENV_NAME}.xml"), returnStdout: true).trim()
            // tcpqa_report_name =~ "nosetests.xml"
            def tcpqa_report_name = sh(script: find_latest("nosetests.xml"), returnStdout: true).trim()
            // tempest_report_name =~ "report_*.xml"
            def tempest_report_name = sh(script: find_latest("report_*.xml"), returnStdout: true).trim()
            // stacklight_report_name =~ "stacklight_report.xml" or "report.xml"
            def stacklight_report_name = sh(script: find_latest("cvp-stacklight*results.xml"), returnStdout: true).trim()
            // cvp_sanity_report_name =~ cvp_sanity_report.xml
            def cvp_sanity_report_name = sh(script: find_latest("cvp-sanity*results.xml"), returnStdout: true).trim()
            common.printMsg(deployment_report_name ? "Found deployment report: ${deployment_report_name}" : "Deployment report not found", deployment_report_name ? "blue" : "red")
            common.printMsg(tcpqa_report_name ? "Found tcp-qa report: ${tcpqa_report_name}" : "tcp-qa report not found", tcpqa_report_name ? "blue" : "red")
            common.printMsg(tempest_report_name ? "Found tempest report: ${tempest_report_name}" : "tempest report not found", tempest_report_name ? "blue" : "red")
            common.printMsg(stacklight_report_name ? "Found stacklight-pytest report: ${stacklight_report_name}" : "stacklight-pytest report not found", stacklight_report_name ? "blue" : "red")
            common.printMsg(cvp_sanity_report_name ? "Found CVP Sanity report: ${cvp_sanity_report_name}" : "CVP Sanity report not found", cvp_sanity_report_name ? "blue" : "red")

            if (deployment_report_name) {
                stage("Deployment report") {
                    testSuiteName = "[MCP] Integration automation"
                    methodname = '{methodname}'
                    testrail_name_template = '{title}'
                    reporter_extra_options = [
                      "--testrail-add-missing-cases",
                    ]
                    ret = shared.upload_results_to_testrail(deployment_report_name, testPlanName, testSuiteName, methodname, testrail_name_template, reporter_extra_options)
                    common.printMsg(ret.stdout, "blue")
                    marked = shared.mark_test_results(testPlanName, testSuiteName)
                    common.printMsg(marked.stdout, "blue")

                    report_url = ret.stdout.split("\n").each {
                        if (it.contains("[TestRun URL]")) {
                            common.printMsg("Found report URL: " + it.trim().split().last(), "blue")
                            description += "<a href=" + it.trim().split().last() + ">${testSuiteName}</a><br>"
                        }
                    }
                    exception_message += ret.exception
                }
            }

            if (tcpqa_report_name) {
                stage("tcp-qa cases report") {
                    testSuiteName = "[MCP_X] integration cases"
                    methodname = "{classname}.{methodname}"
                    testrail_name_template = "{title}"
                    reporter_extra_options = [
                      "--testrail-add-missing-cases",
//                       "--testrail_configuration_name tcp-qa",
                    ]
                    ret = shared.upload_results_to_testrail(tcpqa_report_name, testPlanName, testSuiteName, methodname, testrail_name_template, reporter_extra_options)
                    common.printMsg(ret.stdout, "blue")
                    marked = shared.mark_test_results(testPlanName, testSuiteName)
                    common.printMsg(marked.stdout, "blue")

                    report_url = ret.stdout.split("\n").each {
                        if (it.contains("[TestRun URL]")) {
                            common.printMsg("Found report URL: " + it.trim().split().last(), "blue")
                            description += "<a href=" + it.trim().split().last() + ">${testSuiteName}</a><br>"
                        }
                    }
                    exception_message += ret.exception
                }
            }

            if ('openstack' in stacks && tempest_report_name) {
                stage("Tempest report") {
                    testSuiteName = env.TEMPEST_TEST_SUITE_NAME
                    methodname = "{classname}.{methodname}"
                    testrail_name_template = "{title}"
                    reporter_extra_options = [
                      "--send-duplicates",
                      "--testrail-add-missing-cases",
//                       "--testrail_configuration_name tcp-qa",
                    ]
                    ret = shared.upload_results_to_testrail(tempest_report_name,
                                                            testPlanName,
                                                            testSuiteName,
                                                            methodname,
                                                            testrail_name_template,
                                                            reporter_extra_options)
                    common.printMsg(ret.stdout, "blue")
                    marked = shared.mark_test_results(testPlanName, testSuiteName)
                    common.printMsg(marked.stdout, "blue")

                    report_url = ret.stdout.split("\n").each {
                        if (it.contains("[TestRun URL]")) {
                            common.printMsg("Found report URL: " + it.trim().split().last(), "blue")
                            description += "<a href=" + it.trim().split().last() + ">${testSuiteName}</a><br>"
                        }
                    }
                    exception_message += ret.exception
                }
            }

            if ('stacklight' in stacks && stacklight_report_name) {
                stage("stacklight-pytest report") {
                    testSuiteName = "LMA2.0_Automated"
                    methodname = "{methodname}"
                    testrail_name_template = "{title}"
                    reporter_extra_options = [
                      "--testrail-add-missing-cases",
                      "--testrail-case-custom-fields {\\\"custom_qa_team\\\":\\\"9\\\"}",
                    ]
                    ret = shared.upload_results_to_testrail(stacklight_report_name, testPlanName, testSuiteName, methodname, testrail_name_template, reporter_extra_options)
                    common.printMsg(ret.stdout, "blue")
                    marked = shared.mark_test_results(testPlanName, testSuiteName)
                    common.printMsg(marked.stdout, "blue")

                    report_url = ret.stdout.split("\n").each {
                        if (it.contains("[TestRun URL]")) {
                            common.printMsg("Found report URL: " + it.trim().split().last(), "blue")
                            description += "<a href=" + it.trim().split().last() + ">${testSuiteName}</a><br>"
                        }
                    }
                    exception_message += ret.exception
                }
            }

            if ('cicd' in stacks && cvp_sanity_report_name) {
                stage("CVP Sanity report") {
                    testSuiteName = "[MCP] cvp sanity"
                    methodname = '{methodname}'
                    testrail_name_template = '{title}'
                    reporter_extra_options = [
                      "--send-duplicates",
                      "--testrail-add-missing-cases",
//                       "--testrail_configuration_name tcp-qa",
                    ]
                    uploaded = shared.upload_results_to_testrail(cvp_sanity_report_name, testPlanName, testSuiteName, methodname, testrail_name_template, reporter_extra_options)
                    common.printMsg(uploaded.stdout, "blue")
                    marked = shared.mark_test_results(testPlanName, testSuiteName)
                    common.printMsg(marked.stdout, "blue")

                    report_url = uploaded.stdout.split("\n").each {
                        if (it.contains("[TestRun URL]")) {
                            common.printMsg("Found report URL: " + it.trim().split().last(), "blue")
                            description += "<a href=" + it.trim().split().last() + ">${testSuiteName}</a><br>"
                        }
                    }
                    exception_message += ret.exception
                }
            }


            // Check if there were any exceptions during reporting
            if (exception_message) {
                throw new Exception(exception_message)
            }

        } catch (e) {
            common.printMsg("Job is failed", "purple")
            throw e
        } finally {
            // reporting is failed for some reason
            writeFile(file: "description.txt", text: description, encoding: "UTF-8")
        }
    }
}
} // timeout
