#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import pytest
import os
import json

from threading import Thread
from tcp_tests import logger
from tcp_tests import settings

LOG = logger.logger


@pytest.mark.deploy
class TestDeploy250Computes(object):
    "https://mirantis.jira.com/browse/PROD-36488 Deploy 250vcpu for BM"

    @pytest.mark.grab_versions
    @pytest.mark.parametrize("_", [settings.ENV_NAME])
    @pytest.mark.run_mcp_update
    def test_deploy_250_computes(self, reclass_actions, underlay_actions,
                                 show_step, config,
                                 salt_actions, _):
        """Execute Deploy 250 Computes

        Scenario:
            1. Upload patched parts of model
            2. Generate nodes configs and apply changes to kvm nodes
            3. One by one apply network changes
            4. Check that bridges are available on all nodes
            5. One by one for each kvm create vcmp VMs
            6. Check VMs
            7. Deploy vcmp nodes
            8. Discover new computes
        """
        reclass = reclass_actions
        ssh = underlay_actions
        salt = salt_actions

        cfg_node = [node['node_name'] for node in config.underlay.ssh
                    if 'salt_master' in node.get('roles')][0]

        # ################# Upload patched parts of model  ####################
        show_step(1)
        source_path = os.path.join("{}/tcp_tests/templates/{}/".format(
            settings.LOGS_DIR, settings.ENV_NAME))
        with open(source_path + "vcmp.yml") as f:
            reclass.create_yaml_with_context(
                f.read(),
                "cluster/{}/openstack/compute/vcmp.yml".format(
                    settings.ENV_NAME))
        with open(source_path + "vcmpcompute.yml") as f:
            reclass.create_yaml_with_context(
                f.read(),
                "cluster/{}/openstack/networking/vcmpcompute.yml".format(
                    settings.ENV_NAME))
        with open(source_path + "openstack_vcompute_cluster.yml") as f:
            reclass.create_yaml_with_context(
                f.read(),
                "cluster/{}/infra/openstack_vcompute_cluster.yml".format(
                    settings.ENV_NAME))
        with open(source_path + "env_add.yml") as f:
            reclass.merge_context(f.read(), "environment/*/init.yml")
        with open(source_path + "infra_config_nodes_add.yml") as f:
            reclass.merge_context(f.read(), "cluster/*/infra/config/nodes.yml")
        with open(source_path + "infra_kvm_add.yml") as f:
            reclass.merge_context(f.read(), "cluster/*/infra/kvm.yml")
        with open(source_path + "openstack_init_add.yml") as f:
            reclass.merge_context(f.read(), "cluster/*/openstack/init.yml")
        cmp_nova_init_wa = "parameters:\n" \
                           "  _param:\n" \
                           "    nova_compute_libvirt_allowed_dn_list:\n" \
                           "      all:\n"\
                           "        value: '*CN=*cmp*'"
        reclass.merge_context(
            cmp_nova_init_wa,
            "system/nova/compute/libvirt/ssl/init.yml")

        # ###### Generate nodes configs and apply changes to kvm nodes ########
        show_step(2)

        state_output = ssh.check_call("salt-call state.apply reclass.storage "
                                      "--out=json",
                                      node_name=cfg_node,
                                      verbose=True
                                      )
        need_output = json.loads(state_output['stdout_str'])
        LOG.info(need_output)
        ret = salt.run_state("kvm*", "saltutil.sync_all")
        LOG.info(ret)
        ret = salt.run_state("kvm*", "saltutil.refresh_pillar")
        LOG.info(ret)

        # ################# One by one apply network changes ##################
        show_step(3)

        kvm_nodes = ssh.check_call(
            "salt 'cfg01*' pillar.get linux:network:host | grep '  kvm' "
            "| sed 's/   *//g' | sed 's/:/*/g'",
            node_name=cfg_node
        )['stdout_str'].splitlines()
        for node in kvm_nodes:
            ret = salt.run_state(node, "state.apply", "linux.network")
            LOG.info(ret)

        # ########## Check that bridges are available on all nodes ############
        show_step(4)

        ret = salt.run_state("kvm*", "cmd.run", "brctl show | grep 'ctl'")
        LOG.info(ret)
        ret = salt.run_state("kvm*", "cmd.run", "brctl show | grep 'mesh'")
        LOG.info(ret)

        # ########### One by one for each kvm create vcmp VMs #################
        show_step(5)

        for node in kvm_nodes:
            ret = salt.run_state(node, "state.apply", "salt.control")
            LOG.info(ret)

        # ########################### Check VMs ###############################
        show_step(6)

        ret = salt.cmd_run("kvm01*", "virsh list")
        LOG.info(ret)
        ssh.check_call(
            "salt-key",
            node_name=cfg_node,
            verbose=True
        )

        ssh.check_call(
            "salt-call mine.get '*' x509.get_pem_entries",
            node_name=cfg_node,
            verbose=True
        )

        ret = salt.run_state(cfg_node, "state.apply salt.minion")
        LOG.info(ret)

        # ######################## Deploy vcmp nodes ##########################
        show_step(7)

        deploy_thread = 8
        vcmp_num_start = 3
        vcmp_num_end = 254

        threads = list()
        for n in range(vcmp_num_start, vcmp_num_end, deploy_thread + 1):
            for i in range(n, n + deploy_thread):
                th = Thread(target=deploy_vcmp_node, args=(
                    i, salt_actions, underlay_actions, ))
                threads.append(th)
                th.start()
            for index, thread in enumerate(threads):
                thread.join()

        # ####################### Discover new computes #######################
        show_step(8)

        command = [
            {
                'description': "Run nova.controller for ctl01",
                'node_name': ssh.get_target_node_names('cfg01')[0],
                'cmd': (
                    "set -ex; "
                    "salt 'ctl01.*' state.sls nova.controller"),
                'retry': {
                    'count': 1,
                    'delay': 5}}
        ]
        salt.execute_commands(command,
                              label='New computes discover')


def deploy_vcmp_node(vcmp_num, salt_actions, underlay_actions, ):

    salt = salt_actions
    ssh = underlay_actions

    str_vcmp = str(vcmp_num)
    print("execute vcmp{}".format(
        str_vcmp))
    node_name = ssh.get_target_node_names('cfg01')[0]
    commands = [
        {
            'description': ("Sync grains and refresh pillars"
                            "for vcmp{}".format(str_vcmp)),
            'node_name': node_name,
            'cmd': (
                "set -ex; "
                "salt 'vcmp{num}.*' saltutil.sync_all; "
                "salt 'vcmp{num}.*' saltutil.sync_grains; "
                "salt 'vcmp{num}.*' saltutil.refresh_pillar".format(
                    num=str_vcmp)),
            'retry': {
                'count': 1,
                'delay': 5}},
        {
            'description': "PRE_install packages for vcmp" +
                           str_vcmp,
            'node_name': node_name,
            'cmd': (
                "set -ex; "
                "salt 'vcmp{}.*' state.sls "
                "-l quiet --state-verbose=false "
                "linux.system.repo,linux.network,rsyslog".format(
                    str_vcmp)),
            'retry': {
                'count': 3,
                'delay': 5}},
        {
            'description': "Install packages for vcmp" +
                           str_vcmp,
            'node_name': node_name,
            'cmd': (
                "set -ex; "
                "salt 'vcmp{}.*' state.sls "
                "-l quiet --state-verbose=false "
                "linux,ntp,openssh,salt,nova,ceph.common".format(
                    str_vcmp)),
            'retry': {
                'count': 3,
                'delay': 5}},
        {
            'description': "Run state.highstate for vcmp" +
                           str_vcmp,
            'node_name': node_name,
            'cmd': (
                "set -ex; "
                "salt 'vcmp{}.*' "
                "-l quiet --state-verbose=false "
                "state.highstate".format(
                    str_vcmp)),
            'retry': {
                'count': 3,
                'delay': 5}}
    ]
    salt.execute_commands(commands,
                          label='Deploy vCMP{} node'
                          .format(vcmp_num))
