/**
 *
 * Deploy the product cluster using Jenkins master on CICD cluster
 *
 * Expected parameters:

 *   ENV_NAME                      Fuel-devops environment name
 *   PASSED_STEPS                  Steps passed to install components using Jenkins on CICD cluster: "salt,core,cicd,openstack:3200,stacklight:2400",
                                   where 3200 and 2400 might be timeouts (not used in the testing pipeline)
 *   RUN_TEST_OPTS                 Pytest option -k or -m, with expression to select necessary tests. Additional pytest options are allowed.
 *   PARENT_NODE_NAME              Name of the jenkins slave to create the environment
 *   PARENT_WORKSPACE              Path to the workspace of the parent job to use tcp-qa repo
 *   TCP_QA_REFS                   Reference to the tcp-qa change on Gerrit, like refs/changes/46/418546/41
 *   SHUTDOWN_ENV_ON_TEARDOWN      optional, shutdown fuel-devops environment at the end of the job
 *   TEMPEST_IMAGE_VERSION         Tempest image version: pike by default, can be queens.
 *   TEMPEST_TARGET                Node where tempest will be run
 *   MAKE_SNAPSHOT_STAGES          optional, use "dos.py snapshot" to snapshot stages
 */

@Library('tcp-qa')_

def common = new com.mirantis.mk.Common()
def shared = new com.mirantis.system_qa.SharedPipeline()

def steps = env.PASSED_STEPS
def make_snapshot_stages = false
env.LAB_CONFIG_NAME = env.LAB_CONFIG_NAME ?: env.ENV_NAME
env.TEST_PLAN_NAME_PREFIX = '[2019.2.0-update]Upgraded'

timeout(time: 23, unit: 'HOURS') {
    node ("${PARENT_NODE_NAME}") {
            if (! fileExists("${PARENT_WORKSPACE}")) {
            error "'PARENT_WORKSPACE' contains path to non-existing directory ${PARENT_WORKSPACE} on the node '${PARENT_NODE_NAME}'."
        }
        dir("${PARENT_WORKSPACE}") {
            env.slave_workdir = pwd()
            if (env.TCP_QA_REFS) {
                stage("Update working dir to patch ${TCP_QA_REFS}") {
                    shared.update_working_dir()
                }
            }

            stage("Run tests") {
                try {
                    if (env.RUN_TEST_OPTS) {
                        shared.swarm_run_pytest(steps, "${PARENT_NODE_NAME}", make_snapshot_stages)
                    } else {
                        common.infoMsg("RUN_TEST_OPTS is empty, skipping 'swarm-run-pytest' job", "green")
                    }
                } catch (e) {
                    common.infoMsg("Tests are failed: " + e.message, "purple")
                    currentBuild.result = 'FAILURE'
                }
            } // stage("Run tests")
            if (fileExists("jenkins_agent_description.txt")) {
                def String description  = readFile("jenkins_agent_description.txt")
                currentBuild.description = "${description}"
            }
            stage("Archive all xml reports") {
                dir("${env.slave_workdir }") {
                    archiveArtifacts artifacts: "**/*.xml,**/*.log"
                    }
            }
            stage("report results to testrail from jenkins master") {
                if ("${env.REPORT_TO_TESTRAIL}" != "false") {
                    common.infoMsg("Running on: " + env.PARENT_NODE_NAME, "blue")
                    shared.swarm_testrail_report(steps, env.PARENT_NODE_NAME)
                } else {
                    common.infoMsg("REPORT_TO_TESTRAIL is set to FALSE. Skipped this step ")
                }
            } // stage("report results to testrail from jenkins master")
            stage("Store TestRail reports to job description") {
                if ("${env.REPORT_TO_TESTRAIL}" != "false") {
                    if (fileExists("description.txt")) {
                        def String description  = readFile("description.txt")
                        currentBuild.description += "${description}"
                    }
                } else {
                    common.infoMsg("REPORT_TO_TESTRAIL is set to FALSE. Skipped this step ")
                    println(currentBuild.result)
                }
            } // stage("Store TestRail reports to job description")
        } // dir
    } // node
} // timeout
