import json
import os
from datetime import datetime, timedelta, timezone

from django.conf import settings
from jenkins import Jenkins
from matplotlib import dates as mdates
from matplotlib import pyplot as plt

from .. import models

__all__ = ("update_plot",)


jenkins_client = Jenkins("https://ci.mcp.mirantis.net/")

VIEW_NAME = "MCP2.0 Openstack Periodic CI"


def build_time_obj(original_obj=None):
    def get_attr(attr):
        return getattr(original_obj, attr, 0)

    return datetime(
        year=2000,
        month=1,
        day=1,
        hour=get_attr("hour"),
        minute=get_attr("minute"),
        second=get_attr("second"),
    )


def build_start_time(timestamp):
    return (
        build_time_obj(datetime.utcfromtimestamp(timestamp / 1000))
        .replace(tzinfo=timezone.utc)
        .timestamp()
    )


def process_build(job_name, build):
    build_info = jenkins_client.get_build_info(job_name, build["number"])
    if build_info["result"] != "SUCCESS":
        return None

    return (
        build_start_time(build_info["timestamp"]),
        build_info["duration"] / 1000,
    )


def calculate_average(values):
    return sum(values) / len(values)


def process_job(job_name):
    builds = jenkins_client.get_job_info(job_name)["builds"]
    builds_info = (process_build(job_name, build) for build in builds)

    start_times, durations = zip(*filter(None, builds_info))

    avg_start_time = datetime.utcfromtimestamp(calculate_average(start_times))
    return {
        "duration": calculate_average(durations),
        "start_time": avg_start_time,
    }


def get_aggregated_build_stats():
    return {
        job_name["name"]: process_job(job_name["name"])
        for job_name in jenkins_client.get_jobs(view_name=VIEW_NAME)
    }


def get_lines(current, standard_datetime, next_day):
    start_time = current["start_time"]
    end = start_time + timedelta(seconds=current["duration"])

    if end >= next_day:
        return [
            (standard_datetime, standard_datetime + (end - next_day)),
            (start_time, next_day - timedelta(seconds=1)),
        ]

    return [(start_time, end)]


def build_data_for_jobs_time_plot(jobs):
    standard_datetime = build_time_obj()
    next_day = standard_datetime + timedelta(days=1)
    return {
        job_name: get_lines(jobs[job_name], standard_datetime, next_day)
        for job_name in jobs
    }


def draw_plot(plot_data):
    fig, ax = plt.subplots()

    for index, (job_name, data) in enumerate(plot_data.items(), 1):
        for start, end in data:
            ax.plot([start, end], [index, index], color="#68a39f", linewidth=6)

    hours = mdates.HourLocator()
    hours_fmt = mdates.DateFormatter("%H")
    ax.xaxis.set_major_locator(hours)
    ax.xaxis.set_major_formatter(hours_fmt)

    ax.format_ydata = lambda x: int(x)
    ax.format_xdata = mdates.DateFormatter("%H")

    # Set date limits
    start_time = build_time_obj()
    ax.set_xlim(
        start_time, start_time + timedelta(days=1) + timedelta(seconds=1)
    )

    # Set y axes limits
    jobs_num = len(plot_data) + 1
    ax.set_ylim(0, jobs_num)

    ax.set_yticks(range(1, jobs_num))
    ax.yaxis.tick_right()

    fig.autofmt_xdate()

    # Enable grid
    ax.grid(True, color="#f9f2f2")

    # Save figure
    path = os.path.join(settings.STATIC_ROOT, "plot.png")
    plt.savefig(path, dpi=300)


def save_job_names(plot_data_keys):
    job_names_path = os.path.join(models.fs.location, "job_names.txt")

    with open(job_names_path, "w") as f:
        json.dump(plot_data_keys, f)


def update_plot():
    try:
        jobs_dict = get_aggregated_build_stats()
        plot_data = build_data_for_jobs_time_plot(jobs_dict)
        draw_plot(plot_data)

        save_job_names(list(plot_data.keys()))

        try:
            log_record = models.ActionLog.objects.get(
                name="update_jenkins_plot"
            )
        except models.ActionLog.DoesNotExist:
            log_record = models.ActionLog(name="update_jenkins_plot")
        log_record.date = datetime.now()
        log_record.save()
    finally:
        update = models.ActionLog.objects.get(name="update_plot_started")
        update.delete()
