import glob
import logging
import os
import time
from datetime import datetime

from config import LOGGIGNG_FOLDER, LOGGIGNG_UTILS, LOGS_DIRECTORY

logging.basicConfig(
    format="[%(asctime)s][%(name)s][%(levelname)s] %(message)s",
    datefmt="%d-%m-%Y %H:%M:%S",
    handlers=[
        logging.FileHandler("{}{}".format(LOGGIGNG_FOLDER, LOGGIGNG_UTILS)),
        logging.StreamHandler(),
    ],
    level=logging.INFO,
)
logger = logging.getLogger("jenkins_api")


def get_date_from_timestamp(timestamp):
    if timestamp:
        return datetime.utcfromtimestamp(timestamp).strftime(
            "%d-%m-%Y_%H_%M_%S"
        )
    return ""


def get_last_build_id_from_db(current_database):
    """
    get_last_build_id_from_db
    :param current_database:
        self.db = TinyDB(self.patch_to_db)
        self.current_database = self.db.all()

    :return: int. Last build
    """
    logger.info("Getting last build id from DB")
    build_ids = []
    for field in current_database:
        build_ids.append(field["build"]["id"])
    # print(build_ids)
    # logger.info('Last build id is: {} {}'
    # .format(current_database, max(build_ids)))
    return max(build_ids)


def is_directory_exists(patch_to_directory):
    if patch_to_directory and os.path.exists(patch_to_directory):
        return True
    return False


def create_directory(patch_to_directory, directory_to_create=""):
    """

    :param full_patch_to_directory: string
    :param directory_to_create: string

    :return: full_patch_to_directory
    """
    full_patch_to_directory = patch_to_directory + directory_to_create

    if not is_directory_exists(full_patch_to_directory):
        logger.info("Creating directory: {}".format(full_patch_to_directory))
        os.makedirs(full_patch_to_directory)
    return full_patch_to_directory


def find_and_delete_old_files(patch_to_files, older_than):
    """
    Just finds files older than - by date

    :param patch_to_files: string
    :param older_than: int, 3 three days

    :return: list of matches files
    """
    logger.info(
        "Find all files in {} older than {} days".format(
            patch_to_files, older_than
        )
    )

    now = time.time()
    cutoff = now - (int(older_than) * 86400)

    files = [
        os.path.join(dp, f)
        for dp, dn, filenames in os.walk(patch_to_files)
        for f in filenames
    ]

    found_files = []
    for xfile in files:
        if os.path.isfile(xfile):
            t = os.stat(xfile)
            c = t.st_ctime

            if c < cutoff:
                logger.info("Deleting file: {}".format(xfile))
                os.remove(xfile)
                found_files.append(xfile)
    return files


def save_to_file(filename, data_to_write):
    """
    Saves output to a file

    :param directory:
    :param filename:
    :param data_to_write:
    :return:
    """
    logger.info("Saving to a log file: {}".format(filename))
    data_to_write = data_to_write.encode("ascii", "ignore").decode("ascii")
    with open(filename, "w") as f:
        f.write(data_to_write)

    return filename


def get_patch_to_file(job_name, build_id):
    if job_name:
        patch_to_file = LOGS_DIRECTORY + job_name + "/" + str(build_id) + "*"
        # logger.info(f'Getting patch to the file {patch_to_file}')
        return patch_to_file
    return ""


def generate_filename(
    logs_directory, build_id, build_date, job_name, build_status
):
    logger.info(
        "Saving results to file: {}. Build result: {}".format(
            job_name, build_status
        )
    )
    filename = "{0}{1}/{2}_{3}_{4}_{5}.txt".format(
        logs_directory, job_name, build_id, build_date, job_name, build_status
    )
    return filename


def check_if_file_exists(patch_to_file):
    """
    Checking if a File Exists

    :param patch_to_file: string
    :return: Bool
    """
    # logger.info('Checking if File Exists: {}'.format(patch_to_file))
    if patch_to_file:
        if glob.glob(patch_to_file):
            return True
    return False


def create_directories_for_logs(jobs_names, logs_directory):
    """
    Creates directories for logs

    :param logs_directory: string. Base directory for logs
            Inside this directory other directories will be created
    :param jobs_names: list of string
    :return: None
    """
    for job_name in jobs_names:
        logger.info(
            "Creating directories for logs: {} {}".format(
                logs_directory, job_name
            )
        )
        create_directory(logs_directory, job_name)


def save_job_console_to_file(
    logs_directory, job_name, build_id, build_date, build_status, data_to_write
):
    filename = generate_filename(
        logs_directory=logs_directory,
        job_name=job_name,
        build_id=build_id,
        build_date=build_date,
        build_status=build_status,
    )

    create_directory(
        patch_to_directory=logs_directory, directory_to_create=job_name
    )
    save_to_file(filename=filename, data_to_write=data_to_write)


def generate_patch_to_artifact_file(logs_directory, job_name):
    return "{0}{1}/".format(logs_directory, job_name)


def generate_artifact_file_patch(
    saved_artifact_file_patch,
    patch_to_artifact_file,
    pipeline_job_name,
    build_id,
    build_date,
):
    if "kubeconfig" in saved_artifact_file_patch:
        artifact_extention = "txt"
        artifact_filename = saved_artifact_file_patch.split("/")[-1]
        filename = (
            f"{str(build_id)}_{build_date}_{pipeline_job_name}"
            f"_{artifact_filename}.{artifact_extention}"
        )
        full_patch = patch_to_artifact_file + filename
    else:
        artifact_extention = saved_artifact_file_patch.split(".")[-1]
        filename = (
            f"{str(build_id)}_{build_date}_{pipeline_job_name}"
            f".{artifact_extention}"
        )
        full_patch = patch_to_artifact_file + filename
    logger.info(f"Full file patch: {full_patch}")
    return full_patch


def rename_artifact_file(old_artifact_file_patch, new_artifact_file_patch):
    os.rename(old_artifact_file_patch, new_artifact_file_patch)
    return new_artifact_file_patch
