import os
import click
import ipdb
import urllib
import xunitparser
import logging
import pathlib
import datetime
import re
import wget
import time
import uuid
import jmespath
from typing import Optional
from xml.etree import ElementTree

from reportportal_client import RPClient
from reportportal_client.helpers import timestamp

from .settings import RP_ENDPOINT, RP_APIKEY, RP_PROJECT

LOG = logging.getLogger("rp_reporter")


def obtain_file(report_path:str) -> Optional[str]:
    """
    Returns file's location in file system
    If report_path is an URL script will download it to /tmp folder
    :param report_path: str - file system or network location
    :return:
    """
    if report_path.startswith("http"):
        try:
            downloaded_file = wget.download(report_path,
                                        f"/tmp/{uuid.uuid4()}.xml")
            return downloaded_file
        except urllib.error.HTTPError:
            print("File is absent")
            return None
    if pathlib.Path(report_path).exists():
        return report_path
    else:
        raise FileNotFoundError(report_path)


def is_xml(file_path:str) -> bool:
    try:
        ElementTree.fromstring(open(file_path).read())
        return True
    except ElementTree.ParseError:
        print("File is not in XML format")
        return False

class TestRunStatistic:
    def __init__(self) -> None:
        self.passed = 0
        self.failed = 0
        self.errors = 0

    @property
    def total(self):
        return self.passed + self.failed + self.errors

    @property
    def passrate(self):
        if self.total <= 0:
            return 0
        return (self.passed / self.total) * 100

    def add_passed(self):
        self.passed += 1

    def add_failed(self):
        self.failed += 1

    def add_error(self):
        self.errors += 1

    def __repr__(self):
        if self.passrate < 100:
            passrate = f"`{self.passrate:.2f}%`"
        else:
            passrate = f"{self.passrate:.2f}%"
        return f"P{self.passed}/F{self.failed}/E{self.errors} {passrate}"

class Reporter:
    client: RPClient
    def __init__(self, client=None):
        if not client:
            self.client = RPClient(endpoint=RP_ENDPOINT,
                  project=RP_PROJECT,
                  api_key=RP_APIKEY,
                  is_skipped_an_issue=False,
                  truncate_attributes=False,
                  retries=15
                  )
        self.scheduled = list()
        self.uuid_by_subfolder = dict()
        self.attributes = dict()

    def schedule_finishing(self, item) -> None:
        self.scheduled.append(item)

    def finish_all(self) -> None:
        while self.scheduled:
            finishing_item = self.scheduled.pop()
            self.client.finish_test_item(item_id=finishing_item,
                                    end_time=timestamp()
                                    )
        self.reset_cache()

    def reset_cache(self) -> None:
        self.scheduled = list()
        self.uuid_by_subfolder = dict()

    def create_subfolders(self,
                          list_of_subfolders: list, root_folder=None) -> str:
        parent_folder = root_folder
        for number, subfolder in enumerate(list_of_subfolders):
            subfolder_fullpath = list_of_subfolders[:number]
            subfolder_fullpath.append(subfolder)
            fullpath = ".".join(subfolder_fullpath)
            if fullpath not in self.uuid_by_subfolder.keys():
                created_folder = self.client.start_test_item(
                    name=subfolder,
                    start_time=timestamp(),
                    item_type="suite",
                    parent_item_id=parent_folder
                )
                LOG.debug(f"start_test_item {subfolder=} "
                          f"item_type=suite "
                          f"name={subfolder} "
                          f"parent_item_id={parent_folder}")

                LOG.info(
                    f"Started suite: uuid={created_folder} for suite "
                    f"{subfolder=} in {fullpath=} with {parent_folder=}")
                self.schedule_finishing(created_folder)
                self.uuid_by_subfolder[fullpath] = created_folder
            folder_uuid = self.uuid_by_subfolder.get(fullpath)
            parent_folder = folder_uuid
        return parent_folder

    def report_xml(self, report_path, title, attributes=None,
                   link=None, description="", to_specific_launch=None):
        ts: xunitparser.TestSuite
        tr: xunitparser.TestResult
        tc: xunitparser.TestCase
        stats = TestRunStatistic()
        report_file = obtain_file(report_path)
        if not report_file:
            print("Error occurred with file. Interrupting reporting")
            return
        ts, tr = xunitparser.parse(open(report_file))
        if not ts:
            print(f"{ts=} is empty in {report_file=}. Interrupting reporting")
            return

        # xunitparser can't provide start_time from xml
        # so we can use only now()
        # if tr.timestamp:
        #     start_time = datetime.datetime.fromisoformat(tr.timestamp)
        # ipdb.set_trace()
        # if tr.time
        # start_time = int(os.stat(report_file).st_birthtime * 1000) # to cut float part
        attributes.update(ts.properties)

        last_subfolder = None
        test_suite = None
        launch_id = None
        if to_specific_launch:
            test_suite = self.client.start_test_item(
                name=title,
                start_time=timestamp(),
                attributes=attributes or None,
                item_type="suite",
                description=f"{description} <br>"
                            f"<a href='{link}'>Jenkins Job</a><br> "
                            f"Uploaded from <a href='{report_path}'>report</a>"

            )
            LOG.debug(
                f"start_test_item {test_suite=} item_type=suite name={title}")
            self.schedule_finishing(item=test_suite)
            root_folder = test_suite
        else:
            launch_id = self.client.start_launch(name=title,
                                            start_time=timestamp(),
                                            attributes=attributes or None,
                                            description=f"{link} \n uploaded from report {report_path} "
                                            )
            LOG.debug(f"start_launch {launch_id=} name={title} ")
            print(f"(＾-＾)＿日 report will be here {self.client.get_launch_ui_url()}")
            root_folder = None
            if not launch_id:
                # ipdb.set_trace()
                raise Exception(f"Launch {title} is not published ")

        LOG.info(f"Sending to RP")

        ts_with_progress_bar = click.progressbar(ts,
                                                 label='Sending to RP',
                                                 #  item_show_func=lambda a: a.methodname if a is not None,
                                                 length=tr.testsRun,
                                                 show_percent=True,
                                                 bar_template='[%(bar)s] %(info)s %(label)s'
                                                 )
        with ts_with_progress_bar:
            started_at = time.time()

            for tc in ts_with_progress_bar:
                if tc.classname:
                    last_subfolder = self.create_subfolders(
                        list_of_subfolders=tc.classname.split("."),
                        root_folder=root_folder
                    )
                elif "setup" in tc.methodname.lower() or "teardown" in tc.methodname.lower():
                    # setup and teardown don't have classname but have path in their name like
                    # in tempest:
                    # setUpClass (tempest.api.compute.admin.test_create_server.WindowsServers11Test)
                    found_text: list = re.findall(r"\([\w.]+\)", tc.methodname)
                    if found_text:
                        found_text: str = found_text[-1]
                    if found_text:
                        found_text: str = found_text.strip("()")
                    last_subfolder = self.create_subfolders(
                        list_of_subfolders=found_text.split("."),
                        root_folder=root_folder)
                    # name = f"{tc.classname}.{tc.methodname}"
                # else:
                name = tc.methodname
                elapsed = time.time() - started_at
                ts_with_progress_bar.label = f"{elapsed:.2f}s {name}".replace(
                    "\n", " ")

                # It's a bad way to detect setup\teardown because every testing
                # framework has his own way to log setup\teardown
                # Also test name can contain setup
                # if "setup" in tc.methodname.lower():
                #     item_type="setup"
                # elif "teardown" in tc.methodname.lower():
                #     item_type="teardown"
                # else:
                #     item_type = "STEP"
                item_type = "STEP"

                test_started_at = timestamp()
                item_id = self.client.start_test_item(
                    name=name,
                    start_time=test_started_at,
                    item_type=item_type,
                    description=f"{tc.classname}.{tc.methodname}",
                    parent_item_id=last_subfolder
                )
                LOG.debug(f"start_test_item {item_id=} "
                          f"{name=} "
                          f"{item_type=} "
                          f"parent_item_id={last_subfolder}")
                if not item_id:
                    raise Exception(f"Failed to start test {name}")
                match tc.result:
                    case "success":
                        status = "PASSED"
                        stats.add_passed()
                    case "skipped":
                        status = "SKIPPED"
                    case "failure":
                        status = "FAILED"
                        stats.add_failed()
                    case "error":
                        status = "FAILED"
                        stats.add_error()
                    case _:
                        raise BaseException(f"Unknown {tc.result=} in xml")

                if tc.message:
                    self.client.log(time=timestamp(),
                               message=tc.message,
                               item_id=item_id
                               )
                if tc.trace:
                    delimiter = "Traceback (most recent call last):"
                    for n, log in enumerate(tc.trace.split(delimiter)):
                        # All Traceback will be reporter as ERROR
                        message = delimiter+log if n else log
                        level = "ERROR" if n else "INFO"
                        self.client.log(time=timestamp(),
                                        message=message,
                                        level=level,
                                        item_id=item_id
                                        )
                        # Sleep to save chronological sorting for logs as is
                        # reportportal has timestamps in milliseconds
                        time.sleep(0.001)

                # timestamp() 1739905243451 in milliseconds
                # tc.time datetime.timedelta(microseconds=259000)
                end_time_with_duration = datetime.datetime.fromtimestamp(
                    int(test_started_at) / 1000) + tc.time
                end_time_in_milliseconds = int(
                    end_time_with_duration.timestamp() * 1000)

                self.client.finish_test_item(item_id=item_id,
                                        end_time=str(end_time_in_milliseconds),
                                        status=status,
                                        )
        self.finish_all()
        if not to_specific_launch:
            self.client.finish_launch(end_time=timestamp())
        LOG.info(self.client.get_launch_info())
        return (test_suite or launch_id), stats


if __name__ == "__main__":

    report_path = 'https://artifactory.mcp.mirantis.net/artifactory/oscore-local/jenkins-job-artifacts/oscore-si-tests-runner-k8s/40162/si_test_report.xml'
    title = "Rockoon functional"
    attributes = {
        # "openstack_version": "caracal",
        "mosk_version": "mosk-24.3.3"
    }
    link = ""
    
    reporter = Reporter()
    reporter.report_xml(
        report_path=report_path,
        title = title,
        link=link,
        # to_specific_launch="432ce97b-2727-4e4c-8303-9f1d966a184e"
        )



    # client.terminate()
