import click
import logging
from collections import deque

import yaml
import itertools
import jmespath
from typing import List, Optional

import jenkins_jinny.main as jj
from copy import deepcopy
from .report_from_xml import timestamp, Reporter

LOG = logging.getLogger("rp_reporter")

def grep(text: List, patterns: List, context_size: int) :
    """Works like `grep -rn`
    returns found text with line before and after
    :param text:
    :param patterns: list of strings, strings to find
    :param context_size: count of lines before and after to return
    :return: Yields list of lines
    """
    context = deque(maxlen=context_size)
    result = list()
    after_count = 0
    for num, line in enumerate(text, start=1):
        context.append(f"{num} {line}")
        if any(pattern.lower() in line.lower() for pattern in patterns):
            if not result:
                result = list(context)
            after_count = context_size
            continue
        if after_count > 0:
            result.append(f"{num} {line}")
            after_count -= 1
        if result and after_count == 0:
            yield result
            result = list()
    if result:
        yield result

class Description:
    def __init__(self):
        self.rockoon_version = None
        self.test_results = list()
        self.job_status = None
        self.job_number = None
        self.job_url = None

    @property
    def job_link(self):
        return f"[#{self.job_number} {self.job_status}]({self.job_url}) "

    def add_test_result(self, test_name=None,
                        testrail_url=None, rp_url=None, statistics=None):
        testrail_msg = f"[TestRailURL]({testrail_url})" if testrail_url else ""

        test_result = (f"{test_name} {testrail_msg} "
                       f"[RP_URL]({rp_url}) "
                       f"{statistics}")

        self.test_results.append(test_result)

    def __repr__(self):
        # Return each test result on new line with "* " prefix
        test_results = "\n".join(f"* {r}" for r in self.test_results)
        return (f"\n___\n"
                f"{self.job_link}\n"
                f"`{self.rockoon_version}`\n"
                f"{test_results}\n"
                f"___\n")

def get_tags(job: jj.Build):
    tags = dict()
    tags["start_time"] = job.start_time.strftime("%Y-%m-%d-%H-%M-%S")

    if context_name := job.param.OPENSTACK_CONTEXT_NAME:
        tags["openstack"] = context_name.split("/")[0]
        tags["context"] = context_name.split("/")[-1]

    if "mosk-24.3" in job.name:
        tags["mosk_version"] = "mosk-24.3"
    elif "mosk-25.1" in job.name:
        tags["mosk_version"] = "mosk-25.1"
    elif "mosk-25.2" in job.name:
        tags["mosk_version"] = "mosk-25.2"
    else:
        tags["mosk_version"] = "master"
    return tags

def get_tags_from_osdpl(osdpl_file):
    tags = dict()
    osdpl_content = yaml.safe_load(open(osdpl_file[0], "rb"))
    osdpl_dict = yaml.safe_load(osdpl_content)
    found_osdpl = jmespath.search("items[*].status.version", osdpl_dict)
    if not found_osdpl:
        LOG.error(f"Can't find osdpl info in {osdpl_file}")
        return
    tags["rockoon_version"] = found_osdpl[0]

    if ovn:=jmespath.search("items[*].spec.features.neutron.backend", osdpl_dict):
        tags["neutron.backend"] = ovn[0]
    else:
        tags["neutron.backend"] = "ovs"

    if nova_images:=jmespath.search("items[*].spec.features.nova.images.backend", osdpl_dict):
        tags["nova.images.backend"] = nova_images[0]

    return tags


def report_job_status(job: jj.Build, job_suite_id: str, reporter: Reporter):
    rp_client = reporter.client
    if not job_suite_id:
        return
    subjob_item_id = rp_client.start_test_item(
        name=job.name,
        start_time=timestamp(),
        item_type="STEP",
        description=f"child {job.url} {job.duration} {job.status}",
        parent_item_id=job_suite_id
    )
    match job.status:
        case "SUCCESS":
            status = "PASSED"
        case "FAILURE":
            status = "FAILED"
        case "ABORTED":
            status = "FAILED"
        case "UNSTABLE":
            status = "PASSED"
        case _:
            LOG.error(f"Unknown translation {job.status=} to status")
            status = "FAILED"
    # add logs to test

    catch_logs = [
        "failure",
        "failed",
        "error",
        "exception",
        "assert",
        "discovered openstack controller version is"
    ]

    all_logs = job.get_logs()
    for log in grep(text=all_logs, patterns=catch_logs, context_size=8):
        # LOG.error("Attach logs {}".format("\n".join(log)))
        rp_client.log(time=timestamp(),
                      message="\n".join(log),
                      item_id=subjob_item_id
                      )
    # ipdb.set_trace()
    rp_client.finish_test_item(item_id=subjob_item_id,
                               status=status,
                               end_time=timestamp(),
                               )

def upload_job(job: str|jj.Build,
               suite_per_job: bool=False,
               tags: Optional[dict] =None):
    if isinstance(job, str):
        job = jj.Build(job)
    if not tags:
        tags = dict()

    print(f"│˶˙ᵕ˙˶)꜆ I take {job}")

    tags.update(get_tags(job))

    reporter = Reporter()
    description = Description()
    rp_client = reporter.client
    launch_id = None
    job_suite_id = None
    artifactory_url = None

    description.job_status = job.status
    description.job_number = job.number
    description.job_url = job.url

    if suite_per_job:
        tags["jenkins_job"] = job.number
        launch_id = rp_client.start_launch(
            name=job.name,
            start_time=timestamp(),
            attributes=tags,
            description=f"Deploy job {job.url}"
        )
        print(f"(＾-＾)＿日 report will be here {rp_client.get_launch_ui_url()}")
        job_suite_id = rp_client.start_test_item(
            name="CI jobs",
            start_time=timestamp(),
            item_type="suite",
        )
        reporter.schedule_finishing(job_suite_id)
    rp_client.log(time=timestamp(),
                  message=f"Job status: {job.status}",
                  # item_id=launch_id
                  )

    for child in itertools.chain([job], job.heirs):
        child: jj.Build
        rp_client.log(time=timestamp(),
                      message=f"{child} {child.status} {child.url}",
                      # item_id=launch_id
                      )
        report_job_status(job=child, job_suite_id=job_suite_id,
                          reporter=reporter)

        # test_tags = deepcopy(tags)
        test_tags = dict()
        print(f"⫍‍⌕‍⫎ tests in {child}")
        test_results_files = None
        match child.name:
            case "deploy-openstack-k8s-env":
                osdpl_file = child.get_artifacts("deployed.yaml")
                if not osdpl_file:
                    LOG.error(f"Can't find osdpl file in {job}")
                    continue
                tags.update(get_tags_from_osdpl(osdpl_file))
                rp_client.update_test_item(
                    attributes=tags,
                    item_uuid=launch_id
                )
                description.rockoon_version = tags.get("rockoon_version")
            case "tempest-runner-k8s":
                title = "Tempest"

                test_results_files = [file_url
                               for file_url in child.get_link_from_description()
                               if "tempest_report.xml" in file_url]
            case "stepler-runner-k8s":
                title = "Stepler"
                # report_path = [file_url
                #                for file_url in child.get_link_from_description()
                #                if "stepler_test_results.xml" in file_url ][0]
                test_results_files = child.get_artifacts("stepler_test_results.xml")
                if not test_results_files:
                    LOG.error(f"Can't found 'stepler_test_results.xml' in "
                              f"{child.url} artifacts")
            case "oscore-si-tests-runner-k8s":
                title = "SI tests"
                test_tags["test"] = child.param.RUN_TESTS
                test_results_files = [file_url
                               for file_url in child.get_link_from_description()
                               if "si_test_report.xml" in file_url ]
                if not test_results_files:
                    LOG.error(f"Can't found 'si_test_report.xml' in {child.url}")
            case "oscore-functional-tests-runner":
                title = "Rockoon Functional"
                test_results_files = [file_url
                               for file_url in child.get_link_from_description() 
                               if "si_test_report.xml" in file_url]
                if not test_results_files:
                    LOG.error(f"Can't found 'si_test_report.xml' in {child.url}")
            case "si-test-check-downtime-statistic":
                title = "Downtime tests"
                test_results_files = [
                    f"{file_url}/artifacts/test_check_downtime_statistic_result.xml"
                    for file_url in child.get_link_from_description()
                ]
                if not test_results_files:
                    LOG.error(f"Can't found 'test_check_downtime_statistic_result.xml' in {child.url}")
            case "collect-openstack-kaas-artifacts":
                artifactory_url = child.description.split("url: ")[-1]
                rp_client.log(time=timestamp(),
                              message=f"Pod Logs {artifactory_url}/pod-logs.tar.gz")
        if not test_results_files:
            # We can iterate by child jobs which don't contain any reports
            continue
        testrail_url = [url
                        for url in child.get_link_from_description()
                        if "testrail.com" in url
                        ]
        rp_client.log(time=timestamp(),
                      message=f"Found file to upload: {test_results_files}",
                      )
        report_path = test_results_files[0]
        LOG.info("=== report_xml {kwargs}".format(
            kwargs = dict(report_path=report_path,
                          title = title,
                          attributes=test_tags,
                          link=job.url,
                          to_specific_launch=launch_id))
        )

        print(f"(っ･-･)っ Uploading {report_path}")
        reported_suite_id, stats = reporter.report_xml(
            report_path=report_path,
            title = title,
            attributes=test_tags,
            link=job.url,
            to_specific_launch=launch_id,
        )
        description.add_test_result(
            test_name=f"{title} {test_tags.get('test', '')}",
            testrail_url=testrail_url[0] if testrail_url else None,
            rp_url=rp_client.get_launch_ui_url(),
            statistics=stats
        )
        rp_client.log(time=timestamp(),
                      message=f"Reported with stats: {stats}",
                      )

    rp_client.log(time=timestamp(),
                  message="Reporting completed",
                  )
    if suite_per_job:
        report_url = rp_client.get_launch_ui_url()
        rp_client.finish_launch(end_time=timestamp(),
                                attributes=tags,
                                description=str(description) +
                                            f"\nPod Logs {artifactory_url}/pod-logs.tar.gz")
        print(f"report is here {report_url}")
        print("Pushing new description to job...")
        try:
            job.description = (job.description +
                               f"<br><br> "
                               f"<a href='{report_url}'>Link to ReportPortal</a> <br>"
                               )
        except Exception as e:
            print(f"Can't push description to {job=}: {e}")
    print(f" ʕノ•ᴥ•ʔノ Completed")


def upload_view(view, pattern):
    for job in jj.jobs_in_view(view, ""):
        if not job.number:
            continue
        if pattern and not pattern in job.name:
            continue
        upload_job(job, suite_per_job=True)

@click.group()
def cli():
    pass

@cli.command()
@click.argument("job_url")
def report_job(job_url):
    upload_job(job_url, suite_per_job=True)

@cli.command()
@click.option("--pattern", default=None, help="Upload only job with pattern")
@click.argument('view_url')
def report_view(view_url, pattern):
    """
    :param view_url: Url to the view
    """
    upload_view(view_url, pattern)

@cli.command()
@click.argument('report_path')
def report_xmlfile(report_path):
    """
    :param report_path: Url or file location of xunit report
    """
    title = report_path.split("/")[-1]
    Reporter().report_xml(
        report_path=report_path,
        title=title,
    )


if __name__ == '__main__':
    cli()
