import datetime
import difflib
import json
import logging
from datetime import datetime as dt
from datetime import timedelta
from itertools import islice
from typing import Dict, Iterator, List, Optional, TextIO, Tuple

from .. import models
from ..jira_manager import JiraIssue
from . import filters, test_rail_api
from .enums import StatusEnum
from ..utils import DBlogger

__all__ = ("process_test_run",)


def finish_report(report: models.TestRailReport) -> None:
    """
    Adds mark to the TestReport object that report is finished
    and saves the object to the DB
    :param report: TestRail Report model
    """
    report.finished = True
    report.save()


def apply_filters(
    data: str,
    filter_last_traceback: bool,
    ip_filter: bool,
    uuid_filter: bool,
    filter_func: str,
) -> str:
    """
    Applies various text modifiers (filtering, masking, etc.) to the input
    text.

    :param data: The input text to be modified
    :param filter_last_traceback: A boolean indicating whether to apply the
    last traceback filter
    :param ip_filter: A boolean indicating whether to apply the IP filter
    :param uuid_filter: A boolean indicating whether to apply the UUID filter
    :param filter_func: A Python function as a string that can be executed
    to apply custom text filtering

    :return: The modified text after applying the specified filters and
    functions.
    """

    if not data:
        data = ""

    if filter_last_traceback:
        data = filters.last_traceback_filter(data)

    if ip_filter:
        data = filters.filter_ip(data)

    if uuid_filter:
        data = filters.filter_uuid(data)

    if filter_func:
        exec(filter_func)
        data = locals()["custom_filter"](data)
    return data


def get_runs_by_pattern(
    runs_in_plan: List[dict], test_pattern: str, suite_id: int
) -> List[int]:
    """
    Returns a list of run IDs that are related to a specific Test Suite
    and have names containing a pattern (test_pattern)


    :param runs_in_plan: A list of runs
    :param test_pattern:  A string pattern to match against Test Runs' names
    :param suite_id: The ID of the Test Suite to which the tests should be
    related

    :return: a list of IDs
    """
    run = []
    for t_run in runs_in_plan:
        if test_pattern in t_run["name"] and t_run["suite_id"] == suite_id:
            run.append(t_run["runs"][0]["id"])
    return run


def find_fail_with_same_comment(
    case_id: int,
    last_comment: str,
    plan_name: str,
    testrun_pattern: str,
    created_by_id: int,
    created_after: int,
    created_before: int,
    text_filters: dict,
) -> Iterator[Tuple[Optional[dict], float, int]]:
    """
    Searches for similar failures within a test plan based on specific
    criteria.

    :param case_id: The ID of the test case for which the failure is
     being searched
    :param last_comment: The last comment associated with the failed test
    :param plan_name: The name of the test plan to search within
    :param testrun_pattern: A pattern for filtering test runs
    :param created_by_id: The ID of the user who created the test plan
    :param created_after: The date (created_after) after which the test
    plan was created
    :param created_before: The date (created_before) before which the test
    plan was created
    :param run_name: The name of the test run
    :param text_filters: A dictionary of text filters to apply when
    comparing comments

    :return: An iterator that yields tuples containing information
    about matching test results, including test result data, similarity
    ratio, and the associated run ID.
    """
    end_lookup_date = dt.strptime(
        f"{created_before} 23:59:59", "%Y-%m-%d %H:%M:%S"
    )
    start_lookup_date = dt.strptime(
        f"{created_after} 00:00:00", "%Y-%m-%d %H:%M:%S"
    )
    filters = {
        "created_by": created_by_id,
        "created_before": int(dt.timestamp(end_lookup_date)),
        "created_after": int(dt.timestamp(start_lookup_date)),
        "plan_name": plan_name,
        "status_id": [
            StatusEnum.test_failed,
            StatusEnum.failed,
            StatusEnum.blocked,
            StatusEnum.product_failed,
            StatusEnum.wont_fix,
            StatusEnum.retest,
        ],
        "testrun_pattern": testrun_pattern,
    }

    for n, results in enumerate(
        test_rail_api.get_result_history_for_case(case_id, **filters)
    ):
        if n >= 500 or not results:
            yield None, None, None
            return

        comment = apply_filters(results[-1]["comment"], **text_filters)
        ratio = difflib.SequenceMatcher(
            lambda symbol: symbol in [" ", ",", "\n"],
            last_comment,
            comment,
            autojunk=False,
        ).ratio()

        if ratio > 0.7:
            run_id = test_rail_api.api.tests.get_test(results[0]["test_id"])[
                "run_id"
            ]
            yield results[0], ratio, run_id


def get_project_id(
    f: TextIO, test_run: models.TestRailTestRun, report: models.TestRailReport
) -> Optional[int]:
    """
    Returns the TestRail Project ID associated with a specific test run

    :param f: A file-like object for writing log messages
    :param test_run: An instance of the TestRailTestRun model representing
    the test run
    :param report: An instance of the TestRailReport model for reporting
    purposes

    :return: The TestRail Project ID if found; otherwise, returns None
    """
    project_id = test_rail_api.get_project_id(test_run.project_name)
    if not project_id:
        f.write(
            "Incorrect Project {}. Stopping processing\n".format(
                test_run.project_name
            )
        )
        f.flush()
        finish_report(report)
        return None
    return project_id


def get_plans(
    test_run: models.TestRailTestRun, run_date: dt, project_id: int
) -> List[dict]:
    """
    Get plans which will be processed

    :param test_run: TestRun django object
    :param run_date: retrieve plans created before that date
    :param project_id: project ID
    """
    created_by_id = test_run.created_by_id
    kw = {"limit": 100, "created_before": int(run_date)}
    if created_by_id:
        kw["created_by"] = created_by_id
    return test_rail_api.get_plans(project_id, **kw)


def get_last_comment(case_id: int, run_id: int, text_filters: dict) -> str:
    """
    Retrieve the last comment associated with a test case in a TestRail
    test run.


    :param case_id:  ID of the test case.
    :param run_id: ID of the test run for that test case
    :param text_filters: dictionary with switchers for text filters

    :return: A string containing the filtered last comment for the specified
    test case in the given test run
    """
    last_result = test_rail_api.get_result_for_case(run_id, case_id)

    return apply_filters(last_result[0]["comment"], **text_filters)


def process_old_test(
    logger: logging.Logger,
    case_id: int,
    last_comment: str,
    run_id: int,
    test: dict,
    testrail_filters: dict,
    text_filters: dict,
) -> bool:
    """
    Writes to report file similarity info about the TestCase under the test

    :return: Returns False if no similarities found
    """
    found_unknown_fail = 0
    for sim_result, ratio, old_run_id in find_fail_with_same_comment(
        case_id, last_comment, text_filters=text_filters, **testrail_filters
    ):
        if str(run_id) == str(old_run_id):
            continue
        per = round(100.0 * ratio, 2)
        run_link = test_rail_api.html_link("run", old_run_id, old_run_id)
        if type(sim_result) is not dict:
            logger.error(
                f"Similarity not found due to similarity: {per}, "
                f"in run {run_link}\n"
            )
            return False

        prod_link = (
            "None"
            if str(sim_result["defects"]) == "None"
            else JiraIssue(sim_result["defects"]).html()
        )
        test_link = test_rail_api.html_link(
            "test", sim_result["test_id"], str(sim_result["test_id"])
        )
        status_id = int(sim_result["status_id"])
        if status_id in [
            StatusEnum.retest,
            StatusEnum.failed,
            StatusEnum.blocked,
        ]:
            logger.info(
                f"Found a similar result on the test "
                f"{test_link} with similarity {per}% and "
                f"{StatusEnum(status_id).name} status and {prod_link} "
                f"defect. <i>Continuing...</i>"
            )
            found_unknown_fail += 1
            if found_unknown_fail >= 10:
                logger.error("Detected 10+ consecutive unknown failures")
                return False
            continue
        elif ratio > 0.9:
            testrail_link = (
                f"[{sim_result['test_id']}]"
                f"(https://mirantis.testrail.com"
                f"/index.php?/tests/view/"
                f"{sim_result['test_id']})"
            )
            comment = (
                f"Marked by TestRailBot because of similarity "
                f"with test {testrail_link} {per}%"
            )
            # Copy the original comment if it was not created by this bot
            if (
                str(sim_result["status_id"]) == StatusEnum.wont_fix
                and sim_result["comment"]
                and "Marked by TestRailBot" not in sim_result["comment"]
            ):
                comment = sim_result["comment"]

            update_dict = {
                "status_id": sim_result["status_id"],
                "comment": comment,
                "defects": sim_result["defects"],
            }
            logger.info(
                f"Found a similar result on the test "
                f"{test_link} with similarity {per}% and "
                f"{StatusEnum(status_id).name} status and {prod_link} "
                f"defect\n"
                f"<i style='color:ForestGreen;'>Pushing to TestRail "
                f"{update_dict}"
                f"</i>"
            )
            test_rail_api.add_result(test["id"], update_dict)
            if "Completed" in prod_link:
                return False
            return True
        elif ratio > 0.7:
            logger.error(
                f"Found a similar result on the test "
                f"{test_link} with similarity {per}% and "
                f"{StatusEnum(status_id).name} status and {prod_link} "
                f"defect, but NOT marked by "
                f"TestRailBot because of similarity only, "
                f"you can update manually"
            )
            return False


def process_test(
    test: dict, testrail_filters: dict, text_filters: dict
) -> None:
    """
    Starts processing for the TestResult
    """
    test_result_report, _ = models.TestResult.objects.get_or_create(
        result_id=test["id"]
    )
    LOG = DBlogger(name=str(test["id"]), storage=test_result_report)

    case_id = test["case_id"]
    run_id = test["run_id"]
    run_name = test_rail_api.get_run_name(run_id)
    test_link = test_rail_api.html_link("test", test["id"], test["title"])
    run_link = test_rail_api.html_link("run", run_id, run_name)

    LOG.info(f"<b>Proceeding test {test_link} <br> in {run_link} run</b>")

    last_comment = get_last_comment(case_id, run_id, text_filters)

    found = process_old_test(
        logger=LOG,
        case_id=case_id,
        last_comment=last_comment,
        run_id=run_id,
        test=test,
        testrail_filters=testrail_filters,
        text_filters=text_filters,
    )
    if found:
        test_result_report.action_needed = not found
    else:
        LOG.error(
            f"<b>Automatic test processing failed. "
            f"Please process test manually {test_link}</b>"
        )
        test_result_report.action_needed = True
    test_result_report.save()


def process_test_run(
    bot_run_id: int, report_id: int, path: str, is_testplan: bool
) -> None:
    """
    This function processes a created bot test run. It retrieves a list
    of test plans to process, gathers the failed tests from the test run,
    and passes them for processing using the 'process_test' function.
    All the failed tests are processed

    :param bot_run_id: number of result reports from tab 'Reports'
    :param report_id: number of run from tab 'Test Run'
    :param path: path to report results
    :param is_testplan: flag to show that TestPlan will be proceeded instead
    of TestRun
    """
    report: models.TestRailReport = models.TestRailReport.objects.get(
        pk=report_id
    )
    bot_test_run: models.TestRailTestRun = models.TestRailTestRun.objects.get(
        pk=bot_run_id
    )
    with open(path, "a") as f:
        if is_testplan:
            test_run = test_rail_api.get_plan_by_id(bot_test_run.run_id)
            run_type = "plan"
        else:
            test_run = test_rail_api.get_run_by_id(bot_test_run.run_id)
            run_type = "run"
        link = test_rail_api.html_link(
            run_type, test_run["id"], test_run["name"]
        )
        f.write(f"Start processing {run_type} {link}\n")
        f.flush()

        project_id = get_project_id(f, bot_test_run, report)
        if not project_id:
            return

        # failed_tests: all failed tests in test run/plan
        failed_tests = test_rail_api.get_failed_tests(
            bot_test_run.run_id, by_plans=is_testplan
        )
        for test in failed_tests:
            if (
                bot_test_run.caching_tests_enabled
                and test["id"] in bot_test_run.checked_tests
            ):
                continue
            report.test_results.append(test["id"])
            report.save()
            process_test(
                test,
                bot_test_run.testrail_filters,
                bot_test_run.text_filters,
            )
            bot_test_run.checked_tests.append(test["id"])
            bot_test_run.save()
        f.write("Test processing finished")
        f.flush()
        finish_report(report)


def get_cases(
    project_id: int,
    suite_id: int,
    limit: int = 250,
    max_limit: int = 1000,
    filter: str = None,
    created_after: int = None,
    created_before: int = None,
) -> Iterator[Dict]:
    for offset in range(0, max_limit, limit):
        cases = test_rail_api.api.cases.get_cases(
            project_id=project_id,
            suite_id=suite_id,
            limit=limit,
            offset=offset,
            created_after=created_after,
            created_before=created_before,
            filter=filter,
        ).get("cases")

        if not cases:
            return
        for case in cases:
            yield case


def get_test_passrate_in_suite(diff_id: int, report_id: int) -> Dict:
    """
    Returns a percentage of passed tests for each test in the suite

    :param diff_id: ID of models.DiffOfSuitesPassRates object in the DB to
    retrieve settings for comparing test suites
    :param report_id: ID of models.SuitePassRate object which contains a
    results of fetching passrates for specific test suite

    :return: dict with
    {"test_title": {
        "rate": 40,
        "case_id": 12345
        },
    "test_title2": {
        "rate": 80,
        "case_id": 12346
        }
    }
    """

    diff_obj: models.DiffOfSuitesPassRates = (
        models.DiffOfSuitesPassRates.objects.get(pk=diff_id)
    )
    report: models.SuitePassRate = models.SuitePassRate.objects.get(
        pk=report_id
    )
    suite_id = report.suite_id

    project_id = test_rail_api.get_project_id("Mirantis Cloud Platform")
    report.suite_name = test_rail_api.get_suite_by_id(suite_id)["name"]

    end_lookup_date = datetime.datetime.now()
    start_lookup_date = end_lookup_date + timedelta(days=-15)
    _filters = {
        "created_by": 109,
        "plan_name": "[MCP2.0]OSCORE",
        "created_before": int(dt.timestamp(end_lookup_date)),
        "created_after": int(dt.timestamp(start_lookup_date)),
    }

    passrate_by_cases = dict()
    params = dict(
        project_id=project_id,
        suite_id=suite_id,
        filter=diff_obj.test_keyword,
        limit=200,
    )
    for _n, case in enumerate(get_cases(**params), start=1):
        case_title = case["title"]
        case_id = case["id"]
        report.status = f"Current case: {_n}"
        report.save()
        # Limit generator to the list with  the length defined in the
        # DiffOfSuitesPassRates
        last_case_results = list(
            islice(
                test_rail_api.get_result_history_for_case(case_id, **_filters),
                diff_obj.limit,
            )
        )

        passrate_by_cases[case_title] = dict()
        passrate_by_cases[case_title]["case_id"] = case_id
        if last_case_results:
            passed_tests = [
                x
                for x in last_case_results
                if x[-1]["status_id"] == StatusEnum.passed
            ]
            passrate_by_cases[case_title]["rate"] = (
                len(passed_tests) * 100 / diff_obj.limit
            )
        else:
            passrate_by_cases[case_title]["rate"] = "No result found"
        report.passrate_by_tests = json.dumps(passrate_by_cases, indent=4)
        report.status = f"Current case: {_n}"
        report.save()
    report.finished = True
    report.save()
    return passrate_by_cases
