import logging
from datetime import datetime

import config
import gspread
from oauth2client.service_account import ServiceAccountCredentials

logging.basicConfig(
    format="[%(asctime)s][%(name)s][%(levelname)s] %(message)s",
    datefmt="%d-%m-%Y %H:%M:%S",
    handlers=[
        logging.FileHandler(
            "{}{}".format(config.LOG_FOLDER, config.LOG_FILENAME)
        ),
        logging.StreamHandler(),
    ],
    level=logging.INFO,
)
logger = logging.getLogger("upd_gs")


class UpdateGoogleSheets:
    """
    Working with Google sheets
    """

    def __init__(self):
        scope = [
            "https://spreadsheets.google.com/feeds",
            "https://www.googleapis.com/auth/drive",
        ]
        credentials = ServiceAccountCredentials.from_json_keyfile_name(
            config.GOOGLE_AUTH, scope
        )
        gc = gspread.authorize(credentials)
        self.wks = gc.open("OSCORE QA daily reports").worksheet(
            config.GOOGLE_SHEET_NAME
        )
        logger.info("Opening OSCORE QA daily reports for May 2019")

    def update_gs_jobs(self, gs_jobs, jobs_for_update, column_to_update):
        """
        Updates google sheet column with jobs
        """
        logger.info("Updating google sheet jobs")

        for gs in gs_jobs:
            for all_jobs in jobs_for_update["multi_results"]:
                for upd in jobs_for_update["multi_results"][all_jobs]:
                    if (
                        "oscore-oscc-ci" in upd["full_job_name"]
                        and gs["gs_job_name"] == upd["full_job_name"]
                        and upd["build_status"] != "None"
                    ):
                        cell = f"{column_to_update}{gs['row_number']}"

                        if upd["build_status"] == "FAILURE":
                            upd["build_status"] = "FAILED"

                        logger.info(
                            f"{datetime.fromtimestamp(upd['timestamp'])}, "
                            f"{upd['full_job_name']}, {upd['build_status']}"
                        )

                        if not self.wks.acell(cell).value:
                            self.wks.update_acell(cell, upd["build_status"])
                            # FIXME: need to move this long
                            #  strings into variable
                            if (
                                upd["full_job_name"]
                                == "oscore-oscc-ci_openstack-ovs-core-"
                                "octavia-pike"
                                or upd["full_job_name"]
                                == "oscore-oscc-ci_openstack-ovs-"
                                "core-octavia-queens"
                            ):
                                network_team_cell = get_network_team_cell(
                                    column_to_update, gs["row_number"]
                                )
                                self.wks.update_acell(
                                    network_team_cell,
                                    "Area of networking team",
                                )

                    if (
                        gs["gs_job_name"] == upd["full_job_name"]
                        and upd["build_status"] != "None"
                        and check_datetime_today(upd["timestamp"])
                    ):
                        cell = f"{column_to_update}{gs['row_number']}"

                        if upd["build_status"] == "FAILURE":
                            upd["build_status"] = "FAILED"

                        logger.info(
                            f"{datetime.fromtimestamp(upd['timestamp'])}, "
                            f"{upd['full_job_name']}, {upd['build_status']}"
                        )

                        if not self.wks.acell(cell).value:
                            self.wks.update_acell(cell, upd["build_status"])
                            # FIXME: need to move this long
                            #  strings into variable
                            if (
                                upd["full_job_name"]
                                == "oscore-oscc-ci_openstack-ovs-core"
                                "-octavia-pike"
                                or upd["full_job_name"]
                                == "oscore-oscc-ci_openstack-ovs-core"
                                "-octavia-queens"
                            ):
                                network_team_cell = get_network_team_cell(
                                    column_to_update, gs["row_number"]
                                )
                                self.wks.update_acell(
                                    network_team_cell,
                                    "Area of networking team",
                                )

                    elif (
                        ("oscore-oscc-ci" not in upd["full_job_name"])
                        and gs["gs_job_name"] == upd["full_job_name"]
                        and upd["build_status"] != "None"
                        and not check_datetime_today(upd["timestamp"])
                        and gs["row_number"]
                        and column_to_update
                    ):
                        cell = f"{column_to_update}{gs['row_number']}"

                        logger.info(
                            f"NOT TODAY: "
                            f"{datetime.fromtimestamp(upd['timestamp'])},"
                            f"{upd['full_job_name']}, {upd['build_status']}"
                        )

                        if not self.wks.acell(cell).value:
                            self.wks.update_acell(cell, "NOT EXEСUTED")
                            # FIXME: need to move this long
                            #  strings into variable
                            if (
                                upd["full_job_name"]
                                == "oscore-oscc-ci_openstack-ovs-core"
                                "-octavia-pike"
                                or upd["full_job_name"]
                                == "oscore-oscc-ci_openstack-ovs-core"
                                "-octavia-queens"
                            ):
                                network_team_cell = get_network_team_cell(
                                    column_to_update, gs["row_number"]
                                )
                                self.wks.update_acell(
                                    network_team_cell,
                                    "Area of networking team",
                                )

        for gs in gs_jobs:
            for single_job in jobs_for_update["single_results"]:
                if (
                    jobs_for_update["single_results"][single_job]["job_name"]
                    == gs["gs_job_name"]
                    and jobs_for_update["single_results"][single_job][
                        "build_status"
                    ]
                    != "No Results"
                    and jobs_for_update["single_results"][single_job][
                        "build_status"
                    ]
                    != "None"
                ):
                    logger.info(
                        f"Single jobs, column to update {column_to_update}"
                    )

                    cell = f"{column_to_update}{gs['row_number']}"
                    if (
                        jobs_for_update["single_results"][single_job][
                            "build_status"
                        ]
                        == "FAILURE"
                    ):
                        jobs_for_update["single_results"][single_job][
                            "build_status"
                        ] = "FAILED"

                    if not self.wks.acell(cell).value:
                        self.wks.update_acell(
                            cell,
                            jobs_for_update["single_results"][single_job][
                                "build_status"
                            ],
                        )

    def get_all_gs_jobs(self, column_number="A"):
        """
        Gets all the google sheet jobs for updating from the first column
        """
        logger.info(
            "Getting all the google sheet jobs for updating "
            "from the first column"
        )
        all_jobs = []

        # If delete or remove a job from google sheet, update this:
        for i in range(3, 40):
            all_jobs.append(
                {
                    "gs_job_name": self.wks.acell(
                        column_number + str(i)
                    ).value,
                    "row_number": i,
                    "column_number": column_number,
                }
            )
        logger.info(f"All google jobs: {all_jobs}")
        return all_jobs

    def get_today_date_column(self):
        """
        Gets today column
        Compares value from 3-d row and list_of_columns in column with
        today date.
        If they are the same, returns column value from the list_of_columns
        """
        logger.info("Getting date from gs")

        for i in config.LIST_OF_COLUMNS:
            cell = i + "2"
            now = datetime.now()
            if self.wks.acell(cell).value == now.strftime("%Y-%m-%d"):
                logger.info(f"{cell}, {i}, {self.wks.acell(cell).value}")
                return i


def get_network_team_cell(column_to_update, row_number):
    column_to_update_index = config.LIST_OF_COLUMNS.index(column_to_update) + 1
    cell = config.LIST_OF_COLUMNS[column_to_update_index] + str(row_number)
    return cell


def update_multy_job_names(results):
    """
    Fron Jenkins we get job names and job URLS.
    Combinating this we get the job names.
    Returns updated Jenkins results
    """
    logger.info("Updating multy jobs names")
    for key, value in results["multi_results"].items():
        for i in value:

            if "virtual-mcp11-aio" in i["job_name"]:
                version = i["baseurl"][72:]
                version = version[: version.find("/")]
                new_name = key + "_" + i["job_name"] + version
                i["full_job_name"] = new_name
            else:
                new_name = key + "_" + i["job_name"]
                i["full_job_name"] = new_name

    return results


def check_datetime_today(timestamp):
    """
    If timestamp is appied for today - returns True
    """
    to_date_obj = datetime.fromtimestamp(timestamp)

    today = datetime.today().date()

    # Returns True or False
    return to_date_obj.date() == today


def update_google_sheet(all_jobs_results):
    """
    The main function for updating google sheets.
    All the functions for updating are here
    """
    all_jobs = update_multy_job_names(all_jobs_results)

    gs = UpdateGoogleSheets()
    column = gs.get_today_date_column()

    all_gs_jobs = gs.get_all_gs_jobs()
    gs.update_gs_jobs(all_gs_jobs, all_jobs, column)
