"""
-------------
Generate report
-------------
"""

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import datetime
import logging
import re

import config
import jinja2
import update_google_sheets
from get_artifacts_links_single_jobs import (
    update_all_jobs_results_with_artifacts,
)
from jenkinsapi import custom_exceptions
from jenkinsapi.jenkins import Jenkins
from jenkinsapi.utils.crumb_requester import CrumbRequester
from jinja2 import Template

logging.basicConfig(
    format="[%(asctime)s][%(name)s][%(levelname)s] %(message)s",
    datefmt="%d-%m-%Y %H:%M:%S",
    handlers=[
        logging.FileHandler(
            "{}{}".format(config.LOG_FOLDER, config.LOG_FILENAME)
        ),
        logging.StreamHandler(),
    ],
    level=logging.INFO,
)
logger = logging.getLogger(config.LOGGER)


class GetJobsResults:
    def __init__(self):
        self.server = Jenkins(
            config.JENKINS_URL,
            username=config.USERNAME,
            password=config.PASSWORD,
            requester=CrumbRequester(
                username=config.USERNAME,
                password=config.PASSWORD,
                baseurl=config.JENKINS_URL,
            ),
        )

    def get_console_output(self, job_name):
        logger.info(f"Getting console output from: {job_name}")

        job = self.server.get_job(job_name)

        last_build = job.get_last_build()
        console = last_build.get_console()
        return console.lower()

    def get_run_jobs_from_console_output(self, job_name):
        logger.info(f"Getting run jobs from console output: {job_name}")

        job_console = self.get_console_output(job_name)
        console_list = job_console.split("\n")

        output = []
        for i in console_list:
            if "starting building: oscore-" in i:
                output.append(i)

        jobs_ids = "".join(output)
        jobs_ids = jobs_ids.replace("starting building:", " ")
        jobs_ids = re.findall(r"oscore-[\w-]+ #\d+", jobs_ids)

        res = {}
        for i in jobs_ids:
            name_id = i.split(" #")
            res[name_id[1]] = name_id[0]

        return res

    def get_job_results(self, job_name, get_last_build=False, job_id=None):

        results_multijobs = {}
        try:
            logger.info(f"Getting IDs multijobs: {job_name} {job_id}")

            job = self.server.get_job(job_name)

            if get_last_build:
                last_build = job.get_last_build()
                job_id = last_build.get_number()

            build = job.get_build(int(job_id))
            build_params = build.get_params()
            baseurl = build.baseurl
            build_status = str(build.get_status())
            timestamp = build.get_timestamp().timestamp()

            try:
                job_name = build_params["COOKIECUTTER_TEMPLATE_CONTEXT_FILE"]
            except KeyError:
                try:
                    job_name = build_params["STACK_CLUSTER_NAME"]
                except KeyError:
                    logger.warning(
                        "KeyError, there are no "
                        "COOKIECUTTER_TEMPLATE_CONTEXT_FILE "
                        "or STACK_CLUSTER_NAME"
                    )
                    pass

            results_multijobs["build_status"] = build_status
            results_multijobs["job_name"] = job_name
            results_multijobs["baseurl"] = baseurl
            results_multijobs["timestamp"] = timestamp
            logger.info(
                f"build status: {build_status} job name: "
                f"{job_name} baseurl: {baseurl}"
            )
            return results_multijobs
        except custom_exceptions.NotFound:
            logger.warning(
                f"Exception, NotFound: {type(custom_exceptions.NotFound)}"
            )
            logger.warning(f"Job was erased. Exception, NotFound: {job_name}")
            results_multijobs["build_status"] = "No Results"
            results_multijobs["job_name"] = job_name
            results_multijobs["baseurl"] = "No Results"
            results_multijobs["timestamp"] = "0.0"
            return results_multijobs

    def get_results_multijobs(self, job_names_to_ids):
        logger.info(f"Getting results multijobs: {job_names_to_ids}")

        list_results = []
        for job_id, job_name in job_names_to_ids.items():
            results_multijobs = self.get_job_results(
                job_id=job_id, job_name=job_name
            )

            list_results.append(results_multijobs)
        return list_results

    def get_results_singlejobs(self, job_name):
        logger.info(f"Getting results single jobs: {job_name}")

        results_singlejobs = self.get_job_results(
            job_name=job_name,
            get_last_build=True,
        )
        return results_singlejobs

    def job(self, job_name):
        return self.server.get_job(job_name)


def get_all_jobs_results():
    logger.info("Getting all jobs results")
    jr = GetJobsResults()

    m_jobs = {}
    for job_name in config.MULTIJOBS:
        logger.info(f"Getting results multi jobs: {job_name}")
        job_names_run_ids = jr.get_run_jobs_from_console_output(job_name)

        logger.info(f"Jobs names run IDs: {job_names_run_ids}")
        m_res = jr.get_results_multijobs(job_names_run_ids)
        m_jobs[job_name] = m_res

    s_jobs = {}
    for job_name in config.SINGLEJOBS:
        s_res = jr.get_results_singlejobs(job_name)
        s_jobs[job_name] = s_res

    return {"multi_results": m_jobs, "single_results": s_jobs}


def save_results_to_html(all_jobs_results):
    filename = datetime.datetime.now().strftime("%d-%m-%Y_%H_%M") + ".html"
    logger.info(f"Saving results to html file: {filename}")

    filename = config.GENERATED_REPORT + filename
    html = open(config.REPORT_TEMPLATE).read()
    template = Template(html)

    with open(filename, "w") as fh:
        fh.write(template.render(results=all_jobs_results))
    return filename


def datetimeformat(format):
    """
    Filter for jinja to get date time from timestamp

    :param format:  'timestamp': 1550768955.0
    :return: 2019-03-01 03:49:35
    """
    value = float(format)
    return datetime.datetime.utcfromtimestamp(value).strftime(
        "%d-%m-%Y  %H:%M:%S"
    )


jinja2.filters.FILTERS["datetimeformat"] = datetimeformat


if __name__ == "__main__":
    all_jobs_results = get_all_jobs_results()
    logger.info(f"all_jobs_results: {all_jobs_results}")

    all_jobs_results = update_all_jobs_results_with_artifacts(all_jobs_results)

    save_results_to_html(all_jobs_results)
    update_google_sheets.update_google_sheet(all_jobs_results)
