from typing import Iterator, List, Optional

from django.conf import settings
from django.utils.html import escape
from requests.exceptions import ReadTimeout
from retry import retry
from testrail_api import StatusCodeError, TestRailAPI

from ..utils import cached
from .enums import StatusEnum, TimeEnum

api = TestRailAPI(
    "https://mirantis.testrail.com/",
    settings.TESTRAIL_EMAIL,
    settings.TESTRAIL_PASSWORD,
)


@cached()
def get_project_id(project_name: str) -> Optional[int]:
    project = list(
        filter(
            lambda x: x["name"] == project_name,
            api.projects.get_projects()["projects"],
        )
    )
    if project:
        return project[0]["id"]
    else:
        return None


@cached(timeout=1 * TimeEnum.DAYS)
def get_suite_by_id(suite_id: int) -> dict:
    return api.suites.get_suite(suite_id)


@cached()
def get_suite_name_by_id(suite_id: int) -> str:
    return api.suites.get_suite(suite_id)["name"]


@cached()
def get_suite_test_type(suite_id: int) -> str:
    suite_name = get_suite_name_by_id(suite_id)
    return suite_name.split("]")[1]


@cached(timeout=1 * TimeEnum.HOURS)
def get_plans(project_id: int, **kwargs) -> List[dict]:
    plans = api.plans.get_plans(project_id=project_id, **kwargs)["plans"]
    return plans


@cached(
    timeout=2 * TimeEnum.HOURS,
    condition_for_endless_cache=lambda x: x is not None,
)
def get_planid_by_name(
    name: str, project_name: str, **kwargs
) -> Optional[int]:
    limit_step = 100
    for offset in range(0, 500, limit_step):
        plans = get_plans(
            project_id=get_project_id(project_name),
            limit=limit_step,
            offset=offset,
        )
        if not plans:
            return
        for plan in plans:
            print(f"{plan['name']=}")
            if plan["name"] == name:
                return plan["id"]
    return


@cached(timeout=1 * TimeEnum.HOURS)
def get_entries(plan_id: int) -> List[dict]:
    return api.plans.get_plan(plan_id)["entries"]


def get_run_by_id(run_id: int) -> dict:
    return api.runs.get_run(run_id)


@cached()
def get_run_name(run_id: int) -> str:
    return get_run_by_id(run_id)["name"]


def get_plan_by_id(plan_id: int) -> dict:
    return api.plans.get_plan(plan_id)


def get_result_history_for_case(
    case_id: int,
    status_id: int = None,
    project_name: str = "Mirantis Cloud Platform",
    plan_name: str = None,
    created_after: str = None,
    created_before: str = None,
    created_by: int = None,
    testrun_pattern: str = None,
    **kwargs,
) -> Iterator[List[dict]]:
    limit_step = 100
    suite_id = api.cases.get_case(case_id=case_id)["suite_id"]
    for offset in range(0, 2000, limit_step):
        plans = get_plans(
            project_id=get_project_id(project_name),
            limit=limit_step,
            offset=offset,
            created_after=created_after,
            created_before=created_before,
            created_by=created_by,
        )
        if not plans:
            return
        for plan in plans:
            if plan_name and plan_name not in plan["name"]:
                continue
            entries = get_entries(plan["id"])
            for entry in entries:
                for run in entry["runs"]:
                    if testrun_pattern and testrun_pattern not in run["name"]:
                        continue
                    if suite_id and run["suite_id"] != suite_id:
                        continue
                    if type(status_id) is list:
                        status_id = ",".join(map(lambda x: str(x), status_id))

                    results = get_result_for_case(
                        run_id=run["id"], case_id=case_id, status_id=status_id
                    )
                    if results:
                        yield results


def get_run_id(entries: List[dict], run_name: str) -> Optional[int]:
    entries = list(filter(lambda x: x["name"] == run_name, entries))
    if not entries:
        return None
    return entries[0]["runs"][0]["id"]


@cached(
    timeout=2 * TimeEnum.HOURS, condition_for_endless_cache=lambda x: x is None
)
@retry(ReadTimeout, delay=1, jitter=2, tries=3)
def get_result_for_case(
    run_id: int, case_id: int, **kwargs
) -> Optional[List[dict]]:
    try:
        results = api.results.get_results_for_case(run_id, case_id, **kwargs)[
            "results"
        ]
    except StatusCodeError:
        return None
    return results


def get_failed_tests(last_run_id: int, by_plans=False) -> List[dict]:
    failed_statuses = [StatusEnum.failed, StatusEnum.blocked]
    status_id = ",".join(map(str, failed_statuses))
    if by_plans:
        failed_tests = []
        for entry in get_entries(last_run_id):
            for run in entry["runs"]:
                failed_tests += api.tests.get_tests(
                    run_id=run["id"], status_id=status_id
                )["tests"]
        return failed_tests
    return api.tests.get_tests(last_run_id, status_id=status_id)["tests"]


def add_result(test_id: int, update_dict: dict) -> None:
    api.results.add_result(test_id, **update_dict)


@cached()
def is_testplan(plan_id):
    try:
        plan = api.plans.get_plan(plan_id)
        return bool(plan)
    except StatusCodeError as e:
        print(f"{e=}")
        return False


def html_link(type: str, id: int, title: str) -> str:
    return (
        f"<a href='https://mirantis.testrail.com/index.php?/{type}s/view/"
        f"{id}'>{escape(title)}</a>"
    )
