import os
import logging
from datetime import datetime, timedelta
import subprocess

from .. import models
from . import tasks
from .test_rail_api import get_planid_by_name

LOG = logging.getLogger(__name__)


def task_to_check_today_testplan():
    """
    Searching the today testplan
    Creates BotTestRun with this id
    Creates Periodic task to analyze created TestRun

    :return:
    """
    today = datetime.today().strftime("%Y-%m-%d")
    plan_name = f"[MCP2.0]OSCORE-{today}"
    plan_id = get_planid_by_name(
        name=plan_name, project_name="Mirantis Cloud Platform"
    )
    if not plan_id:
        print(f"Can't found {plan_name} TestPlan")
        return

    testrun_obj, _ = models.TestRailTestRun.objects.get_or_create(
        run_id=plan_id,
    )
    testrun_obj.run_name = f"[by scheduler] {plan_name}"
    testrun_obj.caching_tests_enabled = True
    testrun_obj.save()

    report_name = f"{testrun_obj.run_name}"
    path = os.path.join(models.fs.location, report_name)
    with open(path, "a"):
        pass

    report_obj, _ = models.TestRailReport.objects.get_or_create(
        report_name=report_name, path=path
    )
    report_obj.finished = False
    report_obj.save()

    return tasks.process_run(
        bot_run_id=testrun_obj.id,
        report_id=report_obj.id,
        path=path,
        is_testplan=True,
    )


def task_to_check_testplan(testplan_id: int):
    """
    Creates BotTestRun with this id
    Creates Periodic task to analyze created TestRun
    :param testplan_id:
    :return:
    """

    testrun_obj, _ = models.TestRailTestRun.objects.get_or_create(
        run_id=testplan_id,
    )
    testrun_obj.run_name = f"[by scheduler] plan {testplan_id}"
    testrun_obj.caching_tests_enabled = True
    testrun_obj.save()

    report_name = f"{testrun_obj.run_name}"
    path = os.path.join(models.fs.location, report_name)
    with open(path, "a"):
        pass

    report_obj, _ = models.TestRailReport.objects.get_or_create(
        report_name=report_name, path=path
    )
    report_obj.finished = False
    report_obj.save()

    return tasks.process_run(
        bot_run_id=testrun_obj.id,
        report_id=report_obj.id,
        path=path,
        is_testplan=True,
    )


def task_to_delete_old_2m_testruns():
    """
    Delete test runs after 2 month
    Clear mechanism 31.12.XXXX - 2 month
    delete 1-31.10.XXXX and older
    """
    today = datetime.today()
    border_date = (today - timedelta(days=30 * 2)).strftime("%Y-%m-%d")
    runs = models.TestRailTestRun.objects.all()
    for run in runs:
        if run.created_at.strftime("%Y-%m-%d") < border_date:
            run.delete()
            print(run.created_at.strftime("%Y-%m-%d"))


def task_to_delete_old_2m_reports():
    """
    Delete reports after 2 month
    Clear mechanism 31.12.XXXX - 2 month
    delete 1-31.10.XXXX and older
    """
    today = datetime.today()
    border_date = (today - timedelta(days=30 * 2)).strftime("%Y-%m-%d")

    reports = models.TestRailReport.objects.order_by("-created_at").all()
    for report in reports:
        if report.created_at.strftime("%Y-%m-%d") < border_date:
            LOG.info(f"Deleting {report=}")
            report.delete()

    test_results = models.TestResult.objects.order_by("-updated_at").all()
    for result in test_results:
        if result.updated_at.strftime("%Y-%m-%d") < border_date:
            LOG.info(f"Deleting {result=}")
            result.delete()


def task_to_analize_testrail_reports(plan_name: str):
    """
    Analise reports for testrail
    on one day or plan name
    """
    today = datetime.today().strftime("%Y-%m-%d")
    _plan_name = plan_name or f"[MCP2.0]OSCORE-{today}"

    _url = "https://mirantis.testrail.com"
    _email = os.environ.get("TESTRAIL_EMAIL")
    _password = os.environ.get("TESTRAIL_PASSWORD")
    _project_name = "'Mirantis Cloud Platform'"
    _jira_user = os.environ.get("JIRA_USER")
    _jira_password = os.environ.get("JIRA_PASSWORD")
    _sumReportOptions = [
        "--testrail-host",
        _url,
        "--testrail-user",
        _email,
        "--testrail-user-key",
        _password,
        "--testrail-plan",
        _plan_name,
        "--testrail-project",
        _project_name,
        "--out-type html",
        "--push-to-testrail",
        "--sort-by fails",
        "--jira-host https://mirantis.jira.com/",
        "--jira-user",
        _jira_user,
        "--jira-password",
        _jira_password,
    ]
    _sumReportOptionsList = map(str, _sumReportOptions)
    report_script = f"""\
              export TESTRAIL_URL={_url};
              export TESTRAIL_USER_EMAIL={_email};
              export TESTRAIL_USER_KEY={_password};
              pip install testplan_summary/.;
              python testplan_summary/report.py \
              create-report {" ".join(_sumReportOptionsList)}
              """
    p = subprocess.Popen(
        report_script, shell=True, stdout=subprocess.PIPE
    ).stdout.read()
    return p
