import logging
import os
import time

import config
import requests
import utils
from config import (
    JOBS_FOR_GETTING_LOGS_FROM_OUTPUT,
    LOGGIGNG_FOLDER,
    LOGGIGNG_JENKINS_API,
    LOGS_DIRECTORY,
)
from jenkinsapi import custom_exceptions
from jenkinsapi.jenkins import Jenkins
from jenkinsapi.utils.crumb_requester import CrumbRequester
from manage_files import delete_old_files

logging.basicConfig(
    format="[%(asctime)s][%(name)s][%(levelname)s] %(message)s",
    datefmt="%d-%m-%Y %H:%M:%S",
    handlers=[
        logging.FileHandler(
            "{}{}".format(LOGGIGNG_FOLDER, LOGGIGNG_JENKINS_API)
        ),
        logging.StreamHandler(),
    ],
    level=logging.INFO,
)
logger = logging.getLogger("jenkins_api")


class GetJobsResults:
    """
    Working with Jenkins API and tiny DB
    """

    def __init__(self, pipeline_job_name):
        self.server = Jenkins(
            config.JENKINS_URL,
            username=config.USERNAME,
            password=config.PASSWORD,
            requester=CrumbRequester(
                username=config.USERNAME,
                password=config.PASSWORD,
                baseurl=config.JENKINS_URL,
            ),
        )
        self.job = self.server.get_job(pipeline_job_name)
        self.pipeline_job_name = pipeline_job_name

    def get_all_jobs_ids(self):
        """
        Gets all the builds ID's for the particular job name.

        For example:
        Pipeline_job_name: oscore-artifatcts-collector
            [6150, 6149, 6148, ..., 6121]

        :return: list. Builds ID's
        """
        builds_ids = self.job.get_build_ids()
        logger.info(
            "Getting all builds ids: {}".format(self.pipeline_job_name)
        )
        return list(builds_ids)

    def get_job_build(self, build_id):
        try:
            job_build = self.job.get_build(build_id)
            return job_build
        except requests.exceptions.HTTPError:
            logger.warning("404 Client Error: Not Found for url")

    def manage_build_artifacts(self, job_build, build, build_date):
        build_artifacts = job_build.get_artifacts()

        for build_artfact in build_artifacts:
            if build_artfact:
                patch_to_artifact_file = utils.generate_patch_to_artifact_file(
                    logs_directory=LOGS_DIRECTORY,
                    job_name=self.pipeline_job_name,
                )
                saved_artifact_file_patch = build_artfact.save_to_dir(
                    patch_to_artifact_file
                )

                # rename saved file
                new_artifact_file_patch = utils.generate_artifact_file_patch(
                    saved_artifact_file_patch=saved_artifact_file_patch,
                    patch_to_artifact_file=patch_to_artifact_file,
                    pipeline_job_name=self.pipeline_job_name,
                    build_id=build,
                    build_date=build_date,
                )
                if not utils.check_if_file_exists(
                    patch_to_file=new_artifact_file_patch
                ):
                    new_artifact_filename = utils.rename_artifact_file(
                        old_artifact_file_patch=saved_artifact_file_patch,
                        new_artifact_file_patch=new_artifact_file_patch,
                    )
                    logger.info(
                        f"new_artifact_filename: {new_artifact_filename}"
                    )

                if utils.check_if_file_exists(
                    patch_to_file=saved_artifact_file_patch
                ):
                    os.remove(saved_artifact_file_patch)

    def get_build_artifacts(self, jobs_ids_list):
        """
        Gets all the build artifacts fir the build in a Jenkins Job
        Saves it to DB

        :param jobs_ids_list: List of the IDs for the particular job
        :return: None if Ok
        """
        logger.info(
            "Pipeline job name, jobs IDs list: {} {}".format(
                self.pipeline_job_name, jobs_ids_list
            )
        )
        build_counter = 1
        for build in jobs_ids_list:
            patch_to_file = utils.get_patch_to_file(
                job_name=self.pipeline_job_name, build_id=build
            )
            if not utils.check_if_file_exists(patch_to_file=patch_to_file):
                # If a build ID is not in the DB then add it
                try:
                    job_build = self.get_job_build(build_id=build)

                    # Build status FAILURE, ABORTED, SUCCESS, None
                    # Build status None means 'In progress'
                    job_build_status = job_build.get_status()
                    job_console = job_build.get_console()

                    if job_build and job_build_status:
                        # Check if Build status is not None than job is
                        # not in progress, finished
                        logger.info(
                            "Saving Buid to file: {}: {} "
                            "build: {} from {}".format(
                                self.pipeline_job_name,
                                build,
                                build_counter,
                                len(jobs_ids_list),
                            )
                        )
                        build_counter += 1

                        # When job is finished
                        job_timestamp = job_build.get_timestamp().timestamp()
                        build_date = utils.get_date_from_timestamp(
                            job_timestamp
                        )

                        # save build artifacts
                        self.manage_build_artifacts(
                            job_build=job_build,
                            build=build,
                            build_date=build_date,
                        )

                        # Save data th the file
                        utils.save_job_console_to_file(
                            logs_directory=LOGS_DIRECTORY,
                            job_name=self.pipeline_job_name,
                            build_id=build,
                            build_date=build_date,
                            build_status=job_build_status,
                            data_to_write=job_console,
                        )

                except custom_exceptions.NotFound:
                    logger.warning(
                        "Couldn't find a build: {}: {}".format(
                            self.pipeline_job_name, build
                        )
                    )
                    continue
                continue
            logger.warning(
                "Jenkins log output already saved: {}: "
                "{} build: {} from {}".format(
                    self.pipeline_job_name,
                    build,
                    build_counter,
                    len(jobs_ids_list),
                )
            )
            build_counter += 1

            continue
        return


def gathering_data_from_jenkins_all_jenkins_job(all_jenkins_job):
    """
    Gets console output from Jenkins jobs and save it to Databases
    :param all_jenkins_job: list
    :return: None if Ok
    """
    logger.info("Gathering data from Jenkins")
    for pj in all_jenkins_job:
        try:
            jr = GetJobsResults(pj)
            all_jobs_ids = jr.get_all_jobs_ids()
            jr.get_build_artifacts(all_jobs_ids)
        except requests.exceptions.ConnectionError:
            logger.warning(
                "Got an exception. ConnectionError. "
                "Too many API requests waiting for 700 sec"
            )
            time.sleep(700)
            continue


def gathering_data_from_jenkins_one_jenkins_job(one_jenkins_job):
    """
    Gets console output from Jenkins jobs and save it to Databases
    :param all_jenkins_job: list
    :return: None if Ok
    """
    logger.info("Gathering data from Jenkins parallel")
    try:
        jr = GetJobsResults(one_jenkins_job)
        all_jobs_ids = jr.get_all_jobs_ids()
        jr.get_build_artifacts(all_jobs_ids)
    except requests.exceptions.ConnectionError:
        logger.warning(
            "Got an exception. ConnectionError. "
            "Too many API requests waiting for 700 sec"
        )
        time.sleep(700)


def get_one_jenkins_job_one_id(jenkins_job_name, job_id):
    logger.info(
        "Getting one Jenkins job: {}: {}".format(jenkins_job_name, job_id)
    )
    try:
        jr = GetJobsResults(jenkins_job_name)
        jr.get_build_artifacts([int(job_id)])
    except requests.exceptions.ConnectionError:
        logger.warning(
            "Got an exception. ConnectionError. "
            "Too many API requests waiting for 700 sec"
        )

        time.sleep(700)


if __name__ == "__main__":
    gathering_data_from_jenkins_all_jenkins_job(
        JOBS_FOR_GETTING_LOGS_FROM_OUTPUT
    )
    delete_old_files(config.FILES_OLDER_THAN, config.LOGS_DIRECTORY)
