import logging
import yaml

from os import environ
from pathlib import Path

LOG = logging.getLogger("rp_reporter")

class AnsiColorFormatter(logging.Formatter):
    def format(self, record: logging.LogRecord):
        no_style = '\033[0m'
        bold = '\033[91m'
        grey = '\033[90m'
        yellow = '\033[93m'
        red = '\033[31m'
        red_light = '\033[91m'
        start_style = {
            'DEBUG': grey,
            'INFO': no_style,
            'WARNING': yellow,
            'ERROR': red,
            'CRITICAL': red_light + bold,
        }.get(record.levelname, no_style)
        end_style = no_style
        return f'{start_style}{super().format(record)}{end_style}'



RP_CONFIG_FILE = environ.get("RP_CONFIG_FILE") or (Path.home() / ".reportportal_config")


def from_conf(key_name):
    if not Path(RP_CONFIG_FILE).exists():
        LOG.debug(
            f"Can't get {key_name} because config file " f"not found: {RP_CONFIG_FILE}"
        )
        return None
    with open(RP_CONFIG_FILE) as f:
        yaml_config = yaml.safe_load(f)
        value = yaml_config.get(key_name)
        if value is None:
            LOG.debug(f"Can't get {key_name} because it's absent in {RP_CONFIG_FILE}")
        return value

def call_error(key_name):
    raise Exception(
        f"{key_name} should be defined in {RP_CONFIG_FILE} or "
        f"by environment variable"
    )

RP_LOGGING = environ.get("RP_LOGGING") or from_conf("RP_LOGGING") or logging.WARNING
RP_LOG_FILE = environ.get("RP_LOG_FILE") or from_conf("RP_LOG_FILE")

logger = logging.getLogger()
handler = logging.StreamHandler()
handler.setLevel(RP_LOGGING) # DEBUG INFO WARNING ERROR CRITICAL
formatter = AnsiColorFormatter('{asctime} | {levelname:<8s} | {name:<20s} | {message}', style='{')
handler.setFormatter(formatter)
logger.addHandler(handler)
logger.setLevel(RP_LOGGING) # DEBUG INFO WARNING ERROR CRITICAL

RP_APIKEY = (
    environ.get("RP_APIKEY") or from_conf("RP_APIKEY") or call_error("RP_APIKEY")
)
RP_ENDPOINT = (
    environ.get("RP_ENDPOINT") or from_conf("RP_ENDPOINT") or call_error("RP_ENDPOINT")
)
RP_PROJECT = (
    environ.get("RP_PROJECT") or from_conf("RP_PROJECT") or call_error("RP_PROJECT")
)

RP_LOG_BATCH_SIZE = int(
    environ.get("RP_LOG_BATCH_SIZE") or from_conf("RP_LOG_BATCH_SIZE") or 20
)

RP_LOG_KEYWORDS = (
    environ.get("RP_LOG_KEYWORDS", "").split(",")
    if environ.get("RP_LOG_KEYWORDS")
    else False or from_conf("RP_LOG_KEYWORDS") or [
        "failure",
        "fatal"
        "failed",
        "error",
        "exception",
        "assert",
        "discovered openstack controller version is",
    ]
)

RP_BLACKLISTED_LOGS = (
    environ.get("RP_BLACKLISTED_LOGS", "").split(",")
    if environ.get("RP_BLACKLISTED_LOGS")
    else False or from_conf("RP_BLACKLISTED_LOGS") or [
        "[Pipeline] error",
        "failureDomain:",
        "portprober_arping_target_failure",
        "ingress:[error-pages:",
        "templates/deployment-error.yaml",
        "templates/service-error.yaml",
        "rockoon/rockoon/exception.py",
        "rockoon/rockoon/exporter/exceptions.py",
        "AssertResult",
        "AssertTimestamp",
        "ignore_errors=",
        "FailureAction",
        "OnFailureJobMode",
        "StandardError",
        "SystemCallErrorNumber",
        "TimeoutStartFailureMode",
        "continue_on_error",
        "failed=0",
        "Naive assertion of",
        "All assertions passed"
    ]
)

JENKINS_USER = (
    environ.get('JENKINS_USER') or from_conf("JENKINS_USER")
)
JENKINS_PASSWORD = (
    environ.get('JENKINS_PASSWORD') or from_conf("JENKINS_PASSWORD")
)

TIME_FORMAT = "%Y-%m-%d"

if __name__ == "__main__":
    LOG.info(f"RP_APIKEY: {RP_APIKEY}")
    LOG.info(f"RP_ENDPOINT: {RP_ENDPOINT}")
    LOG.info(f"RP_PROJECT: {RP_PROJECT}")
