#!/usr/bin/python
import random

import click
import jenkins_jinny.main as jj
import jmespath
import itertools
import logging
import yaml
import uuid
import re

from collections import deque
from time import sleep
from typing import List, Optional
from pathlib import Path
from pbr.version import VersionInfo

from rp_reporter.settings import TIME_FORMAT
from rp_reporter.report_from_xml import timestamp, Reporter

from rp_reporter.settings import JENKINS_USER, JENKINS_PASSWORD
from rp_reporter.settings import RP_LOGGING

from rp_reporter.settings import RP_BLACKLISTED_LOGS, RP_LOG_KEYWORDS

LOG = logging.getLogger("rp_reporter")

print(RP_BLACKLISTED_LOGS)

def setup_logging(job: str | jj.Build):
    if isinstance(job, str):
        job = jj.Build(job)
    if RP_LOGGING == 'DEBUG':
        log_file_name = f'{job.name}-{job.number}.log'
        log_file_path = Path(__file__).parent / log_file_name
        LOG.warning(f"Logging {RP_LOGGING} level to file: {log_file_name}")
        file_handler = logging.FileHandler(log_file_path, encoding='utf-8')
        file_handler.setLevel(logging.DEBUG)
        file_handler.setFormatter(logging.Formatter('%(asctime)s | %(levelname)s | %(message)s'))
        logging.getLogger().addHandler(file_handler)


def grep(text: List, patterns: List, context_size: int, blacklist: List[str] = None):
    """Works like `grep -rn`
    returns found text with line before and after
    :param text:
    :param patterns: list of strings, strings to find
    :param context_size: count of lines before and after to return
    :return: Yields list of lines
    """

    context = deque(maxlen=context_size + 1)
    result = list()
    lines_to_read_after_fail = 0

    def found(_line):
        if any(pattern.lower() in _line.lower() for pattern in patterns):
            if any(blacklisted.lower() in _line.lower() for blacklisted in blacklist):
                return False
            return True
        return False

    for num, line in enumerate(text, start=1):
        context.append(f"{num} {line}")

        if result:
            result.append(f"{num} {line}")

        if found(line):
            # import ipdb; ipdb.set_trace()
            if not result:
                result = list(context)
            result.append("^" * 100)
            lines_to_read_after_fail = context_size + 1

        if lines_to_read_after_fail > 0:
            lines_to_read_after_fail -= 1

        if result and lines_to_read_after_fail == 0:
            yield result
            result = list()

    if result:
        yield result


class Description:
    def __init__(self):
        self.rockoon_version = None
        self.test_results = list()
        self.job_status = None
        self.job_number = None
        self.job_url = None
        self.rp_url = None

    @property
    def job_link(self):
        return f"[#{self.job_number} {self.job_status}]({self.job_url}) "

    def add_test_result(
            self, test_name: str = None, testrail_url: str = None, rp_url: str = None, statistics: str = None
    ):
        testrail_msg = f"[TestRailURL]({testrail_url})" if testrail_url else ""
        self.rp_url = rp_url
        test_result = f"{test_name} {testrail_msg} " f"{statistics}"

        self.test_results.append(test_result)

    def __repr__(self):
        # Return each test result on new line with "* " prefix
        test_results = "\n".join(f"* {r}" for r in self.test_results)
        return (
            f"\n___\n"
            f"{self.job_link}\n"
            f"`{self.rockoon_version}`\n"
            f"{test_results}\n"
            f"[RP_URL]({self.rp_url}) \n"
            f"___\n"
        )


def get_tags(job: jj.Build):
    tags = dict()
    tags["start_time"] = job.start_time.strftime(TIME_FORMAT)

    if context_name := job.param.OPENSTACK_CONTEXT_NAME:
        tags["openstack"] = context_name.split("/")[0]
        tags["context"] = context_name.split("/")[-1]

    mosk_versions = re.findall(r'mosk-(\d+\.\d+)', job.name)
    tags["mosk_version"] = f"mosk-{mosk_versions[0]}" if mosk_versions else "master"

    return tags


def get_tags_from_osdpl(osdpl_file):
    tags = dict()
    osdpl_content = yaml.safe_load(open(osdpl_file[0], "rb"))
    osdpl_dict = yaml.safe_load(osdpl_content)
    found_osdpl = jmespath.search("items[*].status.version", osdpl_dict)
    if not found_osdpl:
        LOG.error(f"Can't find osdpl info in {osdpl_file}")
        return
    tags["rockoon_version"] = found_osdpl[0]

    if ovn := jmespath.search("items[*].spec.features.neutron.backend", osdpl_dict):
        tags["neutron.backend"] = ovn[0]
    else:
        tags["neutron.backend"] = "ovs"

    if dvr := jmespath.search("items[*].spec.features.neutron.dvr.enabled", osdpl_dict):
        tags["dvr"] = dvr[0]
    else:
        tags["dvr"] = False

    if vpnaas := jmespath.search(
            "items[*].spec.features.neutron.extensions.vpnaas.enabled", osdpl_dict
    ):
        tags["vpnaas"] = vpnaas[0]
    else:
        tags["vpnaas"] = False

    if nova_images := jmespath.search(
            "items[*].spec.features.nova.images.backend", osdpl_dict
    ):
        tags["nova.images.backend"] = nova_images[0]

    return tags


def get_image_from_describe(
        job_obj, describe_file_pattern, image_pattern, from_artifactory=False
):
    if from_artifactory:
        local_describe_file = job_obj.get_artifacts(describe_file_pattern)
        if not local_describe_file:
            LOG.info(f"Can't find {describe_file_pattern} in {job_obj} artifactory")
        local_describe_file = local_describe_file[0]
    else:
        describe_file_url = job_obj.get_link_from_description(
            describe_file_pattern)
        if not describe_file_url:
            LOG.info(f"Can't find {describe_file_pattern} in {job_obj} " f"description")
            return
        temp_dir = Path("/tmp") / uuid.uuid4().hex.upper()
        temp_dir.mkdir(parents=True, exist_ok=True)
        local_describe_file = temp_dir / describe_file_pattern
        job_obj.download_file(describe_file_url[0], local_describe_file)

    with open(local_describe_file) as file:
        for line in file.readlines():
            if image_pattern in line and "Image:" in line:
                image = line.strip()
                return image


def report_job_status(job: jj.Build, job_suite_id: str, reporter: Reporter):
    rp_client = reporter.client
    if not job_suite_id:
        return
    subjob_item_id = rp_client.start_test_item(
        name=job.name,
        start_time=timestamp(),
        item_type="STEP",
        description=f"child {job.url} {job.duration} {job.status}",
        parent_item_id=job_suite_id,
    )
    match job.status:
        case "SUCCESS":
            status = "PASSED"
        case "BUILDING":
            status = "PASSED"
        case "FAILURE":
            status = "FAILED"
        case "ABORTED":
            status = "FAILED"
        case "UNSTABLE":
            status = "PASSED"
        case _:
            LOG.error(f"Unknown translation {job.status=} to status")
            status = "FAILED"
    # add logs to test

    all_logs = job.get_logs()
    for log in grep(
            text=all_logs, patterns=RP_LOG_KEYWORDS, context_size=8, blacklist=RP_BLACKLISTED_LOGS
    ):
        # LOG.error("Attach logs {}".format("\n".join(log)))
        rp_client.log(time=timestamp(), message="\n".join(log), item_id=subjob_item_id)
        sleep(0.001)
    rp_client.finish_test_item(
        item_id=subjob_item_id,
        status=status,
        end_time=timestamp(),
    )


def upload_job(
        job: str | jj.Build,
        suite_per_job: bool = False,
        tags: Optional[dict] = None
):
    if isinstance(job, str):
        job = jj.Build(job)
    if not tags:
        tags = dict()

    print(f"│˶˙ᵕ˙˶)꜆ I take {job}")

    run_tags = dict()
    run_tags.update(get_tags(job))

    reporter = Reporter()
    description = Description()
    rp_client = reporter.client
    launch_id = None
    job_suite_id = None
    artifactory_url = None

    description.job_status = job.status
    description.job_number = job.number
    description.job_url = job.url

    if suite_per_job:
        run_tags["jenkins_job"] = job.number
        launch_id = rp_client.start_launch(
            name=job.name,
            start_time=timestamp(),
            attributes=run_tags,
            description=f"Deployed job {job.url} by {job.triggered_by}",
        )
        if not rp_client.get_launch_info():
            LOG.error(f"[FIXME] Launch {launch_id} is not created ")
            # import ipdb;
            # ipdb.set_trace()
            # assert False
        print(f"(＾-＾)＿日 report will be here {rp_client.get_launch_ui_url()}")
        job_suite_id = rp_client.start_test_item(
            name="CI jobs",
            start_time=timestamp(),
            item_type="suite",
        )
        reporter.schedule_finishing(job_suite_id)
    rp_client.log(
        time=timestamp(),
        message=f"Job status: {job.status}",
        # item_id=launch_id
    )

    for child in itertools.chain([job], job.heirs):
        child: jj.Build
        rp_client.log(
            time=timestamp(),
            message=f"{child} {child.status} {child.url}",
            # item_id=launch_id
        )
        report_job_status(job=child, job_suite_id=job_suite_id, reporter=reporter)

        # test_tags = deepcopy(tags)
        test_tags = dict()
        print(f"⫍‍⌕‍⫎ tests in {child}")
        test_results_files = None
        suite_description = ""
        match child.name:
            case "deploy-openstack-k8s-env":
                if base_ubuntu := job.param.HEAT_BASE_IMAGE_DISTRO:
                    run_tags["base_ubuntu"] = base_ubuntu
                if masakari := job.param.INSTALL_MASAKARI:
                    run_tags["masakari"] = masakari
                if vault := job.param.INSTALL_VAULT:
                    run_tags["vault"] = vault
                if ceph := job.param.INSTALL_CEPH:
                    run_tags["ceph"] = ceph
                if telemetry := job.param.INSTALL_TELEMETRY:
                    run_tags["telemetry"] = telemetry

                osdpl_file = child.get_artifacts("deployed.yaml")
                if not osdpl_file:
                    LOG.error(f"Can't find osdpl file in {job}")
                    continue

                run_tags.update(get_tags_from_osdpl(osdpl_file))
                rp_client.update_test_item(attributes=run_tags,
                                           item_uuid=launch_id)
                description.rockoon_version = run_tags.get("rockoon_version")

            case "tempest-runner-k8s":
                title = "Tempest"
                test_results_files = child.get_link_from_description("tempest_report.xml")
                if not test_results_files:
                    LOG.error(f"Can't found 'tempest_report.xml' in {child.url}")

                image = get_image_from_describe(
                    job_obj=child,
                    describe_file_pattern="tempest_pod.describe",
                    image_pattern="/openstack/tempest:",
                )
                suite_description += f"{image}<br>"

                if test_scheme := yaml.safe_load(child.param.TEST_SCHEME):
                    suite_description += f"regex: {test_scheme.get('regex')}<br>"

            case "stepler-runner-k8s":
                title = "Stepler"
                test_results_files = child.get_artifacts("stepler_test_results.xml")
                if not test_results_files:
                    LOG.error(
                        f"Can't found 'stepler_test_results.xml' in "
                        f"{child.url} artifacts"
                    )
                image = get_image_from_describe(
                    job_obj=child,
                    describe_file_pattern="stepler_pod.describe",
                    image_pattern="/openstack/stepler:",
                    from_artifactory=True,
                )
                suite_description += f"{image}<br>"

            case "oscore-si-tests-runner-k8s":
                title = "SI tests"
                test_results_files = child.get_link_from_description("si_test_report.xml")
                if not test_results_files:
                    LOG.error(f"Can't found 'si_test_report.xml' in {child.url}")

            case "oscore-functional-tests-runner":
                title = "Rockoon Functional"
                test_results_files = child.get_link_from_description("si_test_report.xml")
                if not test_results_files:
                    LOG.error(f"Can't found 'si_test_report.xml' in {child.url}")

            case "oscore-rulexam-tests-runner":
                title = "Rulexam"
                test_results_files = child.get_link_from_description("si_test_report.xml")
                if not test_results_files:
                    LOG.error(f"Can't found 'si_test_report.xml' in {child.url}")

            case "oscore-perf-analyzer-tests-runner":
                title = "Performance tests"
                test_results_files = child.get_link_from_description("si_test_report.xml")
                if not test_results_files:
                    LOG.error(f"Can't found 'si_test_report.xml' in {child.url}")

            case "si-test-check-downtime-statistic":
                title = "Downtime tests"
                if artifactory_url := child.get_link_from_description():
                    artifactory_url = artifactory_url[0]
                if not artifactory_url:
                    LOG.error(
                        f"Can't found 'test_check_downtime_statistic_result.xml' in {child.url}"
                    )
                test_results_files = [f"{artifactory_url}/artifacts/"
                                      f"test_check_downtime_statistic_result.xml"]

            case "collect-openstack-kaas-artifacts":
                artifactory_url = child.description.split("url: ")[-1]
                rp_client.log(
                    time=timestamp(),
                    message=f"Pod Logs {artifactory_url}/pod-logs.tar.gz",
                )

        if child.param.RUN_TESTS:
            suite_description += f"RUN_TESTS = {child.param.RUN_TESTS}<br>"
            test_tags["test"] = child.param.RUN_TESTS

        if not test_results_files:
            # We can iterate by child jobs which don't contain any reports
            continue
        if testrail_url := child.get_link_from_description("testrail.com"):
            testrail_url = testrail_url[0]
        rp_client.log(
            time=timestamp(),
            message=f"Found file to upload: {test_results_files}",
        )
        report_path = test_results_files[0]
        LOG.info(
            "=== report_xml {kwargs}".format(
                kwargs=dict(
                    report_path=report_path,
                    title=title,
                    attributes=test_tags,
                    link=job.url,
                    description=suite_description,
                    to_specific_launch=launch_id,
                )
            )
        )

        print(f"(っ･-･)っ Uploading {report_path}")
        reported_suite_id, stats = reporter.report_xml(
            report_path=report_path,
            title=title,
            attributes=test_tags,
            link=job.url,
            description=suite_description,
            to_specific_launch=launch_id,
        )
        description.add_test_result(
            test_name=f"{title} {test_tags.get('test', '')} ",
            testrail_url=testrail_url,
            rp_url=rp_client.get_launch_ui_url(),
            statistics=stats,
        )
        rp_client.log(
            time=timestamp(),
            message=f"Reported with stats: {stats}",
        )

    run_tags.update(tags)
    rp_client.log(
        time=timestamp(),
        message="Reporting completed",
    )
    if suite_per_job:
        report_url = rp_client.get_launch_ui_url()
        rp_client.finish_launch(
            end_time=timestamp(),
            attributes=run_tags,
            description=str(description)
                        + f"\nPod Logs {artifactory_url}/pod-logs.tar.gz",
        )
        print(f"report is here {report_url}")
        if JENKINS_USER or JENKINS_PASSWORD:
            print("Pushing new description to job...")
            try:
                job.description = (
                        job.description + f"<br><br> "
                                          f"<a href='{report_url}'>Link to ReportPortal</a> <br>"
                )
            except Exception as e:
                print(f"Can't push description to {job=}: {e}")
    print(f" ʕノ•ᴥ•ʔノ Completed")


def upload_view(view, pattern):
    for job in jj.jobs_in_view(view, ""):
        if not job.number:
            continue
        if pattern and not pattern in job.name:
            continue
        upload_job(job, suite_per_job=True)


@click.group()
def cli():
    pass


@cli.command()
@click.argument("job_url")
@click.option("--tag", 'tags', multiple=True, default=[],
              help="Add/override tag for testRun. "
                   "Use a semicolon as a delimiter --tag version:25.1")
def report_job(job_url, tags: List[str]):
    tags_dict = dict()
    for tag in tags:
        k, v = tag.split(":", 1)
        tags_dict[k] = v

    setup_logging(job=job_url)
    upload_job(job=job_url,
               tags=tags_dict,
               suite_per_job=True)


@cli.command()
@click.option("--pattern", default=None, help="Upload only job with pattern")
@click.argument("view_url")
def report_view(view_url, pattern):
    """
    :param view_url: Url to the view
    """
    upload_view(view_url, pattern)


@cli.command()
@click.argument("report_path")
def report_xmlfile(report_path):
    """
    :param report_path: Url or file location of xunit report
    """
    title = report_path.split("/")[-1]
    Reporter().report_xml(
        report_path=report_path,
        title=title,
    )


@cli.command()
def version():
    package_name = "rp-reporter"
    info = VersionInfo(package_name)
    print("rp-reporter " + info.version_string_with_vcs())


if __name__ == "__main__":
    cli()
