from django.core.files.storage import FileSystemStorage
from django.db import models
from django.utils.timezone import now, timedelta
from django_celery_beat.models import PeriodicTask


class IntegerListField(models.Field):
    def __init__(self, *args, **kwargs):
        kwargs["editable"] = False
        super(IntegerListField, self).__init__(*args, **kwargs)

    def db_type(self, connection):
        return "text"

    def from_db_value(self, value, expression, connection):
        if not value:
            return []
        return [int(x) for x in value.split(",")]

    def to_python(self, value):
        if isinstance(value, list):
            return value
        if not value:
            return []
        return [int(x) for x in value.split(",")]

    def get_prep_value(self, value):
        if not value:
            return ""
        return ",".join(str(int(x)) for x in value)


def default_created_after():
    return now() + timedelta(days=-3 * 30)


class TestRailTestRun(models.Model):
    project_name = models.CharField(
        max_length=300, default="Mirantis Cloud Platform"
    )
    plan_name = models.CharField(max_length=300, default="[MCP2.0]OSCORE")
    run_name = models.CharField(max_length=300, blank=True)
    testrun_pattern = models.CharField(max_length=300, blank=True)
    run_id = models.CharField(max_length=300)
    checked_tests = IntegerListField(default=list())
    caching_tests_enabled = models.BooleanField(default=False)
    created_by_id = models.IntegerField(default="109")
    filter_func = models.TextField(null=True, blank=True)
    ip_filter = models.BooleanField(default=True)
    uuid_filter = models.BooleanField(default=True)
    filter_last_traceback = models.BooleanField(default=True)
    created_before = models.DateField(default=now)
    created_after = models.DateField(default=default_created_after)
    created_at = models.DateTimeField(auto_now_add=True)

    @property
    def text_filters(self):
        return {
            "filter_last_traceback": self.filter_last_traceback,
            "uuid_filter": self.uuid_filter,
            "ip_filter": self.ip_filter,
            "filter_func": self.filter_func,
        }

    @property
    def testrail_filters(self):
        return {
            "created_by_id": self.created_by_id,
            "created_after": self.created_after,
            "created_before": self.created_before,
            "testrun_pattern": self.testrun_pattern,
            "plan_name": self.plan_name,
        }

    class Meta:
        ordering = ["-run_id"]


fs = FileSystemStorage()


class TestRailReport(models.Model):
    path = models.FileField(storage=fs, null=True, blank=True, max_length=500)
    report_name = models.CharField(max_length=300)
    test_results = IntegerListField(default=list())
    finished = models.BooleanField(default=False)
    created_at = models.DateTimeField(auto_now_add=True)


class TestResult(models.Model):
    result_id = models.CharField(max_length=50)
    text = models.CharField(max_length=10000, default="")
    action_needed = models.BooleanField(default=True)
    updated_at = models.DateTimeField(auto_now=True)


class ActionLog(models.Model):
    name = models.CharField(max_length=500)
    date = models.DateTimeField(null=True)


class SuitePassRate(models.Model):
    SUITE_CHOICES = [
        (
            "Tempest",
            (
                ("10650", "[MCP2.0_QUEENS]Tempest"),
                ("10651", "[MCP2.0_ROCKY]Tempest"),
                ("10635", "[MCP2.0_STEIN]Tempest"),
                ("10653", "[MCP2.0_TRAIN]Tempest"),
                ("10710", "[MCP2.0_USSURI]Tempest"),
                ("10888", "[MCP2.0_VICTORIA]Tempest"),
                ("11167", "[MCP2.0_WALLABY]Tempest"),
                ("11188", "[MCP2.0_XENA]Tempest"),
                ("11170", "[MCP2.0_YOGA]Tempest"),
                ("11192", "[MCP2.0_ANTELOPE]Tempest"),
                ("11195", "[MCP2.0_CARACAL]Tempest"),
            ),
        ),
        (
            "Stepler",
            (
                ("10886", "[MCP2.0_USSURI]Stepler"),
                ("10887", "[MCP2.0_VICTORIA]Stepler"),
                ("11171", "[MCP2.0_YOGA]Stepler"),
                ("11193", "[MCP2.0_ANTELOPE]Stepler"),
                ("11196", "[MCP2.0_CARACAL]Stepler"),
            ),
        ),
    ]
    suite_id = models.CharField(max_length=20, choices=SUITE_CHOICES)
    suite_name = models.CharField(max_length=100, blank=True)
    passrate_by_tests = models.JSONField(default="{}", blank=True)
    status = models.TextField(max_length=300, blank=True)
    finished = models.BooleanField(default=False, blank=True)


class DiffOfSuitesPassRates(models.Model):
    limit = models.IntegerField(default=10, blank=True)
    test_keyword = models.CharField(default="", max_length=300, blank=True)
    report1 = models.ForeignKey(
        to=SuitePassRate,
        related_name="report1",
        on_delete=models.CASCADE,
        blank=True,
    )
    report2 = models.ForeignKey(
        to=SuitePassRate,
        related_name="report2",
        on_delete=models.CASCADE,
        blank=True,
    )
    started_at = models.DateTimeField(auto_created=True, auto_now=True)


TASKS = [
    (
        "control.celery_tasks.tasks.check_today_testplan",
        "Check today testplan",
        [],
    ),
    (
        "control.celery_tasks.tasks.check_specific_testplan",
        "Check specific testplan",
        ["testplan_id_arg"],
    ),
    (
        "control.celery_tasks.tasks.delete_old_2m_testruns",
        "Delete previous 2-month TestRuns(for bot view)",
        [],
    ),
    (
        "control.celery_tasks.tasks.delete_old_2m_reports",
        "Delete previous 2-month Reports(for bot view)",
        [],
    ),
    (
        "control.celery_tasks.tasks.analize_testrail_reports",
        "Summary report on one day",
        ["plan_name"],
    ),
]

TASK_CHOICES = list(map(lambda x: x[:-1], TASKS))


class CronPeriodicTask(PeriodicTask):
    cron = models.CharField(default="", max_length=300, blank=False)
    task_name = models.CharField(
        max_length=300, choices=TASK_CHOICES, default=TASK_CHOICES[0][0]
    )
    testplan_id_arg = models.CharField(max_length=30, blank=True, null=True)

    class Meta:
        ordering = ["id"]
