# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import random

import pytest

from si_tests import logger
from si_tests.managers.clustercheck_mos_manager import ClusterCheckMosManager

LOG = logger.logger


class TestTFCassandraRestoration(object):
    # Workload details
    lb_url = None

    @staticmethod
    def _set_cls_attr(request, stack_outputs, attr):
        for output in stack_outputs:
            if output['output_key'] == attr:
                setattr(request.cls, attr, output['output_value'])

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, request, openstack_client_manager, tf_manager):
        LOG.info("Create heat stack with loadbalancer before Cassandra cluster tests")
        stack = ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager)
        self._set_cls_attr(request, stack['outputs'], 'lb_url')
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_delete_cassandra_db_pod(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Check Cassandra database restoration after pod deletion.

        Scenario:
            1. Delete cassandra pod
            2. Wait cassandra cluster become ready
            3. Check cassandra db status
            4. Check tf-api readiness
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_cassandra_pods()
        pod = random.choice(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_casasndracluster_status(status="Pending", timeout=180)
        tf_manager.wait_casasndracluster_status(status="Running")

        show_step(3)
        ClusterCheckMosManager.check_cassandra_nodes_config(os_manager=os_manager)

        show_step(4)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(5)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_drop_data_cassandra_db_pod(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Check Cassandra database restoration after deletion of persistent volume (PV) data.

        Scenario:
            1. Delete pvc one of cassandra pod and restart it
            2. Wait cassandra cluster become ready
            3. Check cassandra db status
            4. Check tf-api readiness
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_cassandra_pods()
        pod = random.choice(pods)
        vol = next(filter(lambda vol: vol['name'] == "data", pod.data['spec']['volumes']))
        pvc_name = vol['persistent_volume_claim']['claim_name']
        pvc = tf_manager.api.pvolumeclaims.get(namespace=tf_manager.tf_namespace, name=pvc_name)
        LOG.info(f"Delete pvc {pvc.data['metadata']['name']} and restart {pod.data['metadata']['name']} pod")
        pvc.delete()
        pod.delete()

        show_step(2)
        tf_manager.wait_casasndracluster_status(status="Pending", timeout=180)
        tf_manager.wait_casasndracluster_status(status="Running")

        show_step(3)
        ClusterCheckMosManager.check_cassandra_nodes_config(os_manager=os_manager, actualize_nodes_config=True)

        show_step(4)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(5)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)
