import pytest
from si_tests import logger
from si_tests import settings


LOG = logger.logger

# (vastakhov) Migrate possible from docker to containerd and vice versa. Behaviour can not be controlled and defined by
# cluster status before migration. (E.g. if cluster runtime is docker - migration to containerd will be performed.
# if cluster runtime is containerd - cluster will be migrated to docker)
#
# Extended test - tl;dr migrate 2 nodes -> revert 2 nodes -> migrate all nodes
# Quick test - tl;dr migrate all nodes


def _get_machines_to_migrate(cluster, machines):
    machines_to_migrate = []
    target_runtime = ''
    for machine in machines:
        if 'kaas.mirantis.com/preferred-container-runtime' in machine.annotations:
            machines_to_migrate.append(machine)
            if machine.annotations['kaas.mirantis.com/preferred-container-runtime'] == 'docker':
                target_runtime = 'containerd'
            elif machine.annotations['kaas.mirantis.com/preferred-container-runtime'] == 'containerd':
                target_runtime = 'docker'

    if not machines_to_migrate and cluster.is_management:
        old_runtime = cluster.runtime.runtime
        assert old_runtime in ['docker', 'containerd'], (f"Cluster have wrong runtime: {old_runtime} or "
                                                         f"migration in progress.")
        if old_runtime == 'docker':
            target_runtime = 'containerd'
        elif old_runtime == 'containerd':
            target_runtime = 'docker'

        LOG.banner(f"Target runtime: {target_runtime}")
        machines = cluster.get_machines()
        machines_to_migrate = []
        control_target = machines[0]
        machines_to_migrate.append(control_target)
    elif not machines_to_migrate and cluster.is_child:
        old_runtime = cluster.runtime.runtime
        assert old_runtime in ['docker', 'containerd'], (f"Cluster have wrong runtime {old_runtime} or "
                                                         f"migration in progress.")
        if old_runtime == 'docker':
            target_runtime = 'containerd'
        elif old_runtime == 'containerd':
            target_runtime = 'docker'

        LOG.banner(f"Target runtime: {target_runtime}")
        machines = cluster.get_machines()
        machines_to_migrate = []
        control_target = [machine for machine in machines if machine.machine_type == 'control'][0]
        machines_to_migrate.append(control_target)
        # get ovs compute
        ovs_nodes = cluster.get_machines(
            k8s_labels={"openstack-compute-node": "enabled"})
        if ovs_nodes:
            machines_to_migrate.append(ovs_nodes[0])
        # get tf compute if exists
        tf_nodes = cluster.get_machines(
            k8s_labels={"tfcontrol": "enabled"})
        if tf_nodes:
            machines_to_migrate.append(tf_nodes[0])
    return machines_to_migrate, target_runtime


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_runtime_migrate_extended(kaas_manager, show_step, _):
    """Test migrate runtime (extended).

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Start partial migrate
        3. Wait migration finished and check cluster state
        4. Rollback migrated nodes to old runtime
        5. Wait migration finished and check cluster state
        6. Migrate all nodes
        7. Wait migration finished and check cluster state
    """

    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    old_runtime = cluster.runtime.runtime
    assert old_runtime in ['docker', 'containerd'], (f"Cluster have wrong runtime {old_runtime} or "
                                                     f"migration in progress.")
    target_runtime = ''
    if old_runtime == 'docker':
        target_runtime = 'containerd'
    elif old_runtime == 'containerd':
        target_runtime = 'docker'

    LOG.banner(f"Target runtime: {target_runtime}")
    machines = cluster.get_machines()
    machines_to_migrate = []
    control_target = [machine for machine in machines if machine.machine_type == 'control'][0]
    machines_to_migrate.append(control_target)
    workers = [machine for machine in machines if machine.machine_type == 'worker']
    if workers:
        # get first of workers if exists
        machines_to_migrate.append(workers[0])
    show_step(2)
    cluster.runtime.partial_migrate(machines_to_migrate, target_runtime)
    show_step(3)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    show_step(4)
    cluster.runtime.partial_migrate(machines_to_migrate, old_runtime)
    show_step(5)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    show_step(6)
    cluster.runtime.partial_migrate(machines, target_runtime)
    show_step(7)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()


@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_runtime_migrate_partial_child(kaas_manager, show_step, _):
    """Test migrate runtime partial, test should find automatically migrated nodes

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Migrate 2 any nodes dockerd <-> containerd, test works in both sides
        3. Wait migration finished and check cluster state
    """

    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    machines = cluster.get_machines()
    machines_to_migrate, target_runtime = _get_machines_to_migrate(cluster, machines)
    show_step(2)
    cluster.runtime.partial_migrate(machines_to_migrate, target_runtime)
    show_step(3)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_runtime_migrate_partial_mgmt(kaas_manager, show_step, _):
    """Test migrate runtime partial, test should find automatically migrated nodes

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Migrate 2 any nodes dockerd <-> containerd, test works in both sides
        3. Wait migration finished and check cluster state
    """

    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    machines = cluster.get_machines()
    machines_to_migrate, target_runtime = _get_machines_to_migrate(cluster, machines)
    show_step(2)
    cluster.runtime.partial_migrate(machines_to_migrate, target_runtime)
    show_step(3)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()


@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_runtime_migrate_quick(kaas_manager, show_step, _):
    """Test migrate runtime (quick)

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Migrate all nodes
        3. Wait migration finished and check cluster state
    """

    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    old_runtime = cluster.runtime.runtime
    assert old_runtime in ['docker', 'containerd'], (f"Cluster have wrong runtime {old_runtime} or "
                                                     f"migration in progress.")
    target_runtime = ''
    if old_runtime == 'docker':
        target_runtime = 'containerd'
    elif old_runtime == 'containerd':
        target_runtime = 'docker'

    LOG.banner(f"Target runtime: {target_runtime}")
    machines = cluster.get_machines()
    show_step(2)
    cluster.runtime.partial_migrate(machines, target_runtime)
    show_step(3)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_runtime_migrate_and_rollback(kaas_manager, show_step, _):
    """Test migrate runtime (extended) with rollback

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Start partial migrate
        3. Wait migration finished and check cluster state
        4. Rollback migrated nodes to old runtime
        5. Wait migration finished and check cluster state
        6. Migrate all nodes
        7. Wait migration finished and check cluster state
        8. Rollback to initial runtime
        9. Wait migration finished and check cluster state
    """

    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    old_runtime = cluster.runtime.runtime
    assert old_runtime in ['docker', 'containerd'], (f"Cluster have wrong runtime {old_runtime} or "
                                                     f"migration in progress.")
    target_runtime = ''
    if old_runtime == 'docker':
        target_runtime = 'containerd'
    elif old_runtime == 'containerd':
        target_runtime = 'docker'

    LOG.banner(f"Target runtime: {target_runtime}")
    machines = cluster.get_machines()
    machines_to_migrate = []
    control_target = [machine for machine in machines if machine.machine_type == 'control'][0]
    machines_to_migrate.append(control_target)
    workers = [machine for machine in machines if machine.machine_type == 'worker']
    if workers:
        # get first of workers if exists
        machines_to_migrate.append(workers[0])
    show_step(2)
    cluster.runtime.partial_migrate(machines_to_migrate, target_runtime)
    show_step(3)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    show_step(4)
    cluster.runtime.partial_migrate(machines_to_migrate, old_runtime)
    show_step(5)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    show_step(6)
    cluster.runtime.partial_migrate(machines, target_runtime)
    show_step(7)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    show_step(8)
    LOG.banner(f"Target runtime: {old_runtime}")
    cluster.runtime.partial_migrate(machines, old_runtime)
    show_step(9)
    cluster.check.check_runtime()
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()


@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_inplace_distribution_upgrade_and_migrate_runtime(kaas_manager, show_step, _):
    """Test migrate runtime (quick) with distribution upgrade

    Scenario:
        1. Get cluster and verify it's in ready state
        2. Migrate all nodes and
        3. Wait migration and upgrade finished and check cluster state
        4. Check uptimes and reboot fact on nodes
    """
    # NOTE(vastakhov): Combined test represents quick migration to target runtime with inplace distribution upgrade
    # included. Origin parts of distrib upgrade used here located in tests/lcm/test_distrib_upgrade.py and introduced
    # by @ddmitriev
    show_step(1)
    ns = settings.TARGET_NAMESPACE
    name = settings.TARGET_CLUSTER
    LOG.info(f"Target cluster: {ns}/{name}")
    managed_ns = kaas_manager.get_namespace(ns)
    cluster = managed_ns.get_cluster(name)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    old_runtime = cluster.runtime.runtime
    assert old_runtime in ['docker', 'containerd'], (f"Cluster have wrong runtime {old_runtime} or "
                                                     f"migration in progress.")
    target_runtime = ''
    if old_runtime == 'docker':
        target_runtime = 'containerd'
    elif old_runtime == 'containerd':
        target_runtime = 'docker'

    target_release_name = cluster.clusterrelease_version
    target_distro = kaas_manager.get_latest_available_clusterrelease_distro(target_release_name)
    updated_machines, not_updated_machines = cluster.get_machines_by_distribution(target_distro)

    if not not_updated_machines:
        msg = (f"Looks like all Machines in the cluster '{ns}/{cluster.name}' "
               f"already upgraded to the distribution '{target_distro}', skipping inplace upgrade")
        LOG.warning(msg)
        pytest.skip(msg)

    if cluster.is_postpone_distribution_upgrade_enabled:
        LOG.info("Disable 'postponeDistributionUpdate' flag to perform inplace distribution upgrade")
        cluster.set_postpone_distribution_upgrade(enabled=False)

    # Get uptimes count before distribution upgrade
    uptimes = cluster.get_machines_uptime()

    LOG.banner(f"Target runtime: {target_runtime}. Target distro: {target_distro}")
    machines = cluster.get_machines()
    show_step(2)
    cluster.runtime.upgrade_distrib_and_migrate(machines, target_runtime, target_distro)
    show_step(3)
    cluster.check.check_runtime(timeout=13200)
    cluster.check.check_cluster_readiness()
    cluster.check.check_inplace_distribution_upgrade_completed(target_distro)
    # Check the Cluster objects
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_k8s_pods()
    cluster.check.check_diagnostic_cluster_status()
    show_step(4)
    uptimes_after = cluster.get_machines_uptime()
    # Check that each Machine got at least one reboot if distribution was upgraded
    reboots_msg = ""
    updated_machines_names = [m.name for m in updated_machines]
    for k, v in uptimes.items():
        # It is possible to have 1-second difference between uptime before and after upgrade
        # due to rounding float numbers in calculations.
        time_delta = abs((uptimes_after[k] - v).total_seconds())
        LOG.debug(f"Time difference between uptime before and after upgrade = {time_delta}")
        if k in updated_machines_names:
            # Machine was already upgraded before the test, and should not be rebooted
            if time_delta > 1:
                reboots_msg += (
                    f"Machine '{k}' distribution was already upgraded before test, but machine uptime has been changed:"
                    f"  uptime before='{v}' , uptime after='{uptimes_after[k]}' (must be unchanged).\n")
        else:
            # Machine should be upgraded and rebooted
            if time_delta < 2:
                reboots_msg += (
                    f"Machine '{k}' distribution just upgraded, but machine uptime is still the same: "
                    f"  uptime before='{v}' , uptime after='{uptimes_after[k]}'\n")
    assert not reboots_msg, (
        f"At least 1 reboot is expected during distribution upgrade\n"
        f"{reboots_msg}")
