import pytest

from si_tests import logger
from si_tests import settings
from si_tests.utils import exceptions, utils
from kubernetes.client.rest import ApiException

LOG = logger.logger


@pytest.mark.parametrize(
    'deletion_policy, new_delete_api',
    [
        ('forced', True)
    ]
)
def test_forced_deletion_policy_for_broken_machine(kaas_manager, show_step, deletion_policy, new_delete_api):
    """Delete broken machine with forced option.

    Scenario:
        1. Just take 1 worker machine
        2. Shutdown machine
        3. Delete machine with forced option
        4. Wait until machine deleted
        5. Add new machine instead of deleted one
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Providers with Ceph are skipped because they need additional steps,
    # VSphere is skipped because of limited resources
    if cluster.provider in [utils.Provider.vsphere, utils.Provider.equinixmetalv2, utils.Provider.baremetal]:
        pytest.skip("skip forced deletion tests, since some left resources may lead to false negative test results")

    show_step(1)
    machines = cluster.get_machines(machine_type="worker", machine_status="Ready")

    if machines:
        machine = machines[0]
        machine_name = machines[0].name
    else:
        raise Exception('Looks like that cluster doesn\'t have worker machines in Ready status')

    LOG.info(f'Get k8s node for machine {machine_name}')
    node_name = machine.get_k8s_node_name()

    # Save machine data for adding new machine back after test
    machine_data = cluster.get_existing_machine_spec(machine)
    LOG.info(f'Machine data before deletion: {machine_data}')

    show_step(2)
    try:
        machine.exec_pod_cmd("sudo shutdown now", timeout=40)
    except exceptions.TimeoutError:
        pass

    LOG.info(f'Wait until k8s node %s become not ready {node_name}')
    cluster.check.wait_k8s_node_status(
        node_name,
        expected_status="NotReady",
        timeout=900)

    show_step(3)
    LOG.info("Check forced policy forbidden to set initially")
    with pytest.raises(ApiException) as exc_info:
        machine.set_deletion_policy('forced')
    assert exc_info.value.status == 400, (f"Machine {machine.name} return unexpected status "
                                          f"on 'deletionPolicy: force', expected status code 400, "
                                          f"actual: {exc_info.value.status}")
    assert exc_info.value.reason == 'Bad Request', (f"Machine {machine.name} return unexpected reason"
                                                    f"on 'deletionPolicy: force', expected reason "
                                                    f"'Bad Request', actual: {exc_info.value.reason}")

    machine.delete(check_deletion_policy=True, new_delete_api=new_delete_api)
    machine.set_deletion_policy(deletion_policy)

    show_step(4)
    cluster.wait_machine_deletion(machine_name, interval=60)

    LOG.info(f"Wait k8s node delete - {node_name}")
    cluster.check.check_deleted_node(node_name)

    LOG.info('Machine was deleted successfully')

    show_step(5)
    LOG.info('Add new machine into cluster instead of deleted')
    new_machine = cluster.create_machine(node_type="node", region=cluster.region_name, **machine_data)
    cluster.check.check_created_machine(new_machine)
    LOG.info('Machine was created successfully, test is finished!')
