import pytest
import yaml

from si_tests import logger
from si_tests import settings
import si_tests.managers.machine_deletion_policy_manager as machine_deletion_policy_manager

LOG = logger.logger


@pytest.fixture()
def machine_delete_scenarios(kaas_manager):
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Here we use unsafe twice, because we have 3 machines and don't need to use force, it's not a misprint
    if cluster.machine_deletion_policy_enabled():
        if cluster.workaround.prodx_27814():
            return (
                machine_deletion_policy_manager.check_machine_graceful_delete,
                machine_deletion_policy_manager.check_machine_delete_without_policy,
                machine_deletion_policy_manager.check_machine_unsafe_delete,
            )
        else:
            return (
                machine_deletion_policy_manager.check_machine_graceful_delete,
                machine_deletion_policy_manager.check_machine_unsafe_delete,
                machine_deletion_policy_manager.check_machine_unsafe_delete,
            )
    else:
        return (
            machine_deletion_policy_manager.check_machine_delete_without_policy,
            machine_deletion_policy_manager.check_machine_delete_without_policy,
            machine_deletion_policy_manager.check_machine_delete_without_policy,
        )


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_delete_master_node(kaas_manager, show_step, machine_delete_scenarios, _):
    """Delete/Create manager machines one by one

    Scenario:
            1. Delete manager machine & wait until deleted
            2. Create new manager machine
            3. Wait for machine spawned & became ready
            4. Check upgrade index
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    LOG.info("Namespace name - %s", namespace_name)

    cluster = ns.get_cluster(cluster_name)
    LOG.info("Cluster name - %s", cluster_name)

    region = cluster.region_name
    machines = cluster.get_machines(machine_type='control')

    for machine, delete_scenario in zip(machines, machine_delete_scenarios):
        LOG.info("Process machine - %s", machine.name)
        machine_data = cluster.get_existing_machine_spec(machine)

        show_step(1)
        if (cluster.workaround.prodx_27814() and
                delete_scenario is machine_deletion_policy_manager.check_machine_delete_without_policy):
            delete_scenario(cluster, machine, check_deletion_policy=True)
        else:
            delete_scenario(cluster, machine)

        show_step(2)
        new_machine = cluster.create_machine(
            node_type="master",
            region=region,
            **machine_data)

        show_step(3)
        cluster.check.check_created_machine(new_machine)

        show_step(4)
        # Check new index is greater than max existed
        LOG.info(f"Check new machine {new_machine.name} upgrade index value")
        current_index = cluster.get_machine(new_machine.name).get_upgrade_index()
        LOG.info(f"New machine {new_machine.name} upgrade index is: {current_index}")
        all_indexes = {machine.name: machine.get_upgrade_index() for machine in cluster.get_machines() if
                       machine.name != new_machine.name}
        fail_message = (f"Wrong index {current_index} for machine {new_machine.name}. "
                        f"New index should be greater than {max(all_indexes.values())}. "
                        f"Existed indexes excluding new machine:\n {yaml.dump(all_indexes)}")
        assert current_index > max(all_indexes.values()), fail_message
