import pytest

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Namespace, Cluster
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
def test_cleanup_openstack_env(kaas_manager, show_step, ):
    """Cleanup OpenStack

    Scenario:
            1. Verify that the OsDpl object is present
            2. Delete the OsDpl object
            3. Verify that all pods and jobs have been deleted
            4. Delete Persistent Volume Claims
            5. Delete the MariaDB state ConfigMap
            6. Delete secrets
            7. Verify that OpenStack ConfigMaps and secrets have been deleted
            8. Check cluster readiness
    """

    managed_ns: Namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster: Cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)
    is_tf_backend = None

    show_step(1)
    osdpl = cluster.k8sclient.openstackdeployment.get(name=settings.OSH_DEPLOYMENT_NAME,
                                                      namespace=settings.OSH_NAMESPACE)
    if not cluster.k8sclient.openstackdeployment.present(name=settings.OSH_DEPLOYMENT_NAME,
                                                         namespace=settings.OSH_NAMESPACE):
        pytest.skip(
            "OsDpl object '{}' is not present in '{}' namespace. Nothing to delete".format(settings.OSH_DEPLOYMENT_NAME,
                                                                                           settings.OSH_NAMESPACE))
    else:
        LOG.info("Found OsDpl object '{}' in namespace '{}'. OS version '{}'".format(
            settings.OSH_DEPLOYMENT_NAME,
            settings.OSH_NAMESPACE,
            osdpl.read().spec.get('openstack_version')))
        is_tf_backend = cluster.tf_enabled()
        LOG.info("Cluster backend is {}".format('TF' if is_tf_backend else 'OVS'))

    show_step(2)
    LOG.info("Delete OsDpl object")
    osdpl.delete()

    def _wait_osdpl_delete():
        if cluster.k8sclient.openstackdeployment.present(name=settings.OSH_DEPLOYMENT_NAME,
                                                         namespace=settings.OSH_NAMESPACE):
            LOG.info("OsDpl object '{}' in '{}' namespace still exist. Waiting...".format(settings.OSH_DEPLOYMENT_NAME,
                                                                                          settings.OSH_NAMESPACE))
            return False
        else:
            LOG.info("OsDpl object successfully deleted")
            return True

    waiters.wait(_wait_osdpl_delete, timeout=3600, interval=60, timeout_msg="OsDpl not removed within timeout")

    show_step(3)

    def _wait_os_pods_and_jobs():
        pods = cluster.k8sclient.pods.list(namespace=settings.OSH_NAMESPACE)
        jobs = cluster.k8sclient.jobs.list(namespace=settings.OSH_NAMESPACE)
        if pods == [] and jobs == []:
            LOG.info("All pods and jobs successfully deleted")
            return True
        else:
            LOG.info("\nNext pods still present: {}\nNext jobs still present: {}\nWaiting..".format(
                ", ".join([pod.name for pod in pods]), ", ".join([job.name for job in jobs])))
            return False

    waiters.wait(_wait_os_pods_and_jobs, timeout=1800, interval=60,
                 timeout_msg="Some pods or jobs not removed within timeout")

    show_step(4)

    def _get_pvc():
        pvc = cluster.k8sclient.pvolumeclaims.list(namespace=settings.OSH_NAMESPACE)
        pvc.extend(cluster.k8sclient.pvolumeclaims.list(namespace="openstack-redis"))
        return pvc

    def _wait_pvc():
        pvc = _get_pvc()
        if not pvc:
            LOG.info("All PVC successfully deleted")
            return True
        else:
            LOG.info("Next PVC still present: {}".format(", ".join([c.name for c in pvc])))
            return False

    for claim in _get_pvc():
        LOG.info(f"Delete pvc: {claim.name}")
        claim.delete()

    LOG.info("Wait for the PVC to be removed")
    waiters.wait(_wait_pvc, timeout=1800, interval=30, timeout_msg="Some PVC not removed within timeout")

    show_step(5)
    configmap_name = "openstack-mariadb-mariadb-state"
    configmap = cluster.k8sclient.configmaps.get(name=configmap_name, namespace=settings.OSH_NAMESPACE)
    LOG.info("Delete ConfigMap '{}' in '{}' namespace".format(configmap_name, settings.OSH_NAMESPACE))
    configmap.delete()

    show_step(6)
    secrets = cluster.k8sclient.secrets.list(namespace=settings.OSH_NAMESPACE)
    LOG.info("Secrets to delete: {}".format(", ".join([secret.name for secret in secrets])))
    for secret in secrets:
        secret.delete()

    show_step(7)
    # based on https://mirantis.jira.com/browse/PRODX-32434
    exclude_configmap = ['kube-root-ca.crt']
    exclude_secrets = ['default-token', 'mke-proxy-secret', 'oidc-cert']

    configmaps = cluster.k8sclient.configmaps.list(namespace=settings.OSH_NAMESPACE)
    configmaps_name = [c.name for c in configmaps]
    failed_configmaps = [item for item in configmaps_name if not item.startswith(tuple(exclude_configmap))]
    secrets = cluster.k8sclient.secrets.list(namespace=settings.OSH_NAMESPACE)
    secrets_name = [s.name for s in secrets]
    failed_secrets = [item for item in secrets_name if not item.startswith(tuple(exclude_secrets))]

    if failed_configmaps:
        LOG.error("Next ConfigMaps still exist: {}".format(", ".join(failed_configmaps)))
    if failed_secrets:
        LOG.error("Next Secrets still exist: {}".format(", ".join(failed_secrets)))
    if failed_configmaps or failed_secrets:
        pytest.fail("Some ConfigMaps or Secrets still exist after cleanup")
    else:
        LOG.info(
            "All ConfigMaps except ({}) and Secrets except ({}) successfully deleted".format(
                ", ".join(exclude_configmap),
                ", ".join(exclude_secrets)))

    show_step(8)
    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status(timeout=1800, interval=30)
    if cluster.workaround.skip_kaascephcluster_usage():
        cluster.check.wait_miracephhealth_state()
    else:
        cluster.check.wait_kaascephcluster_state()
    cluster.check.check_helmbundles()
    if is_tf_backend:
        # related to https://mirantis.jira.com/browse/PRODX-32445
        exp_provider_status = False
        expected_fails = ["Kubernetes"]
        expected_condition_fails = {'Kubernetes': 'not ready: daemonSets: tf/tf-config'}
    else:
        exp_provider_status = True
        expected_fails = None
        expected_condition_fails = None
    cluster.check.check_cluster_readiness(timeout=1800, interval=30, exp_provider_status=exp_provider_status,
                                          expected_fails=expected_fails,
                                          expected_condition_fails=expected_condition_fails)
