import os
import pytest
import uuid
import untangle

from si_tests.deployments.utils import commons
from si_tests.deployments.utils import file_utils
from si_tests.deployments.utils import kubectl_utils
from si_tests import settings

TEMPLATE_PATH = file_utils.join(os.path.dirname(os.path.abspath(__file__)), "templates/advanced_compute.yaml")


def get_instance_pcpu_cells(openstack_client_manager, hypervisor_hostname, domain_obj):
    host_capabilities = untangle.parse(
        openstack_client_manager.virsh_capabilities(hypervisor_hostname)
    ).capabilities.host
    host_cpus_by_cell = {}
    for cell in host_capabilities.topology.cells.cell:
        host_cpus_by_cell[cell.get_attribute("id")] = [
            cpu.get_attribute("id") for cpu in cell.cpus.children]
    instance_pcpus = []
    for vcpupin in domain_obj.domain.cputune.vcpupin:
        instance_pcpus.append(vcpupin.get_attribute("cpuset"))
    instance_pcpu_cells = []
    for cell, host_pcpus in host_cpus_by_cell.items():
        for instance_pcpu in instance_pcpus:
            if instance_pcpu in host_pcpus:
                instance_pcpu_cells.append(cell)
                break
    return instance_pcpu_cells


def save_stack_server_console(openstack_client_manager, stack_name):
    servers_ids = [x["physical_resource_id"] for x in
                   openstack_client_manager.stack.resource_list([stack_name, "--filter", "type=OS::Nova::Server"],
                                                                yaml_output=True)]
    for server in servers_ids:
        openstack_client_manager.server_collect_console(server, settings.ARTIFACTS_DIR)


def test_numa_no_hugepages(openstack_client_manager, request):
    stack_name = f"numa-nohugepages-{uuid.uuid4()}"
    stack_params = {"server_availability_zone": settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME,
                    "server_cpu_policy": "dedicated",
                    "server_numa_nodes": "2",
                    }

    try:
        openstack_client_manager.create_stack(request, stack_name, TEMPLATE_PATH, stack_params=stack_params)
    except Exception as e:
        save_stack_server_console(openstack_client_manager, stack_name)
        raise Exception(f"Failed to create stack: {e}")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split(
        '.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name))
    # This instance should have 2 numa nodes
    assert 2 == len(domain_obj.domain.cpu.numa)

    instance_pcpu_cells = get_instance_pcpu_cells(openstack_client_manager, hypervisor_hostname, domain_obj)

    # Each instance CPU should come from different host NUMA node
    assert 2 == len(instance_pcpu_cells)


def test_nodes_hugepages(openstack_client_manager):
    kubectl = kubectl_utils.Kubectl()

    nodes = kubectl.get_node_name_by_label(
        settings.OPENSTACK_NODES_WITH_HUGEPAGES_LABEL)
    assert (
        len(nodes) != 0
    ), f"There are no nodes with {settings.OPENSTACK_NODES_WITH_HUGEPAGES_LABEL} label"  # noqa: E501
    hugepages_enabled = []
    for node in nodes:
        hp = int(
            openstack_client_manager.exec_libvirt(
                node, ["/sbin/sysctl", "-b", "vm.nr_hugepages"]
            )
        )
        commons.LOG.info(f"{node} => {hp} hugepages are available")
        hugepages_enabled.append(hp > 0)

    if not all(hugepages_enabled):
        pytest.fail("Not all nodes have hugepages enabled")


def test_instance_hugepages(openstack_client_manager, request):
    stack_name = f"hugepages-{uuid.uuid4()}"
    stack_params = {"server_availability_zone": settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME,
                    "server_mem_page_size": "2048"
                    }

    try:
        openstack_client_manager.create_stack(request, stack_name, TEMPLATE_PATH, stack_params=stack_params)
    except Exception as e:
        save_stack_server_console(openstack_client_manager, stack_name)
        raise Exception(f"Failed to create stack: {e}")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server[
        "OS-EXT-SRV-ATTR:hypervisor_hostname"].split('.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    try:
        for hp in domain_obj.domain.memoryBacking.hugepages:
            hp_size = hp.page.get_attribute("size")
            commons.LOG.info(
                f"Compute instance has hugepages with size: {hp_size}"
            )
    except AttributeError as e:
        commons.LOG.error("Compute instance has no hugepages on board")
        raise e


def test_numa_hugepages(openstack_client_manager, request):
    stack_name = f"numa-hugepages-{uuid.uuid4()}"
    stack_params = {"server_availability_zone": settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME,
                    "server_cpu_policy": "dedicated",
                    "server_numa_nodes": "2",
                    "server_mem_page_size": "2048"
                    }

    try:
        openstack_client_manager.create_stack(request, stack_name, TEMPLATE_PATH, stack_params=stack_params)
    except Exception as e:
        save_stack_server_console(openstack_client_manager, stack_name)
        raise Exception(f"Failed to create stack: {e}")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split(
        '.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name))
    # This instance should have 2 numa nodes
    assert 2 == len(domain_obj.domain.cpu.numa)

    instance_pcpu_cells = get_instance_pcpu_cells(openstack_client_manager, hypervisor_hostname, domain_obj)

    # Each instance CPU should come from different host NUMA node
    assert 2 == len(instance_pcpu_cells)
    # check hugepages are allocated to each numa
    try:
        for hp in domain_obj.domain.memoryBacking.hugepages.page:
            hp_size = hp.get_attribute("size")
            hp_nodeset = hp.get_attribute("nodeset")
            commons.LOG.info(
                f"Compute instance nodeset {hp_nodeset} has hugepages with size: {hp_size}"
            )
    except AttributeError as e:
        commons.LOG.error("Compute instance nodeset has no hugepages on board")
        raise e


def test_dpdk(openstack_client_manager, request):
    kubectl = kubectl_utils.Kubectl()
    stack_name = f"dpdk-{uuid.uuid4()}"
    stack_params = {"server_availability_zone": settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME,
                    "server_mem_page_size": "2048"
                    }

    try:
        openstack_client_manager.create_stack(request, stack_name, TEMPLATE_PATH, stack_params=stack_params)
    except Exception as e:
        save_stack_server_console(openstack_client_manager, stack_name)
        raise Exception(f"Failed to create stack: {e}")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split(
        '.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    # Check if TungstenFabric enabled
    ns = settings.OSH_NAMESPACE
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    command = '{} -o yaml'.format(osdpl_name)
    osdpl = kubectl.get('osdpl', command, ns).result_yaml
    backend = osdpl['spec']['features']['neutron'].get('backend')
    tf = True if backend == 'tungstenfabric' else False

    for interface in domain_obj.domain.devices.interface:
        target_dev = interface.target.get_attribute("dev")
        if tf:
            cmd = ['vif', '--list']
            interface_list = openstack_client_manager.exec_vrouter_agent(
                hypervisor_hostname, cmd)
            assert 'PMD: {}'.format(target_dev) in interface_list
        else:
            cmd = ['ovs-vsctl', 'get', 'interface', target_dev, 'type']
            interface_type = openstack_client_manager.exec_neutron_ovs_agent(
                hypervisor_hostname, cmd)
            assert 'dpdk' in interface_type
