import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters
from si_tests.utils.utils import Provider

LOG = logger.logger


@pytest.fixture(scope='module')
def _child_cluster(kaas_manager):
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    yield child_cluster


@pytest.fixture(scope='module')
def _k8s_client(_child_cluster):
    yield _child_cluster.k8sclient


@pytest.fixture(scope='module')
def _dashboard_client(_child_cluster):
    yield _child_cluster.mkedashboardclient


@pytest.fixture(scope='module')
def _swarm_client(_child_cluster):
    swarmclient = _child_cluster.get_dockerclient()
    yield swarmclient


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_ucp_dashboard_nodes(_child_cluster, _k8s_client, _dashboard_client,
                             _swarm_client, show_step, _):
    """Match nodes and node roles from kaas, ucp dashboard, k8s and docker

    Scenario:
        1. Get node names and IP addresses from UCP dashboard
        2. Get node names from docker swarm (requires private SSH key)
        3. Get node names from kubernetes on UCP
        4. Get node IP addresses from KaaS machines
        5. Check that private IP addresses from KaaS 'machines' match
           node addresses from UCP dashboard
        6. Check that node names from UCP dashboard match node names
           from docker swarm
        7. Check that node names from UCP dashboard match node names
           from k8s cluster in UCP
        8. Check that nodes 'management' status from UCP dashboard match
           the 'control' status from KaaS

    Required environment variables:
        KUBECONFIG : config to access KaaS cluster
        TARGET_NAMESPACE : child cluster namespace in KaaS
        TARGET_CLUSTER : child cluster name in KaaS
        KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE : SSH private key for child cluster
    """
    def _test():
        show_step(1)
        dashboard_nodes = _dashboard_client.get_ready_nodes()
        dashboard_node_names = sorted([node["Description"]["Hostname"]
                                       for node in dashboard_nodes])
        dashboard_node_ips_type = {
            node["Status"]["Addr"]: node["Spec"]["Role"]
            for node in dashboard_nodes}
        dashboard_node_ips = sorted(dashboard_node_ips_type.keys())
        LOG.info("UCP dashboard node names: {0}\nUCP dashboard node IPs: {1}"
                 .format(dashboard_node_names, dashboard_node_ips))

        if _child_cluster.provider is Provider.byo:
            LOG.warning("This is BYO cluster. "
                        "Skipping part of ucp dasbhoard tests")
            return True

        show_step(2)
        swarm_node_names = sorted([node["Hostname"]
                                   for node in _swarm_client.node_ls(verbose=True) if node['Status'] == 'Ready'])
        LOG.info("Docker swarm node names: {0}".format(swarm_node_names))

        show_step(3)
        k8s_node_names = sorted([node.name
                                 for node in _k8s_client.nodes.list_all()])
        LOG.info("Kubernetes node names: {0}".format(k8s_node_names))

        show_step(4)
        kaas_machines = _child_cluster.get_machines()
        kaas_node_ips_type = {machine.internal_ip: machine.machine_type
                              for machine in kaas_machines}
        kaas_node_ips = sorted(kaas_node_ips_type.keys())
        LOG.info("KaaS machine IPs: {0}".format(kaas_node_ips))

        show_step(5)
        assert set(dashboard_node_ips) == set(kaas_node_ips), (
            "Cluster node IP addresses from UCP dashboard don't match "
            "the addresses from KaaS machines:\n"
            "  UCP dashboard node addresses: {0}\n"
            "  KaaS machine addresses: {1}"
            .format(dashboard_node_ips, kaas_node_ips))

        show_step(6)
        assert set(dashboard_node_names) == set(swarm_node_names), (
            "Cluster node names from UCP dashboard don't match "
            "the node names from docker swarm:\n"
            "  UCP dashboard node names: {0}\n"
            "  Docker swarm node names: {1}"
            .format(dashboard_node_names, swarm_node_names))

        show_step(7)
        assert set(dashboard_node_names) == set(k8s_node_names), (
            "Cluster node names from UCP dashboard don't match "
            "the node names from kubernetes in UCP cluster:\n"
            "  UCP dashboard node names: {0}\n"
            "  Kubernetes node names: {1}"
            .format(dashboard_node_names, k8s_node_names))

        show_step(8)
        ucp_kaas_mgmt_translate = {
            "manager": "control",
            "worker": "worker",
        }

        kaas_ucp_translate = {
            "control": "control",
            "worker": "worker",
            "storage": "worker",

        }
        dashboard_node_types = sorted([
            f"{ip}/{ucp_kaas_mgmt_translate[nodetype]}"
            for ip, nodetype in dashboard_node_ips_type.items()])
        kaas_node_types = sorted([
            f"{ip}/{kaas_ucp_translate[nodetype]}"
            for ip, nodetype in kaas_node_ips_type.items()])
        LOG.info(dashboard_node_types)
        LOG.info(kaas_node_types)

        assert set(dashboard_node_types) == set(kaas_node_types), (
            "Cluster node types from UCP dashboard don't match "
            "the node types from KaaS for this cluster")

    waiters.wait_pass(_test, timeout=120, interval=20)


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_ucp_dashboard_kubernetes(_k8s_client, _dashboard_client,
                                  show_step, _):
    """Match kubernetes namespaces, pods and services from dashboard and k8s

    Scenario:
        1. Get namespaces from UCP dashboard and from kubernetes
        2. Check that namespace names are the same in UCP dashboard and in k8s
        3. Get pods from UCP dashboard and from kubernetes
        4. Check that pod names are the same in UCP dashboard and in k8s
        5. Get k8s services from UCP dashboard and from kubernetes
        6. Check that k8s service names are the same
           in UCP dashboard and in kubernetes

    Required environment variables:
        KUBECONFIG : config to access KaaS cluster
        TARGET_NAMESPACE : child cluster namespace in KaaS
        TARGET_CLUSTER : child cluster name in KaaS
    """
    def _test():
        show_step(1)
        dashboard_namespace_names = sorted(
            _dashboard_client.get_k8s_namespace_names())
        LOG.info("UCP dashboard namespace names: {0}"
                 .format(dashboard_namespace_names))

        k8s_namespace_names = sorted(
            [ns.name for ns in _k8s_client.namespaces.list_all()])
        LOG.info("Kubernetes namespace names: {0}"
                 .format(k8s_namespace_names))

        show_step(2)
        assert set(dashboard_namespace_names) == set(k8s_namespace_names), (
            "Namespaces from UCP dashboard don't match "
            "the namespaces from kubernetes:\n"
            "  UCP dashboard namespaces: {0}\n"
            "  Kubernetes namespaces: {1}"
            .format(dashboard_namespace_names, k8s_namespace_names))

        show_step(3)
        dashboard_pod_names = []
        for ns in dashboard_namespace_names:
            pod_names = _dashboard_client.get_k8s_pod_names(ns)
            dashboard_pod_names.extend([
                '/'.join([ns, pod]) for pod in pod_names])
        dashboard_pod_names.sort()

        k8s_pod_names = sorted([
            '/'.join([pod.namespace, pod.name])
            for pod in _k8s_client.pods.list_all()])

        show_step(4)
        assert set(dashboard_pod_names) == set(k8s_pod_names), (
            "Pod list from UCP dashboard don't match "
            "the pod list from kubernetes")

        show_step(5)
        dashboard_service_names = []
        for ns in dashboard_namespace_names:
            service_names = _dashboard_client.get_k8s_service_names(ns)
            dashboard_service_names.extend([
                '/'.join([ns, service]) for service in service_names])
        dashboard_service_names.sort()
        LOG.info("UCP dashboard service names: {0}"
                 .format(dashboard_service_names))

        k8s_service_names = sorted([
            '/'.join([service.namespace, service.name])
            for service in _k8s_client.services.list_all()])
        LOG.info("Kubernetes service names: {0}"
                 .format(k8s_service_names))

        show_step(6)
        assert set(dashboard_service_names) == set(k8s_service_names), (
            "K8s services list from UCP dashboard don't match "
            "the services list from kubernetes")

    waiters.wait_pass(_test, timeout=300, interval=20)


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_ucp_dashboard_swarm(_swarm_client, _dashboard_client, _child_cluster,
                             show_step, _):
    """Match docker services and containers from dashboard and from docker cli

    Scenario:
        1. Get swarm services from UCP dashboard and from docker
        2. Check that swarm services are the same
           in UCP dashboard and in docker
        3. Get swarm containers from UCP dashboard and from docker
        4. Check that containers on the swarm nodes are the same
           in UCP dashboard and in docker swarm

    Required environment variables:
        KUBECONFIG : config to access KaaS cluster
        TARGET_NAMESPACE : child cluster namespace in KaaS
        TARGET_CLUSTER : child cluster name in KaaS
        KAAS_CHILD_CLUSTER_PRIVATE_KEY_FILE : SSH private key for child cluster
    """
    def _test():
        show_step(1)
        dashboard_service_names = sorted(
            _dashboard_client.get_swarm_service_names())
        LOG.info("Swarm service names from UCP dashboard: {0}"
                 .format(dashboard_service_names))

        swarm_service_names = sorted(
            [service["Name"] for service in _swarm_client.service_ls()])
        LOG.info("Swarm service names from docker cli: {0}"
                 .format(swarm_service_names))

        show_step(2)
        assert set(dashboard_service_names) == set(swarm_service_names), (
            "Swarm services list from UCP dashboard don't match "
            "the services list from docker")

        show_step(3)
        LOG.info("Get container names from UCP dashboard")
        dashboard_container_names = sorted([
            container
            for containers in _dashboard_client.get_swarm_container_names()
            for container in containers])

        dashboard_nodes = _dashboard_client.get_ready_nodes()
        dashboard_node_ip_names = {
            node["Status"]["Addr"]: node["Description"]["Hostname"]
            for node in dashboard_nodes}

        swarm_container_names = []
        for machine in _swarm_client.machines:
            swarm_node_name = dashboard_node_ip_names[machine.internal_ip]
            LOG.info("Get container names from swarm node {0}"
                     .format(swarm_node_name))
            swarm_container_names.extend(sorted(
                ["/{0}/{1}".format(swarm_node_name, container["Names"])
                 for container in _swarm_client.container_ls(machine.name)]))
        swarm_container_names.sort()

        show_step(4)
        assert set(dashboard_container_names) == set(swarm_container_names), (
            "Swarm container list from UCP dashboard don't match "
            "the container list from docker")

    if _child_cluster.provider is Provider.byo:
        msg = "This is BYO cluster. Skipping this ucp dashboard test"
        LOG.warning(msg)
        pytest.skip(msg)
    waiters.wait_pass(_test, timeout=300, interval=20)
