import pytest
import os
import yaml
import ipaddress
import time

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, templates, locks
from si_tests.deployments.utils import file_utils, kubectl_utils
from si_tests.managers.kaas_manager import Manager

LOG = logger.logger


def render_templates_and_apply(template_class, options, kubectl):
    templates_base_path = settings.CLUSTER_TEMPLATES
    env_name = settings.ENV_CONFIG_NAME

    LOG.info(f"Handling {template_class}")
    class_templates_path = os.path.join(templates_base_path, env_name, template_class)

    out_teplates_dir = os.path.join(settings.MOSK_CHILD_CLUSTER_TEMPLATES_BASE_DIR, template_class)
    os.makedirs(out_teplates_dir, exist_ok=True)

    for file_tempate in os.listdir(class_templates_path):
        data = templates.render_template(os.path.join(class_templates_path, file_tempate), options)
        if not data:
            continue
        with open(os.path.join(out_teplates_dir, file_tempate), 'w') as f:
            f.write(data)
            f.flush()
            kubectl.apply(f.name)


@pytest.mark.usefixtures("log_method_time")
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}".format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_equinixmetalv2_child_templated(_):
    """Deploy child cluster on top of management cluster."""

    env_name = settings.ENV_CONFIG_NAME
    env_parameters = settings.ENV_PARAMETERS_NAME
    lab_tempates = os.path.join(settings.CLUSTER_TEMPLATES, env_name)
    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_MGMT_PATH)

    res_env = {}
    default_env = yaml.safe_load(templates.render_template(os.path.join(lab_tempates, "env", "defaults.yaml")))
    custom_env = yaml.safe_load(templates.render_template(os.path.join(lab_tempates, "env", env_parameters)))

    res_env = utils.merge_dicts(default_env, custom_env)

    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()

    cluster_name = res_env["cluster_name"]
    release_name = res_env["release_name"]
    namespace_name = res_env["namespace"]
    public_key_name = f"{settings.KAAS_CHILD_CLUSTER_NAME}-key"

    # Collecting env data
    dynamic_env = {
        "public_key_name": public_key_name,
        "equinix_credential_name": f"{settings.KAAS_CHILD_CLUSTER_NAME}-cred",
        "public_key": pub_key_content,
    }
    metro = settings.KAAS_EQUINIX_CHILD_METRO
    dynamic_env["equinix_facility"] = metro

    kubectl = kubectl_utils.Kubectl()
    render_templates_and_apply("namespaces", res_env, kubectl)

    tf_data = file_utils.get_yaml_content(settings.TERRAFORM_NETWORK_METADATA)
    nm = locks.NetworkLockManager(
        f"{cluster_name}-lock",
        namespace_name,
        kaas_manager,
        network_data=tf_data,
    )
    network_selector = {}
    for network_type, selector in res_env["network_selector"].items():
        selector.update({"metro": metro})
        network_selector[f"{cluster_name}-{network_type}"] = selector

    res = nm.pick_nets(network_selector)

    pxe_net = res[f"{cluster_name}-pxe"]
    pxe_network = ipaddress.IPv4Network(f"{pxe_net['subnet']}/{pxe_net['mask']}")
    pxe_dhcp_range = f"{pxe_network[11]}-{pxe_network[50]}"
    pxe_ipam_ranges = f"{pxe_network[51]}-{pxe_network[99]}"
    pxe_metallb_ranges = f"{pxe_network[100]}-{pxe_network[200]}"
    pxe_lb_host = str(pxe_network[10])

    ceph_cluster_net = res[f"{cluster_name}-{res_env['ceph_cluster_network_role_name']}"]
    ceph_public_net = res[f"{cluster_name}-{res_env['ceph_public_network_role_name']}"]
    ceph_network = {
        "clusterNet": str(ipaddress.IPv4Network(f"{ceph_cluster_net['subnet']}/{ceph_cluster_net['mask']}")),
        "publicNet": str(ipaddress.IPv4Network(f"{ceph_public_net['subnet']}/{ceph_public_net['mask']}")),
    }
    dynamic_env["ceph_network"] = ceph_network

    additional_vlans = []
    for tag, net in res.items():
        network = ipaddress.IPv4Network(f"{net['subnet']}/{net['mask']}")
        if tag == f"{cluster_name}-pxe":
            continue
        query = network_selector[tag]
        if query["mode"] == "l2":
            additional_vlans.append({"id": str(net["vlan_id"])})
        if query["mode"] == "l3":
            additional_vlans.append(
                {
                    "id": str(net["vlan_id"]),
                    "includeRanges": [f"{network[10]}-{network[100]}"],
                    "cidr": str(network),
                }
            )

    network_spec = {
        "private": True,
        "vlanId": str(pxe_net["vlan_id"]),
        "loadBalancerHost": pxe_lb_host,
        "metallbRanges": [pxe_metallb_ranges],
        "cidr": str(pxe_network),
        "gateway": pxe_net["router_addr"],
        "includeRanges": [pxe_ipam_ranges],
        "dhcpRanges": [pxe_dhcp_range],
        "nameservers": res_env["nameservers"],
        "additionalVlans": additional_vlans,
    }
    dynamic_env["cluster_network"] = network_spec
    utils.merge_dicts(res_env, dynamic_env)

    LOG.info("Available releases: {}".format(kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()

    for template_class in ["publickeys", "credentials", "baremetalhostprofiles"]:
        render_templates_and_apply(template_class, res_env, kubectl)

    # TODO: wait for creds before apply cluster
    time.sleep(60)

    for template_class in ["clusters", "machines", "kaascephclusters"]:
        render_templates_and_apply(template_class, res_env, kubectl)

    # Validation stage
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    nm.set_lock_data_owner(cluster)

    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()

    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()
    cluster.check.check_k8s_nodes()
    cluster.check.check_overlay_encryption_functionality()

    # Collecting artifacts
    cluster.store_k8s_artifacts()
    cluster.provider_resources.save_artifact()
