import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, waiters

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KAAS_CHILD_CLUSTER_NAME}"])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_aws_child_k8s_lma(kaas_manager, show_step, _):
    """This test is designed for creating child cluster and prepare it
    for further openstack deployment. Openstack deployment requires
    ceph cluster and additional networks to be ready. Building on this,
    test creates common child cluster with 3 additional nodes (workers)
    for ceph cluster. Ceph config will be generated according to these
    ceph nodes and applyed. After ceph cluster is ready, some additional
    networks (2 networks) will be added to each of node with openstack
    controller or openstack compute labels. Based on these networks
    new netplans for these nodes will be generated, uploaded and applyed
    on every compute or control node.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating aws credential
        5. Configure bastion (optional)
        6. Create child cluster and machines with SL label
        7. Check cluster readiness
        8. Create additional volumes and Ceph cluster
        9. Pod checks (optional)
        10. Prepare additional networks for openstack deploy
        11. Check bastion VM (optional)

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info(f"Available releases: {kaas_manager.get_clusterrelease_names()}")
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    aws_region = settings.AWS_DEFAULT_REGION
    aws_az = settings.AWS_DEFAULT_AZ
    ami = settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME
    # Number of nodes
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    control_nodes_count = int(settings.KAAS_CHILD_CLUSTER_CONTROL_NODES_COUNT)
    computes_nodes_count = int(settings.KAAS_CHILD_CLUSTER_COMPUTE_NODES_COUNT)
    ceph_nodes_count = int(settings.KAAS_CHILD_CLUSTER_CEPH_NODES_COUNT)
    # Nodes flavors
    master_instance_type = settings.KAAS_CHILD_CLUSTER_MASTER_FLAVOR_NAME
    control_instance_type = settings.KAAS_CHILD_CLUSTER_CONTROL_MACHINE_FLAVOR_NAME
    compute_instance_type = settings.KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_FLAVOR_NAME
    ceph_machine_instance_type = settings.KAAS_CHILD_CLUSTER_CEPH_MACHINE_FLAVOR_NAME
    # Node disk sizes
    master_disk_size = settings.KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_SIZE
    control_disk_size = settings.KAAS_CHILD_CLUSTER_CONTROL_MACHINE_AWS_DISK_SIZE
    compute_disk_size = settings.KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_DISK_SIZE
    ceph_disk_size = settings.KAAS_CHILD_CLUSTER_CEPH_AWS_DISK_SIZE
    # Node disk types
    master_disk_type = settings.KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_TYPE
    control_disk_type = settings.KAAS_CHILD_CLUSTER_CONTROL_MACHINE_AWS_DISK_TYPE
    compute_disk_type = settings.KAAS_CHILD_CLUSTER_COMPUTE_MACHINE_DISK_TYPE
    ceph_disk_type = settings.KAAS_CHILD_CLUSTER_CEPH_AWS_DISK_TYPE
    lma_extra_options = yaml.safe_load(settings.KAAS_CHILD_LMA_EXTRA_OPTIONS) \
        if settings.KAAS_CHILD_LMA_EXTRA_OPTIONS else None
    rnd_string = utils.gen_random_string(6)
    region = kaas_manager.get_mgmt_cluster().region_name
    priv_subnet_cidr = "10.0.0.0/24"

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = f"{cluster_name}-{rnd_string}-key"
    LOG.info(f"Public key name - {public_key_name}")
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Get aws bootstrap credentials from management cluster
    aws_key_id, aws_secret_access_key = kaas_manager.get_aws_credential(
        "cloud-config", "default")

    # Creating aws credential
    show_step(4)
    credential_name = f"{cluster_name}-{rnd_string}-cred"
    LOG.info(f"Credential name - {credential_name}")
    ns.create_aws_credential(
        credential_name, aws_key_id, aws_secret_access_key, region)

    helm_releases = []
    for hrelease in settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES:
        helm_releases.append({"name": hrelease})

    show_step(5)
    if settings.KAAS_BASTION_CONFIG:
        bastion = settings.KAAS_AWS_BASTION_CONFIG
        LOG.info(f"Use custom bastion config:\n{yaml.dump(bastion)}")
    else:
        LOG.info("Deploy child cluster without additional bastion config")
        bastion = None

    show_step(6)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credential_name,
        provider="aws",
        region=region,
        aws_region=aws_region,
        aws_az=aws_az,
        aws_priv_subnet_cidr=priv_subnet_cidr,
        aws_pub_subnet_cidr="10.0.1.0/24",
        services_cidr="10.96.0.0/16",
        pods_cidr="192.168.0.0/16",
        nodes_cidr="10.10.10.0/24",
        dns=child_dns,
        extra_helm_releases=helm_releases,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        lma_extra_options=lma_extra_options,
        public_key_name=public_key_name,
        bastion=bastion)

    os_control_labels = [
        {'key': 'openstack-control-plane', 'value': 'enabled'},
        {'key': 'openstack-gateway', 'value': 'enabled'},
        {'key': 'openvswitch', 'value': 'enabled'}]

    os_computes_labels = [
        {'key': 'openstack-compute-node', 'value': 'enabled'},
        {'key': 'openvswitch', 'value': 'enabled'}]

    stacklight_labels = {'key': 'stacklight', 'value': 'enabled'}

    if settings.STACKLIGHT_ON_COMPUTE_NODES:
        os_computes_labels.append(stacklight_labels)
    else:
        os_control_labels.append(stacklight_labels)

    LOG.info("Creating master nodes")
    for _ in range(master_nodes_count):
        cluster.create_aws_machine(
            node_type="master",
            region=region,
            instance_type=master_instance_type,
            ami=ami,
            root_device_size=master_disk_size,
            root_device_type=master_disk_type)

    os_machine_names = []
    LOG.info("Creating OS control nodes")
    for _ in range(control_nodes_count):
        machine = cluster.create_aws_machine(
            node_type="node",
            region=region,
            instance_type=control_instance_type or master_instance_type,
            ami=ami,
            node_labels=os_control_labels,
            root_device_size=control_disk_size,
            root_device_type=control_disk_type)
        os_machine_names.append(machine.name)

    LOG.info("Creating OS compute nodes")
    for _ in range(computes_nodes_count):
        machine = cluster.create_aws_machine(
            node_type="node",
            region=region,
            instance_type=compute_instance_type or master_instance_type,
            ami=ami,
            node_labels=os_computes_labels,
            root_device_size=compute_disk_size,
            root_device_type=compute_disk_type)
        os_machine_names.append(machine.name)

    LOG.info("Creating nodes for ceph cluster")
    ceph_machine_names = []
    for _ in range(ceph_nodes_count):
        machine = cluster.create_aws_machine(
            node_type="node",
            region=region,
            instance_type=ceph_machine_instance_type or master_instance_type,
            ami=ami,
            root_device_size=ceph_disk_size,
            root_device_type=ceph_disk_type)
        ceph_machine_names.append(machine.name)

    show_step(7)
    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status(timeout=settings.CHECK_MACHINE_STATUS_TIMEOUT)
    cluster.check.check_cluster_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()
    cluster.check.check_k8s_nodes()

    provider_resources = cluster.provider_resources

    show_step(8)
    ceph_nodes_config_path = "si_tests/templates/ceph/ceph-aws-virtual-default.yaml"
    LOG.info(f"Using the following MiraCeph template file: {ceph_nodes_config_path}")
    with open(ceph_nodes_config_path) as f:
        ceph_cluster_conf = yaml.load(f, Loader=yaml.SafeLoader)
    ceph_cluster_conf['spec']['network']['clusterNet'] = priv_subnet_cidr
    ceph_cluster_conf['spec']['network']['publicNet'] = priv_subnet_cidr
    k8s_to_rook_topology_labels = {'topology.kubernetes.io/zone': 'zone',
                                   'topology.kubernetes.io/region': 'region'}
    for machine_name in ceph_machine_names:
        machine_obj = cluster.get_machine(machine_name)
        existed_disks = machine_obj.run_cmd(
            'lsblk -o NAME --json').stdout_yaml
        disks_names_before = [
            d['name'] for d in existed_disks['blockdevices']]
        ceph_nodes_map = {
            'name': 'machine_name',
            'devices': [{'name': '', 'config': {'deviceClass': 'ssd'}}],
            'roles': ['mon', 'mgr', 'mds']
        }
        node = machine_obj.get_k8s_node()
        node_name = node.get_name()
        volume = provider_resources.create_volume(
            name=f"ceph-volume-{node_name}",
            AvailabilityZone=aws_az,
            Size=50,
            VolumeType="gp3"
        )
        volume_id = volume["VolumeId"]

        LOG.info(f"Attaching volume {volume_id} to node {node_name}")
        provider_resources.attach_volume(
            Device="/dev/sdp", InstanceId=node_name, VolumeId=volume_id)

        LOG.info("Checking for new disk")
        disks_after_attach = machine_obj.run_cmd(
            'lsblk -o NAME --json').stdout_yaml
        disks_names_after_attach = [
            d['name'] for d in disks_after_attach['blockdevices']]
        new_disk = [d for d in disks_names_after_attach if
                    d not in disks_names_before]
        assert len(new_disk) > 0, ("No new disks were added")
        LOG.info(f"New disk name: {new_disk[0]}")
        ceph_nodes_map['name'] = node_name
        ceph_nodes_map['devices'][0]['name'] = new_disk[0]
        # WA to prevent rook from removing topology.kubernetes.io labels.
        node_labels = node.data.get('metadata', {}).get('labels', {})
        for k8s_label, rook_label in k8s_to_rook_topology_labels.items():
            if k8s_label in node_labels:
                if 'crush' not in ceph_nodes_map:
                    ceph_nodes_map['crush'] = {}
                ceph_nodes_map['crush'][rook_label] = node_labels[k8s_label]
        ceph_cluster_conf['spec']['nodes'].append(ceph_nodes_map)

    LOG.info("Applying ceph config to child cluster")
    cluster.k8sclient.miracephs.create(namespace="ceph-lcm-mirantis", body=ceph_cluster_conf)

    LOG.info("Waiting for Ceph cluster to be deployed")
    waiters.wait(lambda: cluster.is_ceph_deployed, timeout=1800)
    LOG.info("Ceph cluster deployed")

    LOG.info("Waiting for Ceph cluster HEALTH is OK")
    cluster.check.wait_rook_ceph_health_status()
    LOG.info("Ceph cluster HEALTH is OK")

    show_step(9)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=1600,
                                                 check_all_nss=True)

    show_step(10)
    LOG.info("Getting default security group")
    security_group = provider_resources.get_default_sg_for_cluster_vpc()
    security_group_id = security_group['GroupId']
    LOG.info(f"Found security group with id: {security_group_id}")

    new_subnets = {}
    for i in range(1, 3):
        subnet_name = f"kaas-net-{cluster_name}-{i}"
        subnet_cidr = f"10.0.1{i}.0/24"
        network = provider_resources.create_subnet(
            name=subnet_name,
            CidrBlock=subnet_cidr,
            AvailabilityZone=aws_az
        )
        subnet_id = network["Subnet"]["SubnetId"]
        new_subnets[subnet_name] = {"net_id": subnet_id}
        LOG.info(f"Created subnet {subnet_name} with id {subnet_id}")

    def _configure_netdev(machine, netdev_name, peer_name, filename):
        machine.run_cmd(
            f"sudo bash -c 'cat << EOF > /etc/systemd/network/{filename}\n"
            "[NetDev]\n"
            f"Name={netdev_name}\n"
            "Kind=veth\n"
            "[Peer]\n"
            f"Name={peer_name}\n"
            "EOF'")

    ip_prefix = 5
    for machine_name in os_machine_names:
        machine_obj = cluster.get_machine(machine_name)
        remote_netplan_path = "/etc/netplan/50-cloud-init.yaml"
        netplan = machine_obj.run_cmd(f"sudo cat {remote_netplan_path}").stdout_yaml
        existed_ifaces = netplan["network"]["ethernets"]
        netplan_map = {
            "network": {
                "ethernets": {"veth-vbmc-br": {},
                              "veth-vbmc": {},
                              "veth-bm-br": {},
                              "veth-bm": {},
                              "veth-br": {},
                              "veth-phy": {}},
                "bridges": {"br-public": {
                    "dhcp4": "false",
                    "interfaces": ["veth-br"],
                    "macaddress": "",
                    "addresses": []},
                    "br-baremetal": {
                    "dhcp4": "false",
                    "interfaces": ["veth-bm-br",
                                   "veth-vbmc-br"],
                    "macaddress": "",
                    "addresses": []}}}}

        for k, v in existed_ifaces.items():
            netplan_map["network"]["ethernets"][k] = v

        instance_id = machine_obj.get_k8s_node_name()
        LOG.info(f"Create and attach new network interfaces to node: {instance_id}")
        for i in range(1, 3):
            net_id = new_subnets[f"kaas-net-{cluster_name}-{i}"]["net_id"]
            if i == 1:
                ip = f"10.0.11.{ip_prefix}"
                bridge = "br-public"
                dev_index = 6
            else:
                ip = f"10.0.12.{ip_prefix}"
                bridge = "br-baremetal"
                dev_index = 7

            LOG.info(f"Creating network interface with ip: {ip}")
            network_interface = provider_resources.create_network_interface(
                Description=f"kaas-net-{cluster_name}-{i}",
                SubnetId=net_id,
                PrivateIpAddress=ip,
                Groups=[security_group_id]
            )
            network_interface_mac = network_interface["NetworkInterface"]["MacAddress"]
            network_interface_id = network_interface["NetworkInterface"]["NetworkInterfaceId"]

            LOG.info(f"Attaching network interface {network_interface_id} to instance {instance_id}")
            attach_response = provider_resources.attach_network_interface(
                DeviceIndex=dev_index,
                InstanceId=instance_id,
                NetworkInterfaceId=network_interface_id
            )
            attach_id = attach_response["AttachmentId"]
            # Enable DeleteOnTermination for network interface.
            # Otherwise we cannot delete cluster subnet because of network interface leftovers.
            provider_resources.modify_network_interface_attribute(NetworkInterfaceId=network_interface_id, Attachment={
                "AttachmentId": attach_id,
                "DeleteOnTermination": True
            })
            iface_hex = machine_obj.run_cmd(
                "lspci | grep Ethernet | tail -n 1 | "
                "cut -d ':' -f2 | cut -d '.' -f1").stdout_yaml
            iface_num = int(str(iface_hex), 16)
            new_iface = f"ens{iface_num}"

            LOG.info(f"Found new interface {new_iface} on node {instance_id} with mac {network_interface_mac}")
            netplan_map["network"]["ethernets"][new_iface] = \
                {"match": {"macaddress": network_interface_mac},
                 "dhcp4": True, "dhcp6": False, "set-name": new_iface}

            netplan_map["network"]["bridges"][bridge]["interfaces"].append(new_iface)
            netplan_map["network"]["bridges"][bridge]["addresses"].append(ip + "/24")
            netplan_map["network"]["bridges"][bridge]["macaddress"] = network_interface_mac
        LOG.info("Configure new NetDevs")
        machine_obj.run_cmd(f"sudo bash -c 'cat << EOF > {remote_netplan_path}\n"
                            f"{yaml.dump(netplan_map)}\n"
                            "EOF'")
        _configure_netdev(machine=machine_obj, netdev_name='veth-phy',
                          peer_name='veth-br', filename='10-veth-phy-br.netdev')

        _configure_netdev(machine=machine_obj, netdev_name='veth-bm',
                          peer_name='veth-bm-br', filename='11-veth-bm.netdev')

        _configure_netdev(machine=machine_obj, netdev_name='veth-vbmc',
                          peer_name='veth-vbmc-br', filename='12-veth-vbmc.netdev')
        LOG.info(f"Applying new netplan for node {instance_id}")
        machine_obj.run_cmd(f"sudo netplan apply {remote_netplan_path}")
        ip_prefix += 1

    cluster.check.check_cluster_readiness()

    # Collecting artifacts
    cluster.store_k8s_artifacts()
    cluster.provider_resources.save_artifact()

    show_step(11)
    if bastion:
        LOG.info(f"New bastion IP = {cluster.bastion_ip} for cluster {cluster.name}")
        LOG.info("Check ssh to KaaS machine vie bastion VM")
        machines = cluster.get_machines()
        target_machine = machines[0]
        cluster.check.check_connection_via_bastion(target_machine)
        LOG.info("Check bastion parameters VM in AWS")
        cluster_status = cluster.data.get('status') or {}
        instance_id = cluster_status.get('providerStatus', {}).get('bastion', {}).get('id')
        cluster.check.check_aws_vm(instance_id=instance_id,
                                   instance_type=bastion['instanceType'],
                                   ami_id=bastion['amiId'])
