data "template_file" "cloud_init_config" {
  template = file("${path.module}/cloudinit/cloud_init.yaml")
  vars     = {
    ssh_key      = var.guest_ssh_key_public
    ssh_username = var.guest_ssh_user
  }
}

resource "vsphere_virtual_machine" "vm" {
  count            = var.quantity
  name             = "${var.cluster_name}-${var.name_prefix}-${count.index}"
  resource_pool_id = var.resource_pool_id
  datastore_id     = var.datastore_id
  folder           = var.folder
  guest_id         = var.template_vm.guest_id
  num_cpus         = var.num_cpu
  memory           = var.ram

  network_interface {
    network_id = var.network_id
  }

  disk {
    label            = "${var.cluster_name}-${var.name_prefix}-${count.index}"
    size             = var.disk_size
    thin_provisioned = var.template_vm.disks.0.thin_provisioned
  }

  extra_config = {
    "disk.EnableUUID"             = "TRUE"
    "guestinfo.userdata"          = base64encode(data.template_file.cloud_init_config.rendered)
    "guestinfo.userdata.encoding" = "base64"
    "guestinfo.metadata"          = <<-EOT
         { "local-hostname": "${var.cluster_name}-${var.name_prefix}-${count.index}" }
      EOT
  }

  clone {
    template_uuid = var.template_vm.id
    customize {
      network_interface {
        ipv4_address = var.is_ipam_enabled ? var.ip_addresses["${count.index}"] : null
        ipv4_netmask = var.is_ipam_enabled ? 23 : null
      }
      ipv4_gateway    = var.is_ipam_enabled ? var.gateway : null
      dns_server_list = var.is_ipam_enabled ? split(",", var.dns) : null
      linux_options {
        host_name = "${var.cluster_name}-${var.name_prefix}-${count.index}"
        domain    = "${var.cluster_name}.internal"
      }
    }
  }
}
