import yaml

from si_tests.deployments.utils import (file_utils, wait_utils,
                                        commons, kubectl_utils, helpers)
from si_tests.deployments.utils.namespace import NAMESPACE


class OperatorLanguage:
    python = 1
    golang = 2


def deploy_ceph_cluster(os_manager, timeout):
    yaml_path = file_utils.get_example(
        "miraceph/ceph_local_drives_openstack.yaml")

    # Detect ceph operator language
    ceph_operator_yaml = helpers.get_ceph_operator_yaml()
    ceph_operator = file_utils.get_yaml_content(ceph_operator_yaml)
    images = ceph_operator["spec"]["releases"][0]["values"]["images"]

    operator_lang = OperatorLanguage.python
    if "ceph-controller" in images['cephController']['repository']:
        operator_lang = OperatorLanguage.golang

    commons.LOG.info("Change defaults in ceph config")
    ceph_config = file_utils.get_yaml_content(yaml_path)
    cluster_data = file_utils.get_yaml_content(commons.CLUSTER_INFO_YAML)
    nodes_data = file_utils.get_yaml_content(
        commons.NODES_METADATA_YAML)["ceph"]

    if operator_lang == OperatorLanguage.python:
        nodes = {}
        ceph_config["spec"]["clusterNet"] = cluster_data[
            "storage_backend_network_cidr"]
        ceph_config["spec"]["clientNet"] = cluster_data[
            "storage_frontend_network_cidr"]
    else:
        nodes = []
        ceph_config["spec"]["network"]["clusterNet"] = cluster_data[
            "storage_backend_network_cidr"]
        ceph_config["spec"]["network"]["publicNet"] = cluster_data[
            "storage_frontend_network_cidr"]

    num_nodes = len(nodes_data.keys())
    roles = [["mon", "mgr"], ["mon"]] + [[] * (num_nodes - 2)]
    for (name, node), role in zip(nodes_data.items(), roles):
        metadata = yaml.safe_load(commons.decode(node))

        if operator_lang == OperatorLanguage.python:
            nodes[name] = metadata
            nodes[name]["roles"] = role
        else:
            osd_devices = []
            for device in metadata["storageDevices"]:
                osd_devices.append(
                    {"name": device["name"],
                     "config": {"deviceClass": device["role"]}})
            nodes.append(
                {'name': name, 'devices': osd_devices, 'roles': role})

    ceph_config["spec"]["nodes"] = nodes

    new_yaml_path = file_utils.get_new_name(yaml_path)

    commons.LOG.info("Dump new ceph config to '%s'", new_yaml_path)
    file_utils.save_to_yaml(ceph_config, new_yaml_path)

    kubectl = kubectl_utils.Kubectl()
    commons.LOG.info("Apply new ceph-cluster config")
    kubectl.apply(new_yaml_path)

    wait = wait_utils.Waiter(os_manager, timeout, NAMESPACE.rook)
    commons.LOG.info("Wait for csi-rbdplugin-provisioner")
    wait.deployment("csi-rbdplugin-provisioner", replicas=2)

    commons.LOG.info("Wait for csi-rbdplugin daemonset")
    wait.daemonset("csi-rbdplugin")

    commons.LOG.info("Wait for ceph cluster to become healthy")
    wait.pods("rook-ceph-tool", replicas=1)

    ceph_pod = os_manager.api.pods.list_starts_with(
        "rook-ceph-tool", NAMESPACE.rook)[0]

    # TODO(opetrenko): wait for all ceph cluster elements
    #  (mons, man, rgw) are ready

    def _wait():
        assert "HEALTH_OK" == ceph_pod.exec(["ceph", "health"]).strip()
        return True

    wait.wait_pass(_wait, expected=AssertionError)

    commons.LOG.info("Ceph cluster successfully deployed")
