import uuid

from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


class VMProvider:
    def __init__(self, os_client, vm_name):
        self.os_client = os_client
        self.vm_name = vm_name

    def __enter__(self):
        network_name = 'si-tests-{}'.format(uuid.uuid4().hex)
        self.subnet_name = network_name + '-subnet'

        network = self.os_client.network.create([network_name])
        self.net_id = network['id']
        LOG.info("Created network %s", self.net_id)

        subnets = self.os_client.subnet.list([])
        subnets = list(
            filter(lambda x: x["Name"] == self.subnet_name, subnets))

        if not subnets:
            self.os_client.subnet.create([
                self.subnet_name, '--dhcp',
                '--subnet-range', '10.20.40.0/24',
                '--allocation-pool', 'start=10.20.40.10,end=10.20.40.254',
                '--gateway', '10.20.40.1',
                '--network', network_name])

        LOG.info("Create VM")
        res = self.os_client.server.create(
            [self.vm_name, "--flavor", "m1.tiny_test",
             "--image", 'Ubuntu-18.04',
             "--nic", 'net-id={}'.format(self.net_id)])

        assert isinstance(res, dict)

        LOG.info("VM created %s", res)

        def check():
            server = self.os_client.server.show([self.vm_name])
            return server['status'] == 'ACTIVE'
        LOG.info("Wait VM become ACTIVE")
        waiters.wait(check, timeout=300, interval=10)
        return res["id"]

    def __exit__(self, exc_type, exc_val, exc_tb):
        LOG.info("Cleaning up created resources for %s", self.vm_name)
        self.os_client.server.delete([self.vm_name])
        self.os_client.server.wait_absent(self.vm_name)
        self.os_client.subnet.delete([self.subnet_name])
        self.os_client.subnet.wait_absent(self.subnet_name)
        self.os_client.network.delete([self.net_id])
        self.os_client.network.wait_absent(self.net_id)
