#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
#
# This file is designed as common module for built-in kcm managers utilities. It provides function chunks for the
# tests and helps to avoid circular imports and code duplication in places where fixtures can not be used.
# Functions SHOULD require kcm_manager as

from si_tests import logger
from si_tests import settings

from si_tests.managers.kcm_manager import Manager

LOG = logger.logger


def determine_hosted_parameters(kcm_manager: "Manager"):
    """Determine hosted cluster parameters"""
    if settings.KSI_KCM_REGION_NAME:
        LOG.info("Region name provided, checking region readiness")
        region = kcm_manager.api.k0rdent_regions.get(name=settings.KSI_KCM_REGION_NAME)
        region.wait_for_readiness()
        host_cluster_name = region.clustername
        host_cluster_ns = settings.KCM_NAMESPACE
    else:
        LOG.info("No region name provided, using host cluster as management")
        host_cluster_name = settings.KSI_HOST_CLUSTER_NAME
        host_cluster_ns = settings.KSI_HOST_CLUSTER_NAMESPACE
        assert host_cluster_name and host_cluster_ns, ('KSI_HOST_CLUSTER_NAME and KSI_HOST_CLUSTER_NAMESPACE'
                                                       ' are required for this test')
        LOG.info(f"Initialise cluster {host_cluster_ns}/{host_cluster_name} as kcm manager instance")

    return host_cluster_ns, host_cluster_name


def determine_host_clusters(kcm_manager: "Manager", host_cluster_ns, host_cluster_name):
    """Determine kcm managers based on environment/region data"""
    host_ns = kcm_manager.get_or_create_namespace(host_cluster_ns)
    host_cluster = host_ns.get_cluster_deployment(name=host_cluster_name)
    _, auth_data = host_cluster.get_kubeconfig_from_secret()
    # Initialise host cluster as manager to have API for host/regional cluster
    host_kcm_manager = Manager(kubeconfig=None, auth_data=auth_data)
    # In case of region - cld will be created in the management cluster. In case of host-cluster (second kcm) - cld
    # will be created in the host cluster directly.
    cld_manager = kcm_manager if settings.KSI_KCM_REGION_NAME else host_kcm_manager

    return host_cluster, host_kcm_manager, cld_manager


def determine_regional_objects(kcm_manager: "Manager"):
    """Determine regional kcm manager and region object based on environment data"""
    regional_manager = None
    if settings.KSI_KCM_REGION_NAME:
        LOG.info("Region name provided, checking region readiness")
        region = kcm_manager.api.k0rdent_regions.get(name=settings.KSI_KCM_REGION_NAME)
        region.wait_for_readiness()
        host_cluster_name = region.clustername
        host_cluster_ns = settings.KCM_NAMESPACE
        host_ns = kcm_manager.get_or_create_namespace(host_cluster_ns)
        host_cluster = host_ns.get_cluster_deployment(name=host_cluster_name)
        _, auth_data = host_cluster.get_kubeconfig_from_secret()
        regional_manager = Manager(kubeconfig=None, auth_data=auth_data)

    return kcm_manager if not regional_manager else regional_manager
