from kubernetes.client.rest import ApiException

from si_tests import settings
from si_tests import logger

LOG = logger.logger


def test_apply_servicetemplate(kcm_manager, show_step):
    """Check that servicetemplate can be applied to cluster

    Scenario:
        1. Gather evnironment config
        2. Access target cluster and check that cluster is Ready and accessible
        3. Check that desired template is valid and available in namespace
        4. Patch cluster deployment to apply service templates
        5. Wait for cluster deployment became ready
        6. Check expected pods
    """

    show_step(1)
    svc_templates_list = settings.SERVICE_TEMPLATES_LIST
    assert svc_templates_list, "SERVICE_TEMPLATES_LIST env variable should not be empty or non-existent"
    LOG.info(f"Requested service templates: {svc_templates_list}")

    show_step(2)
    ns = kcm_manager.get_namespace(settings.TARGET_NAMESPACE)
    cld = ns.get_cluster_deployment(settings.TARGET_CLD)
    cld.check.check_actual_expected_pods()
    # Limit readiness check since we should have already deployed working cluster
    cld.check.check_cluster_readiness(timeout=600)
    # assert clusterdeployment.k8sclient.componentstatuses.describe_fixed_resources
    LOG.info(f"Clusterdeployment {cld.namespace}/{cld.name} checked and ready.")

    show_step(3)
    LOG.info('Verifying service templates')
    err = ''
    checked_tmpl = []
    for tmpl_name in svc_templates_list:
        try:
            tmpl = ns.get_service_template_by_name(tmpl_name)
            checked_tmpl.append(tmpl)
        except ApiException as ex:
            if ex.status != 404:
                err += f"Template with name {tmpl_name} does not exist in namespace {ns.name}\n"
            else:
                raise ex

    assert checked_tmpl, "Requested service templates not found at all"

    for template in checked_tmpl:
        if not template.is_valid:
            err += f"Template {template.namespace}/{template.name} is not valid\n"

    assert not err, f"Errors occured during service templates check process:\n {err}"
    LOG.info('Service templates verified')

    show_step(4)
    LOG.info(f"Applying service templates to cluster deployment {cld.namespace}/{cld.name}")
    svc_body = ns.build_service_templates_list(checked_tmpl)
    cld.apply_service_templates(svc_list=svc_body)

    show_step(5)
    cld.check.check_cluster_readiness(timeout=1800)

    show_step(6)
    cld.check.check_actual_expected_pods()

# TODO(add svctmpl removal test)
