import pytest
import requests

from si_tests import settings
from si_tests import logger
from si_tests.managers.harbor_manager import HarborManager

LOG = logger.logger


# Global variables to store test resources for dependency chain
test_resources = {
    'box_cluster_name': settings.KSI_INFERENCE_BOX_CLUSTER_NAME,
    'project_name': settings.KSI_INFERENCE_TEST_PROJECT_NAME,
    'nodegroup_name': settings.KSI_INFERENCE_TEST_NODEGROUP_NAME,
    'usergroup_name': settings.KSI_INFERENCE_TEST_USERGROUP_NAME,
    'user_name': settings.KSI_INFERENCE_TEST_USER_NAME,
    'registry_name': settings.KSI_INFERENCE_TEST_REGISTRY_NAME,
    'registry_user_name': settings.KSI_INFERENCE_TEST_REGISTRY_USER_NAME,
    'registry_storage_limit': settings.KSI_INFERENCE_TEST_REGISTRY_STORAGE_LIMIT,
    'pod_image_src_registry_url': settings.KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_URL,
    'pod_image_src_registry_user': settings.KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_USER or None,
    'pod_image_src_registry_password': settings.KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_PASSWORD or None,
    'pod_image_name': settings.KSI_INFERENCE_TEST_POD_IMAGE_NAME,
    'pod_image_tag': settings.KSI_INFERENCE_TEST_POD_IMAGE_TAG,
    'pod_image_replication_adapter_type': settings.KSI_INFERENCE_TEST_POD_IMAGE_REPLICATION_ADAPTER_TYPE,
    'volume_image_src_registry_url': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_URL,
    'volume_image_src_registry_user': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_USER or None,
    'volume_image_src_registry_password': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_PASSWORD or None,
    'volume_image_name': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_NAME,
    'volume_image_tag': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_TAG,
    'volume_image_replication_adapter_type': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_REPLICATION_ADAPTER_TYPE,
    'volume_image_mount_path': settings.KSI_INFERENCE_TEST_VOLUME_IMAGE_MOUNT_PATH,
    'volume_name': settings.KSI_INFERENCE_TEST_VOLUME_NAME,
    'apikey_name': settings.KSI_INFERENCE_TEST_APIKEY_NAME,
    'inference_flavor_name': settings.KSI_INFERENCE_TEST_FLAVOR_NAME,
    'inference_flavor_cpu': settings.KSI_INFERENCE_TEST_FLAVOR_CPU,
    'inference_flavor_memory': settings.KSI_INFERENCE_TEST_FLAVOR_MEMORY,
    'inference_flavor_gpu': settings.KSI_INFERENCE_TEST_FLAVOR_GPU or None,
    'inference_flavor_gpu_memory': settings.KSI_INFERENCE_TEST_FLAVOR_GPU_MEMORY or None,
    'inference_flavor_gpu_model': settings.KSI_INFERENCE_TEST_FLAVOR_GPU_MODEL or None,
    'inference_flavor_is_gpu_shared': settings.KSI_INFERENCE_TEST_FLAVOR_IS_GPU_SHARED,
    'inference_port': settings.KSI_INFERENCE_TEST_INFERENCE_PORT,
    'inference_name': settings.KSI_INFERENCE_TEST_INFERENCE_NAME,

    # Resources that get created during test execution
    'box_api_client': None,
    'inference_manager': None,
    'inference_user_manager': None,
    'usergroup': None,
    'user': None,
    'user_password': None,
    'apikey': None,
    'project': None,
    'pod_image_replicated': False,
    'volume_image_replicated': False,
    'nodegroup': None,
    'registry': None,
    'registry_user': None,
    'volume': None,
    'flavor': None,
    'inference': None,
    'inference_disabled': False,
    'addresses': None
}


@pytest.fixture(scope='session')
def setup_test_resources(box_api_client, inference_manager):
    """Setup basic test resources - step 1"""
    # Store the clients for reuse
    test_resources['box_api_client'] = box_api_client
    test_resources['inference_manager'] = inference_manager


@pytest.mark.usefixtures('setup_test_resources')
def test_step_1_show_available_resources(inference_manager):
    """Step 1: Show available resources from box-api"""
    LOG.banner("=== Step 1: Show available resources ===")
    # This is a simple check that the API works
    try:
        inference_manager.show_capacity()
        inference_manager.show_gpu_nodes()
        LOG.info("* Available resources shown successfully")
    except Exception as e:
        pytest.fail(f"Failed to show available resources: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_2_get_or_create_project(inference_manager):
    """Step 2: Get or create a Project"""
    LOG.banner("=== Step 2: Get or create a Project ===")
    try:
        project = inference_manager.get_or_create_project(test_resources['project_name'])
        test_resources['project'] = project
        LOG.info(f"* Project '{test_resources['project_name']}' created/retrieved successfully")
    except Exception as e:
        pytest.fail(f"Failed to get or create project: {str(e)}")
    inference_manager.show_projects()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_3_create_flavor(inference_manager):
    """Step 3: Create a Flavor that suits the dummy inference resource requirements (nginx)"""
    LOG.banner("=== Step 3: Create a Flavor ===")
    try:
        flavor = inference_manager.get_or_create_flavor(
            flavor_name=test_resources['inference_flavor_name'],
            cpu=test_resources['inference_flavor_cpu'],
            memory=test_resources['inference_flavor_memory'],
            gpu=test_resources['inference_flavor_gpu'],
            gpu_memory=test_resources['inference_flavor_gpu_memory'],
            gpu_model=test_resources['inference_flavor_gpu_model'],
            is_gpu_shared=test_resources['inference_flavor_is_gpu_shared']
        )
        test_resources['flavor'] = flavor
        LOG.info("* Flavor created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create flavor: {str(e)}")
    inference_manager.show_flavors()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_4_create_nodegroup(inference_manager):
    """Step 4: Create a NodeGroup for all available GPU nodes"""
    LOG.banner("=== Step 4: Create a NodeGroup ===")
    if not test_resources['project']:
        LOG.error("*** SKIP: Required: Project")
        pytest.skip("Required: Project")
    try:
        gpu_nodes = inference_manager.list_gpu_nodes()
        # NodeGroup nodes: { region1_name: [node1_1, node1_2, node1_3], region2_name: [node2_1, node2_2, node2_3], ...}
        nodegroup_nodes = {}
        for gpu_node in gpu_nodes:
            if gpu_node.region_name not in nodegroup_nodes:
                nodegroup_nodes[gpu_node.region_name] = []
            nodegroup_nodes[gpu_node.region_name].append(gpu_node.name)
        # custom_labels = {
        #     'aaa': 'bbb'
        # }
        nodegroup = inference_manager.get_or_create_nodegroup(
            nodegroup_name=test_resources['nodegroup_name'],
            project_names=[test_resources['project_name']],
            node_refs=nodegroup_nodes,
            # custom_labels=custom_labels
        )
        test_resources['nodegroup'] = nodegroup
        LOG.info("* NodeGroup created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create NodeGroup: {str(e)}")
    inference_manager.show_nodegroups()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_5_get_or_create_usergroup(inference_manager):
    """Step 5: Get or create a new UserGroup"""
    LOG.banner("=== Step 5: Get or create a new UserGroup ===")
    if not test_resources['project']:
        LOG.error("*** SKIP: Required: Project")
        pytest.skip("Required: Project")

    try:
        usergroup = inference_manager.get_or_create_usergroup(
            test_resources['usergroup_name'],
            test_resources['project_name']
        )
        test_resources['usergroup'] = usergroup
        LOG.info(f"* UserGroup '{test_resources['usergroup_name']}' created/retrieved successfully")
    except Exception as e:
        pytest.fail(f"Failed to get or create usergroup: {str(e)}")
    inference_manager.show_usergroups()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_6_create_user(inference_manager):
    """Step 6: Create a new Box User"""
    LOG.banner("=== Step 6: Create a new User ===")
    if not test_resources['usergroup']:
        LOG.error("*** SKIP: Required: UserGroup")
        pytest.skip("Required: UserGroup")

    try:
        user = inference_manager.get_or_create_user(
            test_resources['user_name'],
            [test_resources['usergroup_name']],
            test_resources['project_name']
        )
        test_resources['user'] = user
        test_resources['user_password'] = inference_manager.get_user_password(test_resources['user_name'])
        LOG.info(f"* User '{test_resources['user_name']}' created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create user: {str(e)}")
    inference_manager.show_users()
    LOG.info(f"===== USER PASSWORD: {test_resources['user_password']}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_7_init_new_box_api_client(box_api_client, inference_manager):
    """Step 7: Init new box-api client with User credentials"""
    LOG.banner("=== Step 7: Init new box-api client with User credentials ===")
    if not test_resources['user'] or not test_resources['user_password']:
        LOG.error("*** SKIP: Required: Box User credentials")
        pytest.skip("Required: Box User credentials")

    try:
        user_client = box_api_client.__class__(
            keycloak_url=box_api_client.keycloak_url,
            box_api_url=box_api_client.box_api_url,
            username=test_resources['user_name'],
            password=test_resources['user_password'],
            client_id=box_api_client.client_id,
            realm_name=box_api_client.realm_name,
            verify=box_api_client.verify
        )
        test_resources['inference_user_manager'] = inference_manager.__class__(box_api_client=user_client)
        LOG.info("* New box-api client initialized successfully for test User")
    except Exception as e:
        pytest.fail(f"Failed to initialize new box-api client for test User: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_8_create_apikey(inference_manager):
    """Step 8: Create an APIKey (from User)"""
    LOG.banner("=== Step 8: Create an APIKey ===")
    if not test_resources['inference_user_manager']:
        LOG.error("*** SKIP: Required: Box User manager")
        pytest.skip("Required: Box User manager")

    try:
        apikey = test_resources['inference_user_manager'].create_or_recreate_apikey(
            test_resources['apikey_name'],
            test_resources['project_name']
        )
        test_resources['apikey'] = apikey
        LOG.info(f"* APIKey '{test_resources['apikey_name']}' created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create APIKey: {str(e)}")
    test_resources['inference_user_manager'].show_apikeys(test_resources['project_name'])
    LOG.info(f"====== APIKEY '{test_resources['apikey_name']}' secret value: '{test_resources['apikey'].secret}'")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_9_create_registry(inference_manager):
    """Step 9: Create a new Registry (from User)"""
    LOG.banner("=== Step 9: Create a new Registry ===")
    if not test_resources['inference_user_manager']:
        LOG.error("*** SKIP: Required: Box User manager")
        pytest.skip("Required: Box User manager")

    try:
        registry = test_resources['inference_user_manager'].get_or_create_registry(
            test_resources['registry_name'],
            test_resources['project_name'],
            storage_limit=test_resources['registry_storage_limit'],
            public=False
        )
        test_resources['registry'] = registry
        LOG.info(f"* Registry '{test_resources['registry_name']}' created/retrieved successfully")
    except Exception as e:
        pytest.fail(f"Failed to create registry: {str(e)}")
    test_resources['inference_user_manager'].show_registries(test_resources['project_name'])
    # TODO(ddmitriev): in case of using custom CA for the registry, configure the runtime to use this CA
    # to avoid errors while pulling images like:
    #    "tls: failed to verify certificate: x509: certificate signed by unknown authority"


@pytest.mark.usefixtures('setup_test_resources')
def test_step_10_create_registry_user(inference_manager):
    """Step 10: Create a new RegistryUser (from User)"""
    LOG.banner("=== Step 10: Create a new RegistryUser ===")
    if not test_resources['registry']:
        LOG.error("*** SKIP: Required: Registry")
        pytest.skip("Required: Registry")

    try:
        registry_user = test_resources['inference_user_manager'].get_or_create_registry_users(
            test_resources['registry_user_name'],
            test_resources['registry_name'],
            test_resources['project_name']
        )
        test_resources['registry_user'] = registry_user
        LOG.info(f"* RegistryUser '{test_resources['registry_user_name']}' created/retrieved successfully")
    except Exception as e:
        pytest.fail(f"Failed to create registry user: {str(e)}")
    test_resources['inference_user_manager'].show_registry_users(
        test_resources['registry_name'], test_resources['project_name'])
    # registry_user_pull_secret_name = registry_user.pull_secret_name
    # registry_user_credentials = inference_user_manager.get_registry_user_credentials(
    #     registry_user_name, registry_name, project_name)


@pytest.mark.usefixtures('setup_test_resources')
def test_step_11_upload_pod_image_to_registry(inference_manager):
    """Step 11: Upload a pod image to the Registry"""
    LOG.banner("=== Step 11: Upload a pod image ===")
    if not test_resources['registry']:
        LOG.error("*** SKIP: Required: Registry")
        pytest.skip("Required: Registry")

    if inference_manager.is_registry_image_exists(test_resources['pod_image_name'],
                                                  test_resources['pod_image_tag'],
                                                  test_resources['registry_name'],
                                                  test_resources['project_name']):
        LOG.info(f"Image '{test_resources['pod_image_name']}:{test_resources['pod_image_tag']}'"
                 f" already exists in the Registry '{test_resources['registry_name']}'")
    else:
        LOG.info(f"Replicating Image '{test_resources['pod_image_name']}:{test_resources['pod_image_tag']}'"
                 f" to the Registry '{test_resources['registry_name']}' ...")
        try:
            # Use nginx as the image for testing
            harbor_url = settings.KSI_INFERENCE_BOX_HARBOR_URL
            harbor_admin_username = settings.KSI_INFERENCE_BOX_HARBOR_ADMIN_USERNAME
            harbor_admin_password = settings.KSI_INFERENCE_BOX_HARBOR_ADMIN_PASSWORD

            registry_host, registry_namespace = test_resources['registry'].url.split("/")

            # Use Harbor manager to replicate image
            hm = HarborManager(harbor_url, harbor_admin_username, harbor_admin_password, verify_tls=False)

            # Replicate the nginx image
            exec_id = hm.replicate_image(
                test_resources['pod_image_src_registry_url'],
                test_resources['pod_image_src_registry_user'],
                test_resources['pod_image_src_registry_password'],
                test_resources['pod_image_name'],
                test_resources['pod_image_tag'],
                registry_namespace,
                adapter_type=test_resources['pod_image_replication_adapter_type']
            )

            LOG.info(f"* Image is successfully uploaded to the registry, replication id: {exec_id}")
        except Exception as e:
            pytest.fail(f"Failed to upload image: {str(e)}")
    test_resources['pod_image_replicated'] = True
    inference_manager.show_registry_images(test_resources['registry_name'], test_resources['project_name'])


@pytest.mark.usefixtures('setup_test_resources')
def test_step_12_upload_volume_image_to_registry(inference_manager):
    """Step 12: Upload a volume image to the Registry"""
    LOG.banner("=== Step 12: Upload a volume image ===")
    if not test_resources['registry']:
        LOG.error("*** SKIP: Required: Registry")
        pytest.skip("Required: Registry")

    if inference_manager.is_registry_image_exists(test_resources['volume_image_name'],
                                                  test_resources['volume_image_tag'],
                                                  test_resources['registry_name'],
                                                  test_resources['project_name']):
        LOG.info(f"Image '{test_resources['volume_image_name']}:{test_resources['volume_image_tag']}'"
                 f" already exists in the Registry '{test_resources['registry_name']}'")
    else:
        LOG.info(f"Replicating Image '{test_resources['volume_image_name']}:{test_resources['volume_image_tag']}'"
                 f" to the Registry '{test_resources['registry_name']}' ...")
        try:
            # Use nginx as the image for testing
            harbor_url = settings.KSI_INFERENCE_BOX_HARBOR_URL
            harbor_admin_username = settings.KSI_INFERENCE_BOX_HARBOR_ADMIN_USERNAME
            harbor_admin_password = settings.KSI_INFERENCE_BOX_HARBOR_ADMIN_PASSWORD

            registry_host, registry_namespace = test_resources['registry'].url.split("/")

            # Use Harbor manager to replicate image
            hm = HarborManager(harbor_url, harbor_admin_username, harbor_admin_password, verify_tls=False)

            # Replicate the nginx image
            exec_id = hm.replicate_image(
                test_resources['volume_image_src_registry_url'],
                test_resources['volume_image_src_registry_user'],
                test_resources['volume_image_src_registry_password'],
                test_resources['volume_image_name'],
                test_resources['volume_image_tag'],
                registry_namespace,
                adapter_type=test_resources['volume_image_replication_adapter_type']
            )

            LOG.info(f"* Image is successfully uploaded to the registry, replication id: {exec_id}")
        except Exception as e:
            pytest.fail(f"Failed to upload image: {str(e)}")
    test_resources['volume_image_replicated'] = True
    inference_manager.show_registry_images(test_resources['registry_name'], test_resources['project_name'])


@pytest.mark.usefixtures('setup_test_resources')
def test_step_13_create_volume(inference_manager):
    """Step 13: Create a Volume from User using the uploaded image"""
    LOG.banner("=== Step 13: Create a Volume ===")
    if not test_resources['volume_image_replicated']:
        LOG.error("*** SKIP: Required: Replicated Volume image in the Registry")
        pytest.skip("Required: Replicated Volume image in the Registry")
    try:
        registry_host, registry_namespace = test_resources['registry'].url.split("/")
        volume_image_reference = (f"{registry_host}/{registry_namespace}/"
                                  f"{test_resources['volume_image_name']}:{test_resources['volume_image_tag']}")

        volume = test_resources['inference_user_manager'].get_or_create_volume_from_image(
            test_resources['volume_name'],
            test_resources['project_name'],
            [test_resources['box_cluster_name']],
            volume_image_reference
        )
        test_resources['volume'] = volume
        LOG.info("* Volume created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create volume: {str(e)}")
    test_resources['inference_user_manager'].show_volumes(test_resources['project_name'])


@pytest.mark.usefixtures('setup_test_resources')
def test_step_14_create_inference(inference_manager, kcm_manager):
    """Step 14: Create Inference object using the created User, Flavor, Volume, NodeGroup, ApiKey"""
    LOG.banner("=== Step 14: Create Inference ===")

    if not test_resources['project']:
        LOG.error("*** SKIP: Required: Project")
        pytest.skip("Required: Project")
    if not test_resources['flavor']:
        LOG.error("*** SKIP: Required: Flavor")
        pytest.skip("Required: Flavor")
    if not test_resources['inference_user_manager']:
        LOG.error("*** SKIP: Required: Box User manager")
        pytest.skip("Required: Box User manager")
    if not test_resources['apikey']:
        LOG.error("*** SKIP: Required: ApiKey")
        pytest.skip("Required: ApiKey")
    if not test_resources['registry']:
        LOG.error("*** SKIP: Required: Registry")
        pytest.skip("Required: Registry")
    if not test_resources['registry_user']:
        LOG.error("*** SKIP: Required: RegistryUser")
        pytest.skip("Required: RegistryUser")
    if not test_resources['pod_image_replicated']:
        LOG.error("*** SKIP: Required: Pod image is replicated")
        pytest.skip("Required: Pod image is replicated")
    if not test_resources['volume']:
        LOG.error("*** SKIP: Required: Volume")
        pytest.skip("Required: Volume")
    try:
        # Create inference using the volume and registry image
        inference = test_resources['inference_user_manager'].create_inference(
            project_name=test_resources['project_name'],
            inference_name=test_resources['inference_name'],
            flavor_name=test_resources['inference_flavor_name'],
            volume_name=test_resources['volume_name'],
            mount_path=test_resources['volume_image_mount_path'],
            registry_user_name=test_resources['registry_user_name'],
            registry_name=test_resources['registry_name'],
            registry_image_name=test_resources['pod_image_name'],
            registry_image_tag=test_resources['pod_image_tag'],
            inference_port=test_resources['inference_port'],
            inference_regions=[test_resources['box_cluster_name']],
            api_keys=[test_resources['apikey_name']]
        )
        test_resources['inference'] = inference
        LOG.info("* Inference created successfully")

        LOG.info("=== Wait for Inference to be ready ===")
        ns = kcm_manager.get_namespace(settings.TARGET_NAMESPACE)
        cld = ns.get_cluster_deployment(settings.TARGET_CLD)
        test_resources['inference_user_manager'].wait_inferences_ready(
            project_name=test_resources['project_name'],
            inference_names=[test_resources['inference_name']],
            k8sclient=cld.k8sclient
        )
        addresses = test_resources['inference_user_manager'].get_inference_addresses(
            project_name=test_resources['project_name'],
            inference_name=test_resources['inference_name']
        )
        test_resources['addresses'] = addresses
        LOG.info("* Inference is ready")
    except Exception as e:
        pytest.fail(f"Failed to create inference: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_15_validate_apikey_access(inference_manager):
    """
    Step 15: Check that APIKey is working while accessing the Inference endpoint,
    and the endpoint is restricted without the APIKey
    """
    LOG.banner("=== Step 15: Validate APIKey access ===")
    if not test_resources['addresses']:
        LOG.error("*** SKIP: Required: Inference is up and running")
        pytest.skip("Required: Inference is up and running")

    # Try to access the Inference external address using the ApiKey for authorization
    apikey_secret = test_resources['apikey'].secret

    try:
        addresses = test_resources['addresses']

        # Try to access the Inference external address using the ApiKey for authorization
        for region, address in addresses.items():
            LOG.info(f"Testing access to address '{address}' in region '{region}' with API key...")
            response = requests.get(address, headers={'X-API-Key': apikey_secret}, verify=False)

            # Validate that access with valid API key returns 200 (success)
            if response.status_code != 200:
                pytest.fail(f"Expected 200 when accessing with valid API key, but got {response.status_code}")

            LOG.info(f"* Access with API key successful: Status code {response.status_code}")
            LOG.info(f"Response content:\n{response.text}")

        # Try to access the same address without ApiKey and ensure access is forbidden
        for region, address in addresses.items():
            LOG.info(f"Testing access to address '{address}' in region '{region}' without API key...")
            response = requests.get(address, verify=False)

            # Validate that access without API key returns 401 or 403 (forbidden/unauthorized)
            if response.status_code not in [401, 403]:
                pytest.fail(f"Expected 401 or 403 when accessing without API key, but got {response.status_code}")

            LOG.info(f"* Access without API key returned expected status code {response.status_code}")

    except Exception as e:
        pytest.fail(f"Failed to validate APIKey access: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_16_disable_inference(inference_manager):
    """Step 16: Disable the Inference and check that it's endpoint is not available"""
    LOG.banner("=== Step 16: Disable Inference ===")
    if not test_resources['addresses']:
        LOG.error("*** SKIP: Required: Inference is up and running")
        pytest.skip("Required: Inference is up and running")

    try:
        test_resources['inference_user_manager'].disable_inference(
            project_name=test_resources['project_name'],
            inference_name=test_resources['inference_name'])

        test_resources['inference_disabled'] = True
        LOG.info("* Inference is disabled")
    except Exception as e:
        pytest.fail(f"Failed to disable inference: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_17_enable_inference(inference_manager):
    """Step 17: Enable the Inference and check that it's endpoint is available again, but using APIKey only"""
    LOG.banner("=== Step 17: Enable Inference ===")
    if not test_resources['inference_disabled']:
        LOG.error("*** SKIP: Required: Inference is created and then disabled")
        pytest.skip("Required: Inference is created and then disabled")

    apikey_secret = test_resources['apikey'].secret
    try:
        test_resources['inference_user_manager'].enable_inference(
            project_name=test_resources['project_name'],
            inference_name=test_resources['inference_name'])

        addresses = test_resources['addresses']
        for region, address in addresses.items():
            LOG.info(f"Testing access to address '{address}' in region '{region}' with API key...")
            response = requests.get(address, headers={'X-API-Key': apikey_secret}, verify=False)
            # Validate that access with valid API key returns 200 (success)
            if response.status_code != 200:
                pytest.fail(f"Expected 200 when accessing with valid API key, but got {response.status_code}")
            LOG.info(f"* Access with API key successful: Status code {response.status_code}")
            LOG.info(f"Response content:\n{response.text}")

        test_resources['inference_disabled'] = False
        LOG.info("* Inference enabled")
    except Exception as e:
        pytest.fail(f"Failed to enable inference: {str(e)}")
