import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, k8s_utils, kcm_utils

LOG = logger.logger


@pytest.mark.create_cd
@pytest.mark.os
@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures('store_cluster_kubeconfig')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KCM_CLUSTER_DEPLOYMENT_NAME}"])
def test_create_os_hosted_standalone_cluster(kcm_manager, show_step, _):
    """Deploy hosted clusterdeployment on OpenStack

    Scenario:
        1. Collect env data and determine regional/host cluster
        2. Gather host cluster based on environment/region data
        3. Generate unique names and auth parameters
        4. Gather network config of the host cluster for hosted deployment
        5. Create or get NS and check cluster templates presence
        6. Create auth
        7. Prepare clusterdeployment config
        8. Create clusterdeployment
        9. Wait for clusterdeployment progress to finish
        10. Check expected pods, its statuses and k8s nodes
        11. Set default storageclass if not present
        12. Sync CoreDNS Corefile with management cluster
    """
    show_step(1)
    # TODO(va4st): Move to separated test with auth actions, determine auth parameters dynamically
    cluster_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    namespace_name = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE
    host_cluster_ns, host_cluster_name = kcm_utils.determine_hosted_parameters(kcm_manager)

    show_step(2)
    host_cluster, host_kcm_manager, cld_manager = kcm_utils.determine_host_clusters(kcm_manager,
                                                                                    host_cluster_ns,
                                                                                    host_cluster_name)

    show_step(3)
    auth_suffix = f"ksi-openstack-auth-{utils.gen_random_string(4)}"
    # https://github.com/kubernetes/cloud-provider-openstack/blob/master/charts/
    # cinder-csi-plugin/templates/storageclass.yaml#L5
    # https://github.com/k0rdent/kcm/blob/main/templates/cluster/openstack-standalone-cp/
    # templates/k0scontrolplane.yaml#L149
    storageclass = 'csi-cinder-sc-delete'
    auth_secret_name = f"{auth_suffix}-secret"
    cluster_identity_name = f"{auth_suffix}-identity"
    cred_name = f"{auth_suffix}-cred"

    show_step(4)
    # Gathering network config of the host cluster for hosted deployment using mgmt cluster
    LOG.info(f"Gathering network config of the host cluster {host_cluster_ns}/{host_cluster_name} ")
    infracluster = host_cluster.clusterobject.infracluster
    network_name = infracluster.data['status']['network']['name']
    # The first (0) subnet will be used. https://docs.k0rdent.io/latest/admin/hosted-control-plane/hcp-openstack/
    subnet_name = infracluster.data['status']['network']['subnets'][0]['name']
    router_name = infracluster.data['status']['router']['name']

    show_step(5)
    ns = cld_manager.get_or_create_namespace(namespace_name)
    host_cluster_ns = host_kcm_manager.get_or_create_namespace(host_cluster_ns)

    clustertemplate = (settings.KCM_CLUSTER_TEMPLATE_NAME or
                       ns.get_latest_template_by_prefix('openstack-hosted', strategy='chart'))
    available_templates = [tmpl.name for tmpl in ns.get_cluster_templates()]
    assert clustertemplate in available_templates, (f"Clustertemplate {clustertemplate} not found in namespace "
                                                    f"{ns.name}. Available templates: {available_templates}")

    show_step(6)
    auth_sec = ns.create_os_auth_secret(name=auth_secret_name)
    # NOTE(va4st): For OpenStack there no separated auth. Only secret required
    if kcm_manager.condition.oss_1_4_0_and_above() and settings.KSI_OS_IDENTITY_AUTH:
        cluster_identity = ns.create_os_cluster_identity(cluster_identity_name, auth_sec)
        cred_object = cluster_identity
    else:
        cred_object = auth_sec
    cred = ns.create_credential(name=cred_name, cred_object=cred_object)

    # Sveltos template should be created in the host/region cluster ns directly. It will not be
    # provisioned automatically like credentials.
    sveltos_template = host_cluster_ns.create_sveltos_resource_template_cm_using_secret(auth_sec, provider='openstack')
    assert sveltos_template.read(), "sveltos resource template not created"

    show_step(7)
    # TODO(va4st): make configurable through settings and add parameters verification before testrun
    config = ns.get_cluster_config(provider='openstack-hosted',
                                   auth_entity=cred_object,
                                   network_name=network_name,
                                   router_name=router_name,
                                   subnet_name=subnet_name)

    child_labels = cld_manager.kof.get_child_labels()
    config['clusterAnnotations'] = cld_manager.kof.get_child_annotations()

    show_step(8)
    cld = ns.create_cluster_deployment(name=cluster_name,
                                       template=clustertemplate,
                                       credential=cred,
                                       config=config,
                                       labels=child_labels)
    LOG.info(f"Created clusterdeployment {cld.namespace}/{cld.name}")

    show_step(9)
    cld.check.check_cluster_readiness(expected_condition_cld_fails={'Paused': ''})

    show_step(10)
    cld.check.check_k8s_nodes()
    cld.check.check_k8s_pods()
    cld.check.check_actual_expected_pods()

    show_step(11)
    if not cld.is_default_storageclass_present():
        sc = cld.get_storageclass_by_name(storageclass)
        sc.make_default()
        assert sc.is_default, f"StorageClass {storageclass} is not default"

    show_step(12)
    # TODO(va4st): Need a better way to determine the needs for CoreDNS sync (e.g some marker in mgmt config maybe?)
    if (not cld_manager.kof.kof_is_deployed() and cld_manager.kof.kof_ms_deployed()
            and cld_manager.source == 'custom-enterprise'):
        corefile = k8s_utils.get_corefile(cld_manager.api)
        k8s_utils.override_coredns_configmap(cld.k8sclient, corefile)
        coredns_deployment = cld.k8sclient.deployments.get(name='coredns', namespace='kube-system')
        coredns_deployment.rollout_restart()
