import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, k8s_utils

LOG = logger.logger


@pytest.mark.create_cd
@pytest.mark.vsphere
@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures('store_cluster_kubeconfig')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KCM_CLUSTER_DEPLOYMENT_NAME}"])
def test_create_vsphere_regional_cluster(kcm_bootstrap_manager, show_step, _):
    """Deploy managed clusterdeployment on vsphere

    Scenario:
        1. Collect env data
        2. Create or get NS and check cluster templates presence
        3. Create auth
        4. Prepare clusterdeployment config
        5. Create clusterdeployment
        6. Wait for clusterdeployment progress to finish
        7. Check expected pods, its statuses and k8s nodes
        8. Configure kof-collectors and DNS (optional)
    """
    show_step(1)
    kcm_manager = kcm_bootstrap_manager.check.kcm_mgr
    # TODO(va4st): Move to separated test with auth actions, determine auth parameters dynamically
    cluster_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    namespace_name = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE
    vsphere_cluster_id_name = f"ksi-vsphere-cluster-id-{utils.gen_random_string(4)}"
    auth_secret_name = f"{vsphere_cluster_id_name}-secret"
    cred_name = f"{vsphere_cluster_id_name}-cred"
    domain = f"{cluster_name}.kof.example.com"
    dns_mark = ['k0rdent.mirantis.com/kof-vsphere-dns-secrets', 'k0rdent.mirantis.com/kof-azure-dns-secrets']
    region = settings.KSI_REGION_NAME

    show_step(2)
    ns = kcm_manager.get_or_create_namespace(namespace_name)
    clustertemplate = (settings.KCM_CLUSTER_TEMPLATE_NAME or
                       ns.get_latest_template_by_prefix('vsphere-standalone', strategy='chart'))
    available_templates = [tmpl.name for tmpl in ns.get_cluster_templates()]
    assert clustertemplate in available_templates, (f"Clustertemplate {clustertemplate} not found in namespace "
                                                    f"{ns.name}. Available templates: {available_templates}")

    show_step(3)
    auth_sec = ns.create_vsphere_auth_secret(name=auth_secret_name)
    vsphere_cluster_id = ns.create_vsphere_cluster_identity(vsphere_cluster_id_name, auth_sec)
    cred = ns.create_credential(name=cred_name, cred_object=vsphere_cluster_id)
    ns.create_sveltos_resource_template_cm(vsphere_cluster_id)

    show_step(4)
    # TODO(va4st): make configurable through settings and add parameters verification before testrun
    config = ns.get_cluster_config(provider='vsphere')
    LOG.info(f"Initial cluster config collected: \n{yaml.dump(config)}")
    # Get regional config
    regional_conf = kcm_manager.kof.get_regional_config(region=region)
    LOG.info(f"Initial regional config collected: \n{yaml.dump(regional_conf)}")

    cld_annotations = None
    regional_labels = kcm_manager.kof.get_regional_labels()
    annotations = kcm_manager.kof.get_regional_annotations(domain=domain)
    if annotations:
        utils.merge_dicts(regional_conf, annotations)
    LOG.info(f"Final regional config collected: \n{yaml.dump(regional_conf)}")
    utils.merge_dicts(config, regional_conf)
    LOG.info(f"Resulted cluster config built: \n{yaml.dump(config)}")
    if not any(dns in regional_labels.keys() for dns in dns_mark) or not kcm_manager.kof.kof_is_deployed():
        # Only in case if auto-dns or istio not used
        cld_annotations = {
            'k0rdent.mirantis.com/kof-http-config':
                '{"dial_timeout": "10s", "tls_config": {"insecure_skip_verify": true}}'
        }

    show_step(5)
    cld = ns.create_cluster_deployment(name=cluster_name,
                                       template=clustertemplate,
                                       credential=cred,
                                       config=config,
                                       labels=regional_labels,
                                       annotations=cld_annotations)
    LOG.info(f"Created clusterdeployment {cld.namespace}/{cld.name}")

    show_step(6)
    cld.check.check_cluster_readiness()

    show_step(7)
    cld.check.check_actual_expected_pods()
    cld.check.check_k8s_pods()
    cld.check.check_k8s_nodes()

    show_step(8)
    if settings.KSI_KOF_MGMT_STORAGE == 'regional':
        if any(dns in regional_labels.keys() for dns in dns_mark):
            values = kcm_manager.kof.get_kof_collectors_values(mesh='other', value=domain)
            # Just create the collectors, dns passed by outer service
        elif kcm_manager.kof.kof_is_deployed():
            values = kcm_manager.kof.get_kof_collectors_values(mesh='istio', value=cluster_name)
            # Just create the collectors, mesh provided by istio
        else:
            values = kcm_manager.kof.get_kof_collectors_values(mesh='other', value=domain, insecure=True)
            # Patch coredns on mgmt
            corefile = k8s_utils.get_corefile(kcm_manager.api)
            core_editor = k8s_utils.CorefileEditor(corefile)
            ingress = cld.k8sclient.services.get(namespace='kof', name='ingress-nginx-controller')
            ingress.wait_for_external_ip()
            addr = ingress.get_external_addr()
            entries = [
                (addr, f"grafana.{domain}"),
                (addr, f"jaeger.{domain}"),
                (addr, f"vmauth.{domain}")
            ]
            core_editor.insert_hosts_entries(entries)
            k8s_utils.override_coredns_configmap(kcm_manager.api, core_editor.corefile)
            coredns_deployment = kcm_manager.api.deployments.get(name='coredns', namespace='kube-system')
            coredns_deployment.rollout_restart()

        kcm_manager.kof.install_kof_collectors(values=values, source=kcm_manager.source)
        kcm_bootstrap_manager.check.wait_kof_collectors_ready()
        kcm_manager.refresh_expected_objects()
        kcm_bootstrap_manager.check.check_actual_expected_pods()
