#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.


from importlib import resources
import os
from si_tests import settings_func
import time

# ====
# START: Main parameters
# ====

ARTIFACTS_DIR = settings_func.get_var_force_dir(
    "ARTIFACTS_DIR", os.path.join(os.getcwd(), "artifacts/")
)

SI_TESTS_REPO_ROOT = settings_func.get_var("SI_TESTS_REPO_ROOT", "./")
SI_BINARIES_DIR = settings_func.get_var(
    "SI_BINARIES_DIR", settings_func.get_var("SI_BINARIES_DIR", "bin/")
)

LOGS_DIR = settings_func.get_var("LOGS_DIR", ARTIFACTS_DIR)
LOG_NAME = settings_func.get_var("LOG_NAME", "tests.log")
LOG_REST_DEBUG = settings_func.get_var("LOG_REST_DEBUG", True)
ENV_NAME = settings_func.get_var("ENV_NAME")
# DEPRECATED: please for clearly new tests and init logic - use KSI_KUBECONFIG_PATH
KUBECONFIG_PATH = settings_func.get_var("KUBECONFIG", None)
KSI_KUBECONFIG_PATH = settings_func.get_var("KSI_KUBECONFIG_PATH", None)
KSI_CONFIG_PATH = settings_func.get_var(
    "KSI_CONFIG", os.path.join(ARTIFACTS_DIR, "ksi_config.yaml")
)
# Region name for regional API interaction. Will be used in any test which requires/creates/deletes region
KSI_KCM_REGION_NAME = settings_func.get_var("KSI_KCM_REGION_NAME", None)

# It's only possible to deploy in kcm-system ns. To deploy in other ns - need to configure access management object
KCM_NAMESPACE = settings_func.get_var("KCM_NAMESPACE", "kcm-system")
SVELTOS_NAMESPACE = settings_func.get_var("SVELTOS_NAMESPACE", "projectsveltos")
# The namespace created by sveltos for mgmt cluster management
KSI_SVELTOS_MGMT_NAMESPACE = settings_func.get_var("KSI_SVELTOS_MGMT_NAMESPACE", "mgmt")

KSI_SUPPORT_BUNDLE_VERSION = settings_func.get_var(
    "KSI_SUPPORT_BUNDLE_VERSION", "0.117.0"
)
KSI_SUPPORT_BUNDLE_TEMPLATE_PATH = settings_func.get_path(
    "KSI_SUPPORT_BUNDLE_TEMPLATE_PATH",
    "templates/deploy-objects/support-bundle/support-bundle.yaml.tpl",
)

KCM_CLUSTER_DEPLOYMENT_NAME = settings_func.get_var("KCM_CLUSTER_DEPLOYMENT_NAME", "")

# Name and namespace of the cluster which will be used as a host for hosted deployments
KSI_HOST_CLUSTER_NAME = settings_func.get_var("KSI_HOST_CLUSTER_NAME", "")
KSI_HOST_CLUSTER_NAMESPACE = settings_func.get_var("KSI_HOST_CLUSTER_NAMESPACE", "")

KCM_CLUSTER_DEPLOYMENT_NAMESPACE = settings_func.get_var(
    "KCM_CLUSTER_DEPLOYMENT_NAMESPACE", KCM_NAMESPACE
)
KCM_CLUSTER_TEMPLATE_NAME = settings_func.get_var("KCM_CLUSTER_TEMPLATE_NAME", "")

# Target parameters for tests on existing clusters
TARGET_NAMESPACE = settings_func.get_var("TARGET_NAMESPACE", "kcm-system")
TARGET_CLD = settings_func.get_var("TARGET_CLD", KCM_CLUSTER_DEPLOYMENT_NAME)
# comma separated list of clusters
# format "namespace1/cluster1,namespace2/cluster2"
TARGET_CLDS = settings_func.get_var("TARGET_CLUSTERS", "").split(",")

# Custom helm repository parameters where extra ClusterTemplates and ServiceTemplates are stored
CUSTOM_HELM_REPOSITORY_TEMPLATE_PATH = settings_func.get_path(
    "CUSTOM_HELM_REPOSITORY_TEMPLATE_PATH",
    "templates/deploy-objects/repositories/helmrepository.yaml",
)
CUSTOM_HELM_REPOSITORY_SECRET_TEMPLATE_PATH = settings_func.get_path(
    "CUSTOM_HELM_REPOSITORY_SECRET_TEMPLATE_PATH",
    "templates/deploy-objects/repositories/helmrepository-secret.yaml",
)

# Will delete the repository, and it's secret if present, and create new ones
CUSTOM_HELM_REPOSITORY_OVERWRITE = settings_func.get_var_as_bool(
    "CUSTOM_HELM_REPOSITORY_OVERWRITE", False
)

# Parameters to create ServiceTemplate in the test si_tests/tests/lcm/servicetemplates/test_servicetemplate.py
CUSTOM_SERVICETEMPLATE_PATH = settings_func.get_path(
    "CUSTOM_SERVICETEMPLATE_PATH",
    "templates/deploy-objects/servicetemplates/default/servicetemplate.yaml",
)

# Parameters to create ClusterTemplate in the test si_tests/tests/clusterdeployments/test_add_clustertemplate.py
# Example ClusterTemplate yaml: 'templates/deploy-objects/clustertemplates/default/clustertemplate.yaml'
CUSTOM_CLUSTERTEMPLATE_PATH = settings_func.get_path("CUSTOM_CLUSTERTEMPLATE_PATH", "")
# If CUSTOM_SERVICETEMPLATE_PATH is empty, then the KCM_SERVICE_TEMPLATE_NAME will be used
# to deploy an existing ServiceTemplate from the KCM_CLUSTER_DEPLOYMENT_NAMESPACE namespace
KCM_SERVICE_TEMPLATE_NAME = settings_func.get_var("KCM_SERVICE_TEMPLATE_NAME", "")

# If specified, then will be used to create ClusterDeployment object with pre-defined 'services' in cld create_ tests
CUSTOM_CLUSTERDEPLOYMENT_PATH = settings_func.get_var(
    "CUSTOM_CLUSTERDEPLOYMENT_PATH", ""
)
# If specified, then will be used to apply extra services to the existing ClusterDeployment object
CUSTOM_CLUSTERDEPLOYMENT_SERVICES_PATH = settings_func.get_var(
    "CUSTOM_CLUSTERDEPLOYMENT_SERVICES_PATH", ""
)

# Path with template to create a MultiClusterService object
CUSTOM_MULTI_CLUSTER_SERVICES_PATH = settings_func.get_var(
    "CUSTOM_MULTI_CLUSTER_SERVICES_PATH", ""
)

CUSTOM_SERVICETEMPLATE_VALUES_PATH = settings_func.get_var(
    "CUSTOM_SERVICETEMPLATE_VALUES_PATH", ""
)
# Projectsveltos related name/namespace which is used on the target cluster
CUSTOM_CLUSTERSERVICE_NAME = settings_func.get_var("CUSTOM_CLUSTERSERVICE_NAME", "")
CUSTOM_CLUSTERSERVICE_NAMESPACE = settings_func.get_var(
    "CUSTOM_CLUSTERSERVICE_NAMESPACE", ""
)

CHECK_CLUSTER_READINESS_TIMEOUT = int(
    settings_func.get_var("CHECK_CLUSTER_READINESS_TIMEOUT", 7200)
)

# Check cap* controller managers for errors during check_cluster_readiness()
ENABLE_INTROSPECT_CAP_ERRORS = settings_func.get_var_as_bool(
    "ENABLE_INTROSPECT_CAP_ERRORS", True
)

# Comma-separated list of service templates
SERVICE_TEMPLATES_LIST = settings_func.get_var("SERVICE_TEMPLATES_LIST", "")
if SERVICE_TEMPLATES_LIST:
    SERVICE_TEMPLATES_LIST = SERVICE_TEMPLATES_LIST.split(",")

# Skip PVC cleanup parameter
KSI_CLEANUP_CHILD_PVCS = settings_func.get_var_as_bool(
    "KSI_CLEANUP_CHILD_PVCS", True
)

# Expected pods checker parameters
SKIP_EXPECTED_POD_CHECK = settings_func.get_var_as_bool(
    "SKIP_EXPECTED_POD_CHECK", False
)
CHECK_ACTUAL_EXPECTED_PODS_TIMEOUT = int(
    settings_func.get_var("CHECK_ACTUAL_EXPECTED_PODS_TIMEOUT", 10 * 60)
)
WAIT_PODS_READY_TIMEOUT = int(settings_func.get_var("WAIT_PODS_READY_TIMEOUT", 1800))

KCM_MANAGEMENT_NAME = settings_func.get_var("KCM_MANAGEMENT_NAME", "kcm")
KCM_CHART_VERSION = settings_func.get_var("KCM_CHART_VERSION", "0.1.0")
KCM_CHART_NAME = settings_func.get_var("KCM_CHART_NAME", "kcm")
KSI_KCM_TEMPLATES_REPO_NAME = settings_func.get_var("KSI_KCM_TEMPLATES_REPO_NAME", "kcm-templates")
KSI_KCM_CHART_INSTALL_TIMEOUT = settings_func.get_var("KSI_KCM_CHART_INSTALL_TIMEOUT", "10m")
KSI_ENTERPRISE_NAME = settings_func.get_var("KSI_ENTERPRISE_NAME", "k0rdent-enterprise")
KSI_HELM_INSECURE = settings_func.get_var_as_bool("KSI_HELM_INSECURE", False)

# Customisation of kcm installation
# KCM_SOURCE - can be 'opensource'/'enterprise'/'custom-enterprise'/<other>.
# KCM_SOURCE='opensource' and KCM_SOURCE=None results to the
# same behavior - using opensource repos and repo switcher depended on version. (release/ci)
# KCM_SOURCE='custom-enterprise' requires KCM_CUSTOM_REGISTRY to be set
# KCM_SOURCE='enterprise' targets to enterprise repo.
KCM_SOURCE = settings_func.get_var("KCM_SOURCE", "opensource")
# Will be used both for oci:// and image pulling. e.g if you have 'registry.local' in this param - charts will
# be downloaded from oci://registry.local/charts, k0rdent-enterprise will be installed
# from oci://registry.local/charts/k0rdent-enterprise, images pulled with `registry.local` as a base.
# proper setting: KCM_CUSTOM_REGISTRY = 172.19.124.46:5000. without oci proto.
# url oci://172.19.124.46:5000/charts should be present in repo and will be used in constructions of
# images/charts repos
KCM_CUSTOM_REGISTRY = settings_func.get_var("KCM_CUSTOM_REGISTRY", None)
# Set to true to support HTTP proto for kcm registries. This is value for the kcm controller.
KSI_KCM_CUSTOM_INSECURE_REGISTRY = settings_func.get_var_as_bool("KSI_KCM_CUSTOM_INSECURE_REGISTRY", False)
# Registry auth
KSI_KCM_CUSTOM_REGISTRY_USERNAME = settings_func.get_var("KSI_KCM_CUSTOM_REGISTRY_USERNAME", None)
KSI_KCM_CUSTOM_REGISTRY_PASSWORD = settings_func.get_var("KSI_KCM_CUSTOM_REGISTRY_PASSWORD", None)
# Set to True to override values for enterprise KCM chart installation
KSI_OVERRIDE_ENTERPRISE_VALUES = settings_func.get_var("KSI_OVERRIDE_ENTERPRISE_VALUES", False)
# Timeout for HelmRelease instantiation, e.g "15m"
KSI_KCM_HELM_RELEASE_TIMEOUT = settings_func.get_var("KSI_KCM_HELM_RELEASE_TIMEOUT", None)
# Use self-signed registry certs
KSI_CUSTOM_REGISTRY_CERT_TEXT = settings_func.get_var("KSI_CUSTOM_REGISTRY_CERT_TEXT", None)
KSI_CUSTOM_REGISTRY_CERT_PATH = settings_func.get_path("KSI_CUSTOM_REGISTRY_CERT_PATH", None)

KCM_REPO_DICT = {
    "oss-repo": "ghcr.io/k0rdent/kcm",
    "enterprise-repo": "registry.mirantis.com/k0rdent-enterprise",
}
KSI_KCM_CTL_NAME_DICT = {
    "oss": "controller",
    "oss-ci": "controller-ci",
    "enterprise": "kcm-controller"
}
KSI_KCM_TELEMETRY_NAME_DICT = {
    "oss": "telemetry",
    "oss-ci": "telemetry-ci",
    "enterprise": "kcm-telemetry"
}
# Used in custom repo or airgapped deployments.
# Binary should be available by URL {KSI_K0S_URL}/k0s-v1.32.5+k0s.1-amd64.
# if KSI_K0S_URL = http://172.19.124.46:8080 - http://172.19.124.46:8080/k0s-v1.32.5+k0s.1-amd64 should be available
KSI_K0S_URL = settings_func.get_var("KSI_K0S_URL", None)

# kof variables and customisation
KSI_KOF_NAMESPACE = settings_func.get_var("KSI_KOF_NAMESPACE", "kof")
KSI_KOF_ISTIO_NAMESPACE = settings_func.get_var("KSI_KOF_ISTIO_NAMESPACE", "istio-system")
KSI_KOF_OP_CHART_NAME = settings_func.get_var("KSI_KOF_OP_CHART_NAME", "kof-operators")
KSI_KOF_MS_CHART_NAME = settings_func.get_var("KSI_KOF_MS_CHART_NAME", "kof-mothership")
KSI_KOF_RE_CHART_NAME = settings_func.get_var("KSI_KOF_RE_CHART_NAME", "kof-regional")
KSI_KOF_CH_CHART_NAME = settings_func.get_var("KSI_KOF_CH_CHART_NAME", "kof-child")
KSI_KOF_IS_CHART_NAME = settings_func.get_var("KSI_KOF_IS_CHART_NAME", "kof-istio")
KSI_KOF_CO_CHART_NAME = settings_func.get_var("KSI_KOF_CO_CHART_NAME", "kof-collectors")
KSI_KOF_ST_CHART_NAME = settings_func.get_var("KSI_KOF_ST_CHART_NAME", "kof-storage")
# This should have same version for all kof charts.
KSI_KOF_CHART_VERSION = settings_func.get_var("KSI_KOF_CHART_VERSION", "1.1.0")
KSI_KOF_SOURCE = settings_func.get_var("KSI_KOF_SOURCE", KCM_SOURCE)
KSI_KOF_CUSTOM_REGISTRY = settings_func.get_var("KSI_KOF_CUSTOM_REGISTRY", KCM_CUSTOM_REGISTRY)
# Where to store mgmt metrics. Possible choises - 'management' and 'regional'.
# When 'regional' is set - right after regional deployment (in the regional test)
# the collectors will be installed on mgmt and configured to regional.
# 'management' will require different test execution since the storage chart should be installed also
KSI_KOF_MGMT_STORAGE = settings_func.get_var("KSI_KOF_MGMT_STORAGE", 'management')

KSI_KOF_REPO_DICT = {
    "oss-repo": "ghcr.io/k0rdent/kof",
    "enterprise-repo": "registry.mirantis.com/k0rdent-enterprise",
}

# https://github.com/kubernetes-sigs/cluster-api/blob/main/api/v1beta1/machine_types.go#L597
CAPI_OPERATOR_APIVERSION = settings_func.get_var("CAPI_OPERATOR_APIVERSION", "v1beta2")
REMOVE_RESOURCES_ON_KCM_CLEANUP = settings_func.get_var_as_bool(
    "REMOVE_RESOURCES_ON_KCM_CLEANUP", True
)
KSI_NESTED_CLUSTER_CHECK_ENABLED = settings_func.get_var_as_bool("KSI_NESTED_CLUSTER_CHECK_ENABLED", True)

KSI_REGION_NAME = settings_func.get_var("KSI_REGION_NAME", 'region-one')
KSI_KOF_EMAIL = settings_func.get_var("KSI_KOF_EMAIL", 'the_mighty_qa@mirantis.com')

# Backup related settings
KSI_BACKUP_NAME = settings_func.get_var("KSI_BACKUP_NAME", None)

KSI_VELERO_PLUGIN_PROVIDER_NAME = settings_func.get_var("KSI_VELERO_PLUGIN_PROVIDER_NAME", "aws")
KSI_VELERO_PLUGIN_PROVIDER_TAG = settings_func.get_var("KSI_VELERO_PLUGIN_PROVIDER_TAG", "v1.13.0")
KSI_VELERO_RESTORE_NAME = settings_func.get_var("KSI_VELERO_RESTORE_NAME", f"ksi-restore-{int(time.time())}")
KSI_VELERO_BACKUP_NAME = settings_func.get_var("KSI_VELERO_BACKUP_NAME", None)
KSI_VELERO_BACKUP_STORAGE_REGION = settings_func.get_var("KSI_VELERO_BACKUP_STORAGE_REGION", 'us-east-1')
KSI_VELERO_BACKUP_STORAGE_BUCKET = settings_func.get_var("KSI_VELERO_BACKUP_STORAGE_BUCKET", None)


# ====
# END: Main parameters
# ====

# ====
# START: Template path parameters
# ====

MACHINE_PRIVELEGED_POD_YAML = settings_func.get_path(
    "MACHINE_PRIVELEGED_POD_YAML", "templates/pods/machine_priveleged_pod.yaml"
)

EXPECTED_PODS_TEMPLATES_DIR = (
    settings_func.get_path("EXPECTED_PODS_TEMPLATES_DIR", "templates/expected-pods")
    + "/"
)

# ====
# END: Template path parameters
# ====

# ====
# START: e2e Conformance parameters
# ====

K8S_CONFORMANCE_POD_YAML = settings_func.get_path(
    "K8S_CONFORMANCE_POD_YAML", "templates/pods/k8s_conformance_test_pod.yaml"
)

K8S_CONFORMANCE_NAMESPACE = settings_func.get_var(
    "K8S_CONFORMANCE_NAMESPACE", "conformance"
)
K8S_CONFORMANCE_POD_NAME = settings_func.get_var(
    "K8S_CONFORMANCE_POD_NAME", "conformance"
)
K8S_CONFORMANCE_REPORTS_DIR = settings_func.get_var(
    "K8S_CONFORMANCE_REPORTS_DIR", "/report"
)

K8S_CONFORMANCE_RESULTS_WAIT_TIMEOUT = int(
    settings_func.get_var("K8S_CONFORMANCE_RESULTS_WAIT_TIMEOUT", 5400)
)

K8S_CONFORMANCE_IMAGE = settings_func.get_var(
    "K8S_CONFORMANCE_IMAGE", "lcm/kubernetes/k8s-conformance"
)

# TODO: remove when PRODX-4679 is Done
K8S_CONFORMANCE_IMAGE_VERSION_CHECK_PATH = settings_func.get_var(
    "K8S_CONFORMANCE_IMAGE_VERSION_CHECK_PATH",
    "https://docker-dev-kaas-virtual.docker.mirantis.net/artifactory/docker-dev-kaas-virtual/"
    + K8S_CONFORMANCE_IMAGE,
)

K8S_CONFORMANCE_IMAGE_VERSION = settings_func.get_var(
    "K8S_CONFORMANCE_IMAGE_VERSION", ""
)
K8S_CONFORMANCE_CLUSTER_DOMAIN = settings_func.get_var(
    "K8S_CONFORMANCE_CLUSTER_DOMAIN", ""
)
K8S_CONFORMANCE_USER_DEFINED_SKIP_REGEX = settings_func.get_var(
    "K8S_CONFORMANCE_USER_DEFINED_SKIP_REGEX", ""
)
K8S_CONFORMANCE_CHECK_RESULTS = settings_func.get_var_as_bool(
    "K8S_CONFORMANCE_CHECK_RESULTS", True
)

K8S_CONFORMANCE_NON_BLOCKING_TAINTS = settings_func.get_var_as_bool(
    "K8S_CONFORMANCE_NON_BLOCKING_TAINTS", False
)

K8S_CONFORMANCE_DISABLE_OFFLINE_LOGIC = settings_func.get_var_as_bool(
    "K8S_CONFORMANCE_DISABLE_OFFLINE_LOGIC", False
)

# ====
# END: e2e Conformance parameters
# ====

# ====
# START: Provider specific configs
# ====

KCM_CLUSTER_CONTROLPLANE_NUMBER = int(
    settings_func.get_var("KCM_CLUSTER_CONTROLPLANE_NUMBER", 1)
)
KCM_CLUSTER_WORKERS_NUMBER = int(settings_func.get_var("KCM_CLUSTER_WORKERS_NUMBER", 1))
# In the most cases KSI_K0S_ARCH defaults to amd64. Keep it None to use defaults from the templates
KSI_K0S_ARCH = settings_func.get_var("KSI_K0S_ARCH", None)
# Keep it None to use defaults from the templates
KSI_K0S_VERSION = settings_func.get_var("KSI_K0S_VERSION", None)

# START: AWS specific
AWS_PROVIDER_NAME = "aws"
AWS_SECRET_ACCESS_KEY = settings_func.get_var("AWS_SECRET_ACCESS_KEY", None)
AWS_ACCESS_KEY_ID = settings_func.get_var("AWS_ACCESS_KEY_ID", None)
AWS_DEFAULT_REGION = settings_func.get_var("AWS_DEFAULT_REGION", "us-east-2")
AWS_INSTANCE_TYPE_CTL = settings_func.get_var("AWS_INSTANCE_TYPE_CTL", "t3.small")
AWS_INSTANCE_TYPE_WRK = settings_func.get_var("AWS_INSTANCE_TYPE_WRK", "t3.small")
AWS_INSTANCE_AMI_CTL = settings_func.get_var("AWS_INSTANCE_AMI_CTL", "")
AWS_INSTANCE_AMI_WRK = settings_func.get_var("AWS_INSTANCE_AMI_WRK", "")
AWS_INSTANCE_ROOT_SIZE_CTL = int(settings_func.get_var("AWS_INSTANCE_ROOT_SIZE_CTL", 0))
AWS_INSTANCE_ROOT_SIZE_WRK = int(settings_func.get_var("AWS_INSTANCE_ROOT_SIZE_WRK", 0))
# Will be used for both control and worker nodes along with EKS clusters
AWS_SSH_KEY_NAME = settings_func.get_var("AWS_SSH_KEY_NAME", "")

# START: AWS custom hybrid specific
KSI_AWS_ARM_WORKERS_NUMBER = int(settings_func.get_var("KSI_AWS_ARM_WORKERS_NUMBER", 0))
# no default value to not fill parameter if not needed. However, t4g.small is suitable for arm64 (2vCPU/2GB RAM)
KSI_AWS_INSTANCE_TYPE_ARM_WRK = settings_func.get_var("KSI_AWS_INSTANCE_TYPE_ARM_WRK", None)
# END: AWS custom hybrid specific

# END: AWS specific

# START: EKS specific
EKS_PROVIDER_NAME = "eks"
# END: AWS specific

# START: Azure specific
AZURE_PROVIDER_NAME = "azure"
AZURE_SUBSCRIPTION_ID = settings_func.get_var("AZURE_SUBSCRIPTION_ID", None)
AZURE_TENANT_ID = settings_func.get_var("AZURE_TENANT_ID", None)
AZURE_CLIENT_ID = settings_func.get_var("AZURE_CLIENT_ID", None)
AZURE_CLIENT_SECRET = settings_func.get_var("AZURE_CLIENT_SECRET", None)
AZURE_CLUSTER_LOCATION = settings_func.get_var("AZURE_CLUSTER_LOCATION", "westus2")
AZURE_VM_SIZE_CTL = settings_func.get_var("AZURE_VM_SIZE_CTL", "Standard_A4_v2")
AZURE_VM_SIZE_WRK = settings_func.get_var("AZURE_VM_SIZE_WRK", "Standard_A4_v2")
# 0 treated as false - will use default size for the template
# https://github.com/k0rdent/kcm/blob/main/templates/cluster/azure-standalone-cp/values.yaml#L30
KSI_AZURE_ROOT_VOL_SIZE_CTL = int(settings_func.get_var("KSI_AZURE_ROOT_VOL_SIZE_CTL", 0))
KSI_AZURE_ROOT_VOL_SIZE_WRK = int(settings_func.get_var("KSI_AZURE_ROOT_VOL_SIZE_WRK", 0))
KSI_AZURE_SSH_PUBKEY = settings_func.get_var("KSI_AZURE_SSH_PUBKEY", None)
KSI_AZURE_STORAGECLASS_YAML = settings_func.get_path(
    "KSI_AZURE_STORAGECLASS_YAML",
    "templates/deploy-objects/storageclasses/azurediskcsi.yaml",
)
# END: Azure specific


# This auth way slightly differs from usual.
# To acquire app-creds navigate to User > Identification > Application access
# START: Openstack specific
OPENSTACK_PROVIDER_NAME = "openstack"
OS_AUTH_URL = settings_func.get_var(
    "OS_AUTH_URL", "https://keystone.ic-eu.ssl.mirantis.net/"
)
OS_APPLICATION_CREDENTIAL_ID = settings_func.get_var(
    "OS_APPLICATION_CREDENTIAL_ID", None
)
OS_APPLICATION_CREDENTIAL_SECRET = settings_func.get_var(
    "OS_APPLICATION_CREDENTIAL_SECRET", None
)
OS_REGION_NAME = settings_func.get_var("OS_REGION_NAME", "RegionOne")
OS_INTERFACE = settings_func.get_var("OS_INTERFACE", "public")
OS_IDENTITY_API_VERSION = settings_func.get_var("OS_IDENTITY_API_VERSION", "3")
OS_AUTH_TYPE = settings_func.get_var("OS_AUTH_TYPE", "v3applicationcredential")
# cfg.dev - standard flavor with 4c/8g/20gb
OS_CTL_MACHINE_FLAVOR = settings_func.get_var("OS_CTL_MACHINE_FLAVOR", "cfg.dev")
# k8s_contrail.cid - standard flavor with 2c/4g/20gb
OS_WRK_MACHINE_FLAVOR = settings_func.get_var("OS_WRK_MACHINE_FLAVOR", "k8s_contrail.cid")
OS_INSTANCE_IMAGE_NAME = settings_func.get_var("OS_INSTANCE_IMAGE_NAME", "ubuntu-24.04-x86_64")

KSI_OS_SECURITY_GROUP = settings_func.get_var("KSI_OS_SECURITY_GROUP", None)
KSI_OS_KEYPAIR_NAME = settings_func.get_var("KSI_OS_KEYPAIR_NAME", None)

# Suitable only for 1.1.1 enterprise version. Enable or not the creation of OpenStackClusterIdentity object
KSI_OS_IDENTITY_AUTH = settings_func.get_var("KSI_OS_IDENTITY_AUTH", False)
# END: Openstack specific


# START: VSphere specific
VSPHERE_PROVIDER_NAME = "vsphere"
VSPHERE_ENABLED = settings_func.get_var_as_bool("VSPHERE_ENABLED", False)
VSPHERE_SERVER_ADDR = settings_func.get_var("VSPHERE_SERVER_ADDR", None)
VSPHERE_DATACENTER_NAME = settings_func.get_var("VSPHERE_DATACENTER_NAME", None)
VSPHERE_DATASTORE_PATH = settings_func.get_var("VSPHERE_DATASTORE_PATH", None)
VSPHERE_USER = settings_func.get_var("VSPHERE_USER", None)
VSPHERE_PASSWORD = settings_func.get_var("VSPHERE_PASSWORD", None)
VSPHERE_NETWORK_PATH = settings_func.get_var(
    "VSPHERE_NETWORK_PATH",
)
VSPHERE_VM_TEMPLATE_PATH = settings_func.get_var("VSPHERE_VM_TEMPLATE_PATH", None)
VSPHERE_RESOURCE_POOL_PATH = settings_func.get_var("VSPHERE_RESOURCE_POOL_PATH", None)
VSPHERE_FOLDER_PATH = settings_func.get_var("VSPHERE_FOLDER_PATH", None)
VSPHERE_SERVER_INSECURE = settings_func.get_var("VSPHERE_SERVER_INSECURE", True)
# TODO(va4st): Add mech for determining the LB addr based on provider networks
VSPHERE_NETWORK_CIDR = settings_func.get_var("VSPHERE_NETWORK_CIDR", None)
VSPHERE_CIDR_INCLUDE_RANGES = settings_func.get_var("VSPHERE_CIDR_INCLUDE_RANGES", None)
VSPHERE_CIDR_EXCLUDE_RANGES = settings_func.get_var("VSPHERE_CIDR_EXCLUDE_RANGES", None)
# TODO(va4st): Use CIDR, excluded ranges and allocated IPs via vclient to get proper static IP
VSPHERE_CONTROL_PLANE_ENDPOINT_IP = settings_func.get_var(
    "VSPHERE_CONTROL_PLANE_ENDPOINT_IP", None
)
VSPHERE_MACHINE_CPU = int(settings_func.get_var("VSPHERE_MACHINE_CPU", 2))
VSPHERE_MACHINE_RAM = int(settings_func.get_var("VSPHERE_MACHINE_RAM", 2048))
VSPHERE_ROOT_VOLUME_SIZE = int(settings_func.get_var("VSPHERE_ROOT_VOLUME_SIZE", 20))
# Using this for both specs - control and worker
VSPHERE_SSH_USER = settings_func.get_var("VSPHERE_SSH_USER", "ubuntu")
# In RAW string.
VSPHERE_SSH_PUBKEY = settings_func.get_var("VSPHERE_SSH_PUBKEY", "ubuntu")
# END: VSphere specific

# START: GCP specific
GCP_PROVIDER_NAME = "gcp"
GCP_AUTH_TYPE = settings_func.get_var("GCP_AUTH_TYPE", "service_account")
GCP_PROJECT_ID = settings_func.get_var("GCP_PROJECT_ID", None)
GCP_PRIVATE_KEY_ID = settings_func.get_var("GCP_PRIVATE_KEY_ID", None)
GCP_PRIVATE_KEY = settings_func.get_var("GCP_PRIVATE_KEY", None)
GCP_CLIENT_EMAIL = settings_func.get_var("GCP_CLIENT_EMAIL", None)
GCP_CLIENT_ID = settings_func.get_var("GCP_CLIENT_ID", None)
GCP_AUTH_URI = settings_func.get_var("GCP_AUTH_URI", "https://accounts.google.com/o/oauth2/auth")
GCP_TOKEN_URI = settings_func.get_var("GCP_TOKEN_URI", "https://oauth2.googleapis.com/token")
GCP_AUTH_PROVIDER_X509_CERT_URI = settings_func.get_var(
    "GCP_AUTH_PROVIDER_X509_CERT_URI", "https://www.googleapis.com/oauth2/v1/certs"
)
GCP_CLIENT_X509_CERT_URI = settings_func.get_var("GCP_CLIENT_X509_CERT_URI", None)
GCP_UNIVERSE_DOMAIN = settings_func.get_var("GCP_UNIVERSE_DOMAIN", "googleapis.com")

GCP_REGION = settings_func.get_var("GCP_REGION", "europe-west4")
GCP_NETWORK_NAME = settings_func.get_var("GCP_NETWORK_NAME", None)
GCP_CP_MACHINE_TYPE = settings_func.get_var("GCP_CP_MACHINE_TYPE", "n2d-standard-2")
GCP_WORKER_MACHINE_TYPE = settings_func.get_var(
    "GCP_WORKER_MACHINE_TYPE", GCP_CP_MACHINE_TYPE
)
GCP_CP_IMAGE = settings_func.get_var(
    "GCP_CP_IMAGE",
    "projects/ubuntu-os-cloud/global/images/ubuntu-2404-noble-amd64-v20250409",
)
GCP_WRK_IMAGE = settings_func.get_var("GCP_WRK_IMAGE", GCP_CP_IMAGE)
# 0 treated as false - will use default size for the template
# https://github.com/k0rdent/kcm/blob/main/templates/cluster/gcp-standalone-cp/values.yaml#L38
KSI_GCP_CP_ROOT_VOLUME_SIZE = int(settings_func.get_var("KSI_GCP_CP_ROOT_VOLUME_SIZE", 0))
KSI_GCP_WRK_ROOT_VOLUME_SIZE = int(settings_func.get_var("KSI_GCP_WRK_ROOT_VOLUME_SIZE", 0))
KSI_GCP_CP_ROOT_VOLUME_TYPE = settings_func.get_var("KSI_GCP_CP_ROOT_VOLUME_TYPE", "")
KSI_GCP_WRK_ROOT_VOLUME_TYPE = settings_func.get_var("KSI_GCP_WRK_ROOT_VOLUME_TYPE", "")

# Can be unexpectedly changed by GCP - they removing old versions easily, so hardcoded defaults in the template can
# cause non-working cluster.
KSI_GKE_K8S_VERSION = settings_func.get_var("KSI_GKE_K8S_VERSION", None)
# Can break expected pods check process due unexpected nodes count in different time frames
KSI_GKE_AUTOSCALING = settings_func.get_var_as_bool("KSI_GKE_AUTOSCALING", True)
# None - means use the template default
# https://github.com/k0rdent/kcm/blob/main/templates/cluster/gcp-gke/values.yaml#L22
KSI_GKE_RELEASE_CHANNEL = settings_func.get_var("KSI_GKE_RELEASE_CHANNEL", None)
# NOTE(va4st): Dumb stub for gke metrics server version naming pattern. There no data about naming outside
# of the cluster. This can be discovered only experimentally.
KSI_GKE_METRICS_SERVER_NAMING_PATTERN = {
    '1.31': 'metrics-server-v1.31.0',
    '1.32': 'metrics-server-v1.32.2',
    '1.33': 'metrics-server-v1.33.0',
}
# END: GCP specific

# START: REMOTE SSH specific
KSI_REMOTE_PROVIDER_NAME = "remotessh"
KSI_REMOTE_KEY_PATH = settings_func.get_path("KSI_REMOTE_KEY_PATH", None)
KSI_REMOTE_MACHINES_ADDRESSES = settings_func.get_var("KSI_REMOTE_MACHINES_ADDRESSES", '').split(",")
KSI_REMOTE_MACHINES_SSH_USER = settings_func.get_var("KSI_REMOTE_MACHINES_SSH_USER", "ubuntu")
# END: REMOTE SSH specfic

# START: env-on-openstack backend specific
# NOTE(alexz-kh): to be used on ksi internally only, DON'T provide\mix it with OS_ params for provider logic!
KSI_OS_CLOUD_NAME = settings_func.get_var("KSI_OS_CLOUD_NAME", "NONE_KSI_OS_CLOUD_NAME")
KSI_OS_APP_SECRET = settings_func.get_var("KSI_OS_APP_SECRET", "NONE_KSI_OS_APP_SECRET")
KSI_OS_CLOUDS_YAML_PATH = settings_func.get_var(
    "KSI_OS_CLOUDS_YAML_PATH",
    str(resources.files(__package__) / "test_env_data/ic_access/clouds.yaml"),
)
# do we need to delete=>create heat stack?
KSI_KEEP_ENV_BEFORE = settings_func.get_var_as_bool("KSI_KEEP_ENV_BEFORE", True)
# env_config , under test_env_data//labs/{}/
KSI_ENV_CONFIG_NAME = settings_func.get_var("KSI_ENV_CONFIG_NAME", "")

KSI_HEAT_DATA_PATH = settings_func.get_var(
    "KSI_HEAT_DATA_PATH_",
    str(
        resources.files(__package__)
        / "test_env_data/labs/{}/heat_data/".format(KSI_ENV_CONFIG_NAME or ENV_NAME)
    ),
)

KSI_BOOTSTRAP_DATA_PATH = settings_func.get_var(
    "KSI_BOOTSTRAP_DATA_PATH",
    str(
        resources.files(__package__)
        / "test_env_data/labs/{}/bootstrap_data/main.yaml.j2".format(
            KSI_ENV_CONFIG_NAME or ENV_NAME
        )
    ),
)
#  seed node IP, might be overwritten or fetched from ksi_config.yaml
KSI_SEED_IP = settings_func.get_var("KSI_SEED_IP", None)

# this url will be attached to openstack-heat stack(as for now), useful to trace-back orphaned envs
KSI_INFRA_BUILD_URL = settings_func.get_var("KSI_INFRA_BUILD_URL", "")

# infra key, to access for underlay openstack nodes
KSI_INFRA_OS_KEYPAIR_NAME = settings_func.get_var("KSI_INFRA_OS_KEYPAIR_NAME", "")
# START: k0s bootstrap related

KSI_BOOTSTRAP_K0S_VERSION = settings_func.get_var("KSI_BOOTSTRAP_K0S_VERSION", "v1.32.3+k0s.0")
KSI_K0S_APPLY_REGISTRY_HACK = settings_func.get_var_as_bool(
    "KSI_K0S_APPLY_REGISTRY_HACK", False
)
# r\o access to registry.ci.mirantis.com
KSI_SECRET_REGISTRY_CI_MIRANTIS_COM_USERNAME = settings_func.get_var(
    "KSI_SECRET_REGISTRY_CI_MIRANTIS_COM_USERNAME", None
)
KSI_SECRET_REGISTRY_CI_MIRANTIS_COM_PASSWORD = settings_func.get_var(
    "KSI_SECRET_REGISTRY_CI_MIRANTIS_COM_PASSWORD", None
)

# END: k0s bootstrap related

# HW access list
KSI_SECRET_BMC_CZ5570 = settings_func.get_var("KSI_SECRET_BMC_CZ5570", None)
KSI_SECRET_BMC_CZ5571 = settings_func.get_var("KSI_SECRET_BMC_CZ5571", None)
KSI_SECRET_BMC_CZ8204 = settings_func.get_var("KSI_SECRET_BMC_CZ8204", None)
KSI_SECRET_BMC_CZ5573 = settings_func.get_var("KSI_SECRET_BMC_CZ5573", None)
KSI_SECRET_BMC_CZ5572 = settings_func.get_var("KSI_SECRET_BMC_CZ5572", None)
# END: HW access list

# START: metal3 specific
KSI_METAL3_PROVIDER_NAME = "metal3"

KSI_BAREMETAL_CAPI_PROVIDER_METAL3_CHART_VERSION = settings_func.get_var(
    "KSI_BAREMETAL_CAPI_PROVIDER_METAL3_CHART_VERSION", None
)
KSI_BAREMETAL_CAPI_PROVIDER_METAL3_CHART_NAME = 'cluster-api-provider-metal3'

KSI_BAREMETAL_CAPI_PROVIDER_METAL3_SOURCE = settings_func.get_var(
    "KSI_BAREMETAL_CAPI_PROVIDER_METAL3_SOURCE", KCM_SOURCE
)

KSI_BAREMETAL_CAPI_PROVIDER_METAL3_CUSTOM_REGISTRY = settings_func.get_var(
    "KSI_BAREMETAL_CAPI_PROVIDER_METAL3_CUSTOM_REGISTRY", KCM_CUSTOM_REGISTRY
)

KSI_BAREMETAL_CAPI_PROVIDER_METAL3_REPO_DICT = {
    "ci-registry": "oci://registry.ci.mirantis.com/k0rdent-bm/charts",
    "enterprise-registry": "oci://registry.mirantis.com/k0rdent-bm/charts",
}
# END: metal3 specific
# ====
# END: Provider specific configs
# ====


# START: GCore inference specific
KSI_INFERENCE_BOX_CLUSTER_NAME = settings_func.get_var("KSI_INFERENCE_BOX_CLUSTER_NAME", "dummy-cluster")
KSI_INFERENCE_BOX_KEYCLOAK_URL = settings_func.get_var("KSI_INFERENCE_BOX_KEYCLOAK_URL", "")
KSI_INFERENCE_BOX_KEYCLOAK_CLIENT_ID = settings_func.get_var("KSI_INFERENCE_BOX_KEYCLOAK_CLIENT_ID",
                                                             KSI_INFERENCE_BOX_CLUSTER_NAME)
KSI_INFERENCE_BOX_KEYCLOAK_REALM_NAME = settings_func.get_var("KSI_INFERENCE_BOX_KEYCLOAK_REALM_NAME", "realmWithLdap")
KSI_INFERENCE_BOX_API_URL = settings_func.get_var("KSI_INFERENCE_BOX_API_URL", "")
KSI_INFERENCE_BOX_API_USERNAME = settings_func.get_var("KSI_INFERENCE_BOX_API_USERNAME", "")
KSI_INFERENCE_BOX_API_PASSWORD = settings_func.get_var("KSI_INFERENCE_BOX_API_PASSWORD", "")

KSI_INFERENCE_ROOT_APP_NAME = settings_func.get_var("KSI_INFERENCE_ROOT_APP_NAME", "the-app-of-apps")
KSI_INFERENCE_ROOT_APP_NAMESPACE = settings_func.get_var("KSI_INFERENCE_ROOT_APP_NAMESPACE", "argocd")
KSI_INFERENCE_ROOT_APP_REPO_URL = settings_func.get_var(
    "KSI_INFERENCE_ROOT_APP_REPO_URL",
    "http://gitea-http.gitea.svc.cluster.local:3000/box/box-cluster.git")
KSI_INFERENCE_ROOT_APP_PATH = settings_func.get_var("KSI_INFERENCE_ROOT_APP_PATH", "apps-of-apps/")
KSI_INFERENCE_ROOT_APP_BASE_TARGET_REVISION = settings_func.get_var(
    "KSI_INFERENCE_ROOT_APP_BASE_TARGET_REVISION",
    "inference")
KSI_INFERENCE_ROOT_APP_UPGRADE_TARGET_REVISION = settings_func.get_var(
    "KSI_INFERENCE_ROOT_APP_UPGRADE_TARGET_REVISION",
    "inference-upgrade")
KSI_INFERENCE_BOX_CLUSTER_BASE_IMAGES = settings_func.get_var("KSI_INFERENCE_BOX_CLUSTER_BASE_IMAGES", "")
KSI_INFERENCE_BOX_CLUSTER_UPGRADE_IMAGES = settings_func.get_var("KSI_INFERENCE_BOX_CLUSTER_UPGRADE_IMAGES", "")

# Harbod installed as part of the Box
KSI_INFERENCE_BOX_HARBOR_URL = settings_func.get_var("KSI_INFERENCE_BOX_HARBOR_URL", "")
KSI_INFERENCE_BOX_HARBOR_ADMIN_USERNAME = settings_func.get_var("KSI_INFERENCE_BOX_HARBOR_ADMIN_USERNAME", "admin")
KSI_INFERENCE_BOX_HARBOR_ADMIN_PASSWORD = settings_func.get_var("KSI_INFERENCE_BOX_HARBOR_ADMIN_PASSWORD", "")

# Inference object test parameters
KSI_INFERENCE_TEST_PROJECT_NAME = settings_func.get_var("KSI_INFERENCE_TEST_PROJECT_NAME", "test-project")
KSI_INFERENCE_TEST_NODEGROUP_NAME = settings_func.get_var("KSI_INFERENCE_TEST_NODEGROUP_NAME", "test-nodegroup")
KSI_INFERENCE_TEST_USERGROUP_NAME = settings_func.get_var("KSI_INFERENCE_TEST_USERGROUP_NAME", "test-group")
KSI_INFERENCE_TEST_USER_NAME = settings_func.get_var("KSI_INFERENCE_TEST_USER_NAME", "test-user")
KSI_INFERENCE_TEST_REGISTRY_NAME = settings_func.get_var("KSI_INFERENCE_TEST_REGISTRY_NAME", "test-registry")
KSI_INFERENCE_TEST_REGISTRY_USER_NAME = settings_func.get_var(
    "KSI_INFERENCE_TEST_REGISTRY_USER_NAME", "test-registry-user")
KSI_INFERENCE_TEST_REGISTRY_STORAGE_LIMIT = settings_func.get_var("KSI_INFERENCE_TEST_REGISTRY_STORAGE_LIMIT", "100G")
KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_URL = settings_func.get_var(
    "KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_URL", "https://registry.hub.docker.com/")
KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_USER = settings_func.get_var(
    "KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_USER", None)
KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_PASSWORD = settings_func.get_var(
    "KSI_INFERENCE_TEST_POD_IMAGE_SRC_REGISTRY_PASSWORD", None)
KSI_INFERENCE_TEST_POD_IMAGE_NAME = settings_func.get_var("KSI_INFERENCE_TEST_POD_IMAGE_NAME", "library/nginx")
KSI_INFERENCE_TEST_POD_IMAGE_TAG = settings_func.get_var("KSI_INFERENCE_TEST_POD_IMAGE_TAG", "latest")
KSI_INFERENCE_TEST_POD_IMAGE_REPLICATION_ADAPTER_TYPE = settings_func.get_var(
    "KSI_INFERENCE_TEST_POD_IMAGE_REPLICATION_ADAPTER_TYPE", "docker-registry")
KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_URL = settings_func.get_var(
    "KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_URL", "https://registry.hub.docker.com/")
KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_USER = settings_func.get_var(
    "KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_USER", None)
KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_PASSWORD = settings_func.get_var(
    "KSI_INFERENCE_TEST_VOLUME_IMAGE_SRC_REGISTRY_PASSWORD", None)
KSI_INFERENCE_TEST_VOLUME_IMAGE_NAME = settings_func.get_var("KSI_INFERENCE_TEST_VOLUME_IMAGE_NAME", "library/nginx")
KSI_INFERENCE_TEST_VOLUME_IMAGE_TAG = settings_func.get_var("KSI_INFERENCE_TEST_VOLUME_IMAGE_TAG", "latest")
KSI_INFERENCE_TEST_VOLUME_IMAGE_REPLICATION_ADAPTER_TYPE = settings_func.get_var(
    "KSI_INFERENCE_TEST_VOLUME_IMAGE_REPLICATION_ADAPTER_TYPE", "docker-registry")
KSI_INFERENCE_TEST_VOLUME_IMAGE_MOUNT_PATH = settings_func.get_var(
    "KSI_INFERENCE_TEST_VOLUME_IMAGE_MOUNT_PATH", "/model")
KSI_INFERENCE_TEST_VOLUME_NAME = settings_func.get_var("KSI_INFERENCE_TEST_VOLUME_NAME", "test-volume")
KSI_INFERENCE_TEST_APIKEY_NAME = settings_func.get_var("KSI_INFERENCE_TEST_APIKEY_NAME", "test-apikey")
KSI_INFERENCE_TEST_FLAVOR_NAME = settings_func.get_var("KSI_INFERENCE_TEST_FLAVOR_NAME", "test-flavor-1vcpu-1gib")
KSI_INFERENCE_TEST_FLAVOR_CPU = settings_func.get_var("KSI_INFERENCE_TEST_FLAVOR_CPU", "1")
KSI_INFERENCE_TEST_FLAVOR_MEMORY = settings_func.get_var("KSI_INFERENCE_TEST_FLAVOR_MEMORY", "1Gi")
KSI_INFERENCE_TEST_FLAVOR_GPU = settings_func.get_var("KSI_INFERENCE_TEST_FLAVOR_GPU", None)
KSI_INFERENCE_TEST_FLAVOR_GPU_MEMORY = settings_func.get_var("KSI_INFERENCE_TEST_FLAVOR_GPU_MEMORY", None)
KSI_INFERENCE_TEST_FLAVOR_GPU_MODEL = settings_func.get_var("KSI_INFERENCE_TEST_FLAVOR_GPU_MODEL", None)
KSI_INFERENCE_TEST_FLAVOR_IS_GPU_SHARED = settings_func.get_var_as_bool(
    "KSI_INFERENCE_TEST_FLAVOR_IS_GPU_SHARED", False)
KSI_INFERENCE_TEST_INFERENCE_PORT = int(settings_func.get_var("KSI_INFERENCE_TEST_INFERENCE_PORT", 80))
KSI_INFERENCE_TEST_INFERENCE_NAME = settings_func.get_var("KSI_INFERENCE_TEST_INFERENCE_NAME", "test-nginx")

KSI_INFERENCE_TEST_MLAPP_NAME = settings_func.get_var("KSI_INFERENCE_TEST_MLAPP_NAME", "test-app")
KSI_INFERENCE_TEST_MLAPPTEMPLATE_NAME = settings_func.get_var("KSI_INFERENCE_TEST_MLAPPTEMPLATE_NAME", "qwen-qwen3-14b")
KSI_INFERENCE_TEST_MLAPP_UI_LOGIN = settings_func.get_var("KSI_INFERENCE_TEST_MLAPP_UI_LOGIN", "admin@box.local")
KSI_INFERENCE_TEST_MLAPP_UI_PASSWORD = settings_func.get_var("KSI_INFERENCE_TEST_MLAPP_UI_PASSWORD", None)

KSI_INFERENCE_APP_LABEL_SELECTOR = settings_func.get_var("KSI_INFERENCE_BOX_STACK_APP_LABEL_SELECTOR",
                                                         "cloud.gcore.com/app=box-stack")
KSI_INFERENCE_APP_EXPECTED_STATUS = settings_func.get_var("KSI_INFERENCE_APPS_EXPECTED_STATUS",
                                                          "perfect")  # perfect, as_before
# END: GCore inference specific

# START: Upgrade specific
KSI_UPGRADE_TARGET_RELEASE = settings_func.get_var("KSI_UPGRADE_TARGET_RELEASE", None)
# END: Upgrade specific

# START: StorageClass and Provisioner
KSI_CHECK_FOR_DEFAULT_STORAGECLASS = settings_func.get_var_as_bool("KSI_CHECK_FOR_DEFAULT_STORAGECLASS", "false")
KSI_STORAGECLASS_NAME = settings_func.get_var("KSI_STORAGECLASS_NAME", "standard")
KSI_LOCAL_STORAGECLASS_YAML = settings_func.get_path("KSI_LOCAL_STORAGECLASS_YAML",
                                                     "templates/deploy-objects/storageclasses/localpathstorage.yaml")
KSI_DEFAULT_STORAGECLASS = settings_func.get_var_as_bool("KSI_DEFAULT_STORAGECLASS", True)
KSI_PROVISIONER_NAME = settings_func.get_var("KSI_PROVISIONER_NAME", "local-path-provisioner")
KSI_PROVISIONER_YAML = settings_func.get_path("KSI_PROVISIONER_YAML",
                                              "templates/deploy-objects/provisioners/localpathprovisioner.yaml")
KSI_PROVISIONER_IMAGE = settings_func.get_var(
    "KSI_PROVISIONER_IMAGE",
    "docker-kaas-local.artifactory.mcp.mirantis.net/general/external/rancher/local-path-provisioner:v0.0.32"
    )
# END: StorageClass and Provisioner
# START: Ansible specific
KSI_ANSIBLE_HOME_PATH = settings_func.get_var(
    "KSI_ANSIBLE_HOME_PATH",
    str(resources.files(__package__) / "test_env_data/")
)
KSI_ANSIBLE_PB_PATH = settings_func.get_var(
    "KSI_ANSIBLE_PB_PATH",
    str(resources.files(__package__) / "test_env_data/labs/{}/ansible_data")
    .format(KSI_ENV_CONFIG_NAME or ENV_NAME)
)
# END: Ansible specific
