#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import json
from retry import retry
import requests
from tabulate import tabulate
import yaml

from si_tests import logger
from si_tests.utils import utils, waiters

LOG = logger.logger


class InferenceManager(object):
    """GCore inference manager"""

    def __init__(self, box_api_client):
        self.box_api_client = box_api_client

    # Capacity API
    def list_capacities(self):
        """List box cluster capacity: how many instances can be deployed with each flavor"""
        capacity = self.box_api_client.capacity.v1_list_capacities()
        return capacity

    def show_capacity(self):
        """Show box cluster capacity"""
        capacity = self.list_capacities()
        LOG.debug(f"Capacity: {capacity}")
        headers = ["Flavor", "Region", "Capacity"]
        capacity_data = [[k, region, v]
                         for region, cap in capacity.items() for k, v in cap.items()]
        status_msg = tabulate(capacity_data, tablefmt="presto", headers=headers)
        LOG.info(f"Capacity:\n{status_msg}\n")

    # GPU nodes API
    def list_gpu_nodes(self):
        """List nodes with GPU that are available to use for inference workloads"""
        gpu_nodes = self.box_api_client.nodes.v1_list_gpu_nodes()
        return gpu_nodes.results

    def show_gpu_nodes(self):
        """Show GPU nodes details"""
        gpu_nodes = self.list_gpu_nodes()
        LOG.debug(f"GPU nodes: {gpu_nodes}")
        headers = ["GPU node name",
                   "Region",
                   "Node group",
                   "GPU",
                   "Available CPU",
                   "Available memory",
                   "Available storage"]
        nodes_data = [[data.name,
                       data.region_name,
                       data.node_group,
                       data.gpu_model,
                       f"{data.available_resources['cpu']} / {data.total_resources['cpu']}",
                       (f"{utils.convert_to_gb(data.available_resources['memory'])} GiB / "
                        f"{utils.convert_to_gb(data.total_resources['memory'])} GiB"),
                       (f"{utils.convert_to_gb(data.available_resources['ephemeral-storage'])} GiB / "
                        f"{utils.convert_to_gb(data.total_resources['ephemeral-storage'])} GiB")
                       ]
                      for data in gpu_nodes]
        status_msg = tabulate(nodes_data, tablefmt="presto", headers=headers)
        LOG.info(f"GPU Nodes:\n{status_msg}\n")

    # Flavors API
    def list_flavors(self):
        """List available flavors for inference workloads"""
        flavors = self.box_api_client.flavors.v1_list_flavors()
        LOG.debug(f"Flavors: {flavors}")
        return flavors.results

    def get_or_create_flavor(self, flavor_name, cpu, memory, gpu=None, gpu_memory=None,
                             gpu_model=None, gpu_compute_capability="8.0", is_gpu_shared=False):
        """
        Create a new flavor if it does not exist, otherwise return the existing one.

        :param flavor_name: str, Name of the flavor to create or retrieve.
        :param cpu: str, CPU amount (e.g., "1000m").
        :param memory: str, Memory amount (e.g., "4Gi").
        :param gpu: str, GPU amount (e.g., "1").
        :param gpu_memory: str, required GPU VRAM (e.g., "16Gi").
        :param gpu_model: str, GPU model (e.g., "NVIDIA A100").
        :param gpu_compute_capability: str, GPUComputeCapability is Nvidia hardware generation, e.g. "8.0" for Ampere.
        :param is_gpu_shared: bool, Whether the GPU is shared (MIG-enabled).
        :return: The flavor object.
        """
        # List all existing flavors
        flavors = self.list_flavors()
        match_flavors = [f for f in flavors if f.name == flavor_name]

        if not match_flavors:
            LOG.info(f"Flavor '{flavor_name}' not found, creating")
            create_request = self.box_api_client.flavorsmanagement.V1CreateFlavorRequest(
                name=flavor_name,
                cpu=cpu,
                memory=memory,
                gpu=gpu,
                gpu_memory=gpu_memory,
                gpu_model=gpu_model,
                gpu_compute_capability=gpu_compute_capability,
                is_gpu_shared=is_gpu_shared
            )
            LOG.info(create_request)
            flavor = self.box_api_client.flavorsmanagement.v1_create_flavor(create_request)
            self.wait_resource_readiness(flavor_name, self.list_flavors)
        else:
            flavor = match_flavors[0]

        return flavor

    def show_flavors(self):
        """Show available flavors"""
        flavors = self.list_flavors()
        headers = ["Flavor name", "CPU", "Memory", "GPU", "Network", "Additional Node Selector"]
        flavors_data = [[data.name,
                         data.cpu,
                         data.memory,
                         f"{data.gpu} x {data.gpu_model} , {data.gpu_memory}" if int(data.gpu) else "-",
                         data.network,
                         data.additional_node_selector,
                         ]
                        for data in flavors]
        status_msg = tabulate(flavors_data, tablefmt="presto", headers=headers)
        LOG.info(f"Flavors:\n{status_msg}\n")

    def is_flavor_exists(self, flavor_name):
        """Check if the flavor with the specified name exists"""
        objs = self.list_flavors()
        return any(obj.name == flavor_name for obj in objs)

    def delete_flavor(self, flavor_name):
        LOG.warning(f"Deleting Flavor '{flavor_name}'")
        self.box_api_client.flavorsmanagement.v1_delete_flavor(flavor_name)
        self.wait_resource_absense(flavor_name, lambda: self.list_flavors())

    # Projects API
    def list_projects(self):
        """List box projects (special namespaces in kubernetes)"""
        projects = self.box_api_client.projects.v1_list_projects()
        LOG.debug(f"Projects: {projects}")
        return projects.results

    def get_or_create_project(self, project_name):
        """Create a new project only if not exists"""
        projects = self.list_projects()
        match_projects = [p for p in projects if project_name == p.name]
        if not match_projects:
            LOG.info(f"Project {project_name} not found, creating")
            create_project_request = self.box_api_client.projects.V1CreateProjectRequest(name=project_name)
            project = self.box_api_client.projects.v1_create_project(create_project_request)
            self.wait_resource_readiness(project_name, self.list_projects)
        else:
            project = match_projects[0]
        return project

    def show_projects(self):
        """Show box projects"""
        projects = self.list_projects()
        headers = ["Project name", "Created at"]
        projects_data = [[data.name,
                          data.created_at,
                          ]
                         for data in projects]
        status_msg = tabulate(projects_data, tablefmt="presto", headers=headers)
        LOG.info(f"Projects:\n{status_msg}\n")

    def is_project_exists(self, project_name):
        """Check if the project with the specified name exists"""
        objs = self.list_projects()
        return any(obj.name == project_name for obj in objs)

    def delete_project(self, project_name):
        LOG.warning(f"Deleting Project '{project_name}'")
        self.box_api_client.projects.v1_delete_project(project_name)
        self.wait_resource_absense(project_name, lambda: self.list_projects())

    # UserGroups API
    def list_usergroups(self):
        """List box usergroups"""
        usergroups = self.box_api_client.user_groups.v1_list_groups()
        LOG.debug(f"UserGroups: {usergroups}")
        return usergroups.results

    def get_or_create_usergroup(self, usergroup_name, project_name):
        """Create a new usergroup only if not exists"""
        usergroups = self.list_usergroups()
        match_usergroups = [ug for ug in usergroups if usergroup_name == ug.name]
        if not match_usergroups:
            LOG.info(f"UserGroup {usergroup_name} not found, creating")
            access_rule = self.box_api_client.user_groups.V1AccessRule(access_type="rw", namespace_name=project_name)
            create_usergroup_request = self.box_api_client.user_groups.V1CreateGroupRequest(
                name=usergroup_name, access_rules=[access_rule])
            usergroup = self.box_api_client.user_groups.v1_create_group(create_usergroup_request)
            self.wait_resource_readiness(usergroup_name, self.list_usergroups)
        else:
            usergroup = match_usergroups[0]
        return usergroup

    def show_usergroups(self):
        """Show box usergroups"""
        usergroups = self.list_usergroups()
        headers = ["UserGroup name", "Is superuser?", "Access rules", "Created at"]
        usergroups_data = [[data.name,
                            data.is_super_user,
                            data.access_rules,
                            data.created_at,
                            ]
                           for data in usergroups]
        status_msg = tabulate(usergroups_data, tablefmt="presto", headers=headers)
        LOG.info(f"UserGroups:\n{status_msg}\n")

    def is_usergroup_exists(self, usergroup_name):
        """Check if the usergroup with the specified name exists"""
        objs = self.list_usergroups()
        return any(obj.name == usergroup_name for obj in objs)

    def delete_usergroup(self, usergroup_name):
        LOG.warning(f"Deleting UserGroup '{usergroup_name}'")
        self.box_api_client.user_groups.v1_delete_group(usergroup_name)
        self.wait_resource_absense(usergroup_name, lambda: self.list_usergroups())

    # Users API
    def list_users(self):
        """List box users"""
        users = self.box_api_client.users.v1_list_users()
        LOG.debug(f"Users: {users}")
        return users.results

    def get_or_create_user(self, user_name, usergroups, project_name):
        """Create a new user only if not exists"""
        users = self.list_users()
        match_users = [u for u in users if user_name == u.name]
        if not match_users:
            LOG.info(f"User {user_name} not found, creating")
            access_rule = self.box_api_client.users.V1AccessRule(access_type="rw", namespace_name=project_name)
            create_user_request = self.box_api_client.users.V1CreateUserRequest(
                name=user_name, access_rules=[access_rule], email=f"{user_name}@test.local", groups=usergroups)
            user = self.box_api_client.users.v1_create_user(create_user_request)
            self.wait_resource_readiness(user_name, self.list_users)
            self.wait_resource_readiness(user_name, lambda: self._get_user_secret_for_wait(user_name))
        else:
            user = match_users[0]
        return user

    def show_users(self):
        """Show box users"""
        users = self.list_users()
        headers = ["User name", "Is superuser?", "Groups", "Access rules", "Created at"]
        users_data = [[data.name,
                       data.is_super_user,
                       data.groups,
                       data.access_rules,
                       data.created_at,
                       ]
                      for data in users]
        status_msg = tabulate(users_data, tablefmt="presto", headers=headers)
        LOG.info(f"Users:\n{status_msg}\n")

    def get_user_secret(self, user_name):
        """Get box user's secret with credentials"""
        try:
            secret = self.box_api_client.users.v1_get_user_secret(user_name)
            return secret
        except Exception:
            LOG.debug(f"Secret for User '{user_name}' not found ...")
            return None

    def _get_user_secret_for_wait(self, user_name):
        """Prepare a list from the user secret, to use with wait_resource_readiness()"""
        secret = self.get_user_secret(user_name)
        if not secret:
            # Return empty list until the secret is created
            return []
        # Return the list of users, so the wait_resource_readiness()
        # could pass the check for user_name when the secret is created
        return self.list_users()

    def get_user_password(self, user_name):
        """Get box user's password"""
        secret = self.get_user_secret(user_name)
        LOG.debug(f"User secret: {secret}")
        return secret.secret

    def is_user_exists(self, user_name):
        """Check if the user with the specified name exists"""
        objs = self.list_users()
        return any(obj.name == user_name for obj in objs)

    def delete_user(self, user_name):
        LOG.warning(f"Deleting Box User '{user_name}'")
        self.box_api_client.users.v1_delete_user(user_name)
        self.wait_resource_absense(user_name, lambda: self.list_users())

    # Registry API
    def list_registries(self, project_name):
        """List box registries"""
        registries = self.box_api_client.registries.v1_list_registries(project_name)
        LOG.debug(f"Registries: {registries}")
        return registries.results

    def list_registry_images(self, registry_name, project_name):
        """List box registry images"""
        registry_images = self.box_api_client.registries.v1_list_registry_images(project_name, registry_name)
        LOG.debug(f"Registry images: {registry_images}")
        return registry_images.results

    def get_or_create_registry(self, registry_name, project_name, storage_limit, public=False):
        """Create a new Registry only if not exists"""
        registries = self.list_registries(project_name)
        match_registries = [r for r in registries if registry_name == r.name]
        if not match_registries:
            LOG.info(f"Registry {registry_name} not found, creating")
            preheat_policies = None  # TBD(ddmitriev): implement preheat policies list
            create_registry_request = self.box_api_client.registries.V1CreateRegistryRequest(
                name=registry_name, preheat_policies=preheat_policies, public=public, storage_limit=storage_limit)
            self.box_api_client.registries.v1_create_registry(project_name, create_registry_request)
            self.wait_resource_readiness(registry_name,
                                         lambda: self.list_registries(project_name),
                                         readiness_attr='status',
                                         readiness_value='Ready')
            registry = self.box_api_client.registries.v1_get_registry(project_name, registry_name)
        else:
            registry = match_registries[0]
        return registry

    def show_registries(self, project_name):
        """Show box Registries"""
        registries = self.list_registries(project_name)
        headers = ["Registry name", "Project", "Preheat policies", "Public", "Status", "Storage limit", "URL"]
        registries_data = [[data.name,
                            data.namespace,
                            data.preheat_policies,
                            data.public,
                            data.status,
                            data.storage_limit,
                            data.url,
                            ]
                           for data in registries]
        status_msg = tabulate(registries_data, tablefmt="presto", headers=headers)
        LOG.info(f"Registries:\n{status_msg}\n")

    def show_registry_images(self, registry_name, project_name):
        """Show box Registry images"""
        registry_images = self.list_registry_images(registry_name, project_name)
        headers = ["Display name", "Registry", "Name", "Tags", "Size"]
        registry_images_data = [[data.display_name,
                                 data.repository,
                                 data.name,
                                 data.tags,
                                 data.size,
                                 ]
                                for data in registry_images]
        status_msg = tabulate(registry_images_data, tablefmt="presto", headers=headers)
        LOG.info(f"Registries:\n{status_msg}\n")

    def is_registry_exists(self, registry_name, project_name):
        """Check if the registry with the specified name exists"""
        objs = self.list_registries(project_name)
        return any(obj.name == registry_name for obj in objs)

    def is_registry_image_exists(self, image_name, image_tag, registry_name, project_name):
        """Check if the registry image with the specified name and tag exists"""
        objs = self.list_registry_images(registry_name, project_name)
        registry_image_name = f"{registry_name}.{image_name}".replace('/', '.')
        return any((obj.name == registry_image_name and image_tag in obj.tags) for obj in objs)

    def delete_registry(self, registry_name, project_name):
        LOG.warning(f"Deleting Registry '{registry_name}' from the Project '{project_name}'")
        self.box_api_client.registries.v1_delete_registry(project_name, registry_name)
        self.wait_resource_absense(registry_name, lambda: self.list_registries(project_name))

    # RegistryUsers API
    def list_registry_users(self, registry_name, project_name):
        """List box RegistryUsers"""
        registry_users = self.box_api_client.registry_users.v1_list_registry_users(project_name, registry_name)
        LOG.debug(f"RegistryUsers: {registry_users}")
        return registry_users.results

    def get_or_create_registry_users(self, registry_user_name, registry_name, project_name, read_only=False):
        """Create a new RegistryUser only if not exists"""
        registry_users = self.list_registry_users(registry_name, project_name)
        match_registry_users = [u for u in registry_users if registry_user_name == u.name]
        if not match_registry_users:
            LOG.info(f"RegistryUser {registry_user_name} not found, creating")
            create_registry_user_request = self.box_api_client.registry_users.V1CreateRegistryUserRequest(
                name=registry_user_name, read_only=read_only)
            registry_user = self.box_api_client.registry_users.v1_create_registry_user(
                project_name, registry_name, create_registry_user_request)
            self.wait_resource_readiness(registry_user_name,
                                         lambda: self.list_registry_users(registry_name, project_name),
                                         readiness_attr='pull_secret_name',
                                         readiness_value=None)
            self.wait_resource_readiness(f"{registry_name}-{registry_user_name}",
                                         lambda: self._get_registry_user_credentials_for_wait(
                                             registry_user_name, registry_name, project_name))
        else:
            registry_user = match_registry_users[0]
        return registry_user

    def show_registry_users(self, registry_name, project_name):
        """Show box RegistryUsers"""
        registry_users = self.list_registry_users(registry_name, project_name)
        headers = ["RegistryUser name", "PullSecret name", "ReadOnly", "Registry name", "PullSecret Credentials"]
        registry_users_data = [[data.name,
                                data.pull_secret_name,
                                data.read_only,
                                data.registry_name,
                                self.get_registry_user_credentials(data.name, registry_name, project_name),
                                ]
                               for data in registry_users]
        status_msg = tabulate(registry_users_data, tablefmt="presto", headers=headers)
        LOG.info(f"RegistryUsers:\n{status_msg}\n")

    def get_registry_user_credentials(self, registry_user_name, registry_name, project_name):
        """Get box RegistryUser's credentials"""
        try:
            pull_secret = self.box_api_client.registry_users.v1_get_registry_user_pull_secret(
                project_name, registry_name, registry_user_name)
            LOG.debug(f"RegistryUser pull secret: {pull_secret}")
            # V1PullSecretResponse:
            # login: pull_secret.login,
            # name: pull_secret.name,
            # password: pull_secret.password,
            # registry: pull_secret.registry,
            return pull_secret
        except Exception:
            LOG.warning(f"RegistryUser pull secret for User '{registry_user_name}' not found ...")
            return None

    def _get_registry_user_credentials_for_wait(self, registry_user_name, registry_name, project_name):
        """Prepare a list from the user secret, to use with wait_resource_readiness()"""
        pull_secret = self.get_registry_user_credentials(registry_user_name, registry_name, project_name)
        if not pull_secret:
            # Return empty list until the secret is created
            return []
        # Return the list with a pull secret, so the wait_resource_readiness()
        # could pass the check for registry_user_name when the pull_secret is created
        return [pull_secret]

    def is_registry_user_exists(self, registry_user_name, registry_name, project_name):
        """Check if the registry_user with the specified name exists"""
        objs = self.list_registry_users(registry_name, project_name)
        return any(obj.name == registry_user_name for obj in objs)

    def delete_registry_user(self, registry_user_name, registry_name, project_name):
        LOG.warning(f"Deleting RegistryUser '{registry_user_name}' "
                    f"from the Registry '{registry_name}' in the Project '{project_name}'")
        self.box_api_client.registry_users.v1_delete_registry_user(project_name, registry_name, registry_user_name)
        self.wait_resource_absense(registry_user_name, lambda: self.list_registry_users(registry_name, project_name))

    # ApiKeys API
    def list_apikeys(self, project_name):
        """List box ApiKeys in the specified Project"""
        apikeys = self.box_api_client.api_keys.v1_list_api_keys(project_name)
        LOG.debug(f"ApiKeys: {apikeys}")
        return apikeys.results

    def create_or_recreate_apikey(self, apikey_name, project_name, description=None, expires_at=None):
        """Create a new ApiKey. If other ApiKey exists with the same name, it will be re-created"""
        apikeys = self.list_apikeys(project_name)
        match_apikeys = [ak for ak in apikeys if apikey_name == ak.name]
        if match_apikeys:
            LOG.info(f"Found existing ApiKey with name '{apikey_name}', delete and re-create it")
            self.delete_apikey(apikey_name, project_name)

        LOG.info(f"Creating ApiKey '{apikey_name}'")
        create_apikey_request = self.box_api_client.api_keys.V1CreateApiKeyRequest(
            name=apikey_name, expires_at=expires_at, description=description or f"apikey name = {apikey_name}")
        apikey = self.box_api_client.api_keys.v1_create_api_key(project_name, create_apikey_request)
        self.wait_resource_readiness(apikey_name, lambda: self.list_apikeys(project_name))
        return apikey

    def is_apikey_exists(self, apikey_name, project_name):
        """Check if the apikey with the specified name exists"""
        objs = self.list_apikeys(project_name)
        return any(obj.name == apikey_name for obj in objs)

    def delete_apikey(self, apikey_name, project_name):
        LOG.warning(f"Deleting ApiKey '{apikey_name}' from the Project '{project_name}'")
        self.box_api_client.api_keys.v1_delete_api_key(project_name, apikey_name)
        self.wait_resource_absense(apikey_name, lambda: self.list_apikeys(project_name))

    def show_apikeys(self, project_name):
        """Show box ApiKeys in the specified Project"""
        apikeys = self.list_apikeys(project_name)
        headers = ["ApiKey name", "Created at", "Expired at", "Inferences use the apikey", "Description"]
        apikeys_data = [[data.name,
                         data.created_at,
                         data.expires_at,
                         data.inferences,
                         data.description,
                         ]
                        for data in apikeys]
        status_msg = tabulate(apikeys_data, tablefmt="presto", headers=headers)
        LOG.info(f"ApiKeys:\n{status_msg}\n")

    # NodeGroups API
    def list_nodegroups(self):
        """List box NodeGroups"""
        nodegroups = self.box_api_client.node_groups.v1_list_node_groups()
        LOG.debug(f"NodeGroups: {nodegroups}")
        return nodegroups.results

    def get_or_create_nodegroup(self, nodegroup_name, project_names, node_refs, custom_labels=None):
        """Create a new NodeGroup only if not exists

        :param nodegroup_name: str, Name of the node group
        :param namespaces: list[str], List of Box project names
        :param custom_labels: dict[str, str], CustomLabels contains the list of custom labels
        :param node_refs: dict[str, list[str]], NodeRefs contains the list of node references by region

        """
        nodegroups = self.list_nodegroups()
        match_nodegroups = [ng for ng in nodegroups if nodegroup_name == ng.name]
        if not match_nodegroups:
            LOG.info(f"NodeGroup '{nodegroup_name}' not found, creating")
            create_nodegroup_request = self.box_api_client.node_groups.V1CreateNodeGroupRequest(
                name=nodegroup_name,
                namespaces=project_names,
                custom_labels=custom_labels,
                node_refs=node_refs)
            nodegroup = self.box_api_client.node_groups.v1_create_node_group(create_nodegroup_request)
            self.wait_resource_readiness(nodegroup_name, lambda: self.list_nodegroups())
        else:
            nodegroup = match_nodegroups[0]
        return nodegroup

    def is_nodegroup_exists(self, nodegroup_name):
        """Check if the nodegroup with the specified name exists"""
        objs = self.list_nodegroups()
        return any(obj.name == nodegroup_name for obj in objs)

    def delete_nodegroup(self, nodegroup_name):
        LOG.warning(f"Deleting NodeGroup '{nodegroup_name}'")
        self.box_api_client.node_groups.v1_delete_node_group(nodegroup_name)
        self.wait_resource_absense(nodegroup_name, lambda: self.list_nodegroups())

    def show_nodegroups(self):
        """Show box NodeGroups"""
        nodegroups = self.list_nodegroups()

        headers = ["Name", "Created at", "Projects", "Custom labels", "Nodes"]
        nodegroups_data = [[data.name,
                            data.created_at,
                            data.namespaces,
                            data.custom_labels,
                            data.node_refs,
                            ]
                           for data in nodegroups]
        status_msg = tabulate(nodegroups_data, tablefmt="presto", headers=headers)
        LOG.info(f"NodeGroups:\n{status_msg}\n")

    # Volumes API
    def list_volumes(self, project_name):
        """List volumes in the specified project."""
        volumes = self.box_api_client.volumes.v1_list_volumes(project_name=project_name)
        LOG.debug(f"Volumes: {volumes}")
        return volumes.results

    def get_or_create_volume_from_emptydir(self, volume_name, project_name, volume_regions,
                                           medium="Default", size_limit="10Gi"):
        """
        Get or Create a new volume from empty dir

        :param medium: str, Medium specifies the storage backing for EmptyDir, such as default disk, memory (tmpfs)
                       for example: "Default", "Memory"
        :param size_limit: str, size limit for the Volume
        """
        empty_dir_volume_source = self.box_api_client.volumes.V1EmptyDirVolumeSource(
            medium=medium, size_limit=size_limit)
        return self._get_or_create_volume(volume_name, project_name, volume_regions,
                                          empty_dir_volume_source=empty_dir_volume_source)

    def get_or_create_volume_from_image(self, volume_name, project_name, volume_regions, volume_image_reference):
        """
        Get or Create a new volume from an oci image

        :param reference: str, uri for the oci image to upload to the volume
        """
        image_volume_source = self.box_api_client.volumes.V1ImageVolumeSource(
            reference=volume_image_reference)
        return self._get_or_create_volume(volume_name, project_name, volume_regions,
                                          image_volume_source=image_volume_source)

    def get_or_create_volume_from_pvc(self, volume_name, project_name, volume_regions,
                                      access_mode, capacity, storage_class):
        """
        Get or Create a new volume from pvc

        :param access_mode: str, volume access mode, one of "ReadOnlyMany", "ReadWriteMany"
        :param capacity: str, volume size, e.g. "5Gi"
        :param storage_class: str, storage class to use
        """
        pvc_volume_source = self.box_api_client.volumes.V1PVCVolumeSource(
            access_mode=access_mode,
            capacity=capacity,
            storage_class=storage_class)
        return self._get_or_create_volume(volume_name, project_name, volume_regions,
                                          pvc_volume_source=pvc_volume_source)

    def _get_or_create_volume(self, volume_name, project_name, volume_regions, empty_dir_volume_source=None,
                              image_volume_source=None, pvc_volume_source=None):
        """
        Create a new volume if it does not exist, otherwise return the existing one.

        :param volume_name: str, name of the Volume to create or retrieve.
        :param project_name: str, name of the Project where the volume will be created
        :param regions: list of str, Regions where the volume should be available.

        :param storage_class: Storage class for the volume.
        :param size: Size of the volume (e.g., "10Gi").
        :param access_modes: List of access modes (e.g., ["ReadWriteOnce"]).
        :param volume_mode: Volume mode (e.g., "Filesystem").
        :return: The volume object.
        """
        volumes = self.list_volumes(project_name)
        match_volumes = [v for v in volumes if v.name == volume_name]

        if not match_volumes:
            LOG.info(f"Volume '{volume_name}' not found, creating")
            create_request = self.box_api_client.volumes.V1CreateVolumeRequest(
                name=volume_name,
                empty_dir=empty_dir_volume_source,
                image=image_volume_source,
                pvc=pvc_volume_source,
                regions=volume_regions
            )
            volume = self.box_api_client.volumes.v1_create_volume(project_name, create_request)
            self.wait_resource_readiness(volume_name, lambda: self.list_volumes(project_name))
        else:
            volume = match_volumes[0]

        return volume

    def show_volumes(self, project_name):
        """Show volumes in the specified project."""
        volumes = self.list_volumes(project_name)
        headers = ["Volume name", "Status", "Regions", "EmptyDir", "Image", "PVC"]
        volumes_data = [
            [
                data.name,
                data.status,
                data.regions,
                data.empty_dir,
                data.image,
                data.pvc,
            ]
            for data in volumes
        ]
        status_msg = tabulate(volumes_data, tablefmt="presto", headers=headers)
        LOG.info(f"Volumes:\n{status_msg}\n")

    def is_volume_exists(self, volume_name, project_name):
        """Check if the volume with the specified name exists"""
        objs = self.list_volumes(project_name)
        return any(obj.name == volume_name for obj in objs)

    def delete_volume(self, volume_name, project_name):
        """Delete a volume and wait until it is removed."""
        if not any(obj.name == volume_name for obj in self.list_volumes(project_name)):
            LOG.warning(f"Volume '{volume_name}' not found in the Project '{project_name}', nothing to delete")
            return
        LOG.warning(f"Deleting volume '{volume_name}' from project '{project_name}'")
        self.box_api_client.volumes.v1_delete_volume(project_name, volume_name)
        self.wait_resource_absense(volume_name, lambda: self.list_volumes(project_name))

    # Apps API
    def create_application_from_data(self, project_name, application_data, dry_run):
        """
        Create a new Application from the catalog

        project_name - str, name of the project where to create the inference workload
        application_data - dict, all the parameters for Application object
        """
        LOG.info(f"Create Application '{application_data.get('name')}' "
                 f"from MLApp template '{application_data.get('application_name')}' in project '{project_name}'")
        app_deployment_create_request = self.box_api_client.apps.SchemasAppDeploymentCreateRequest.from_dict(
            application_data)
        app = self.box_api_client.apps.v1_deploy_app(project_name, app_deployment_create_request, dry_run=dry_run)

        self.wait_resource_readiness(application_data['name'], lambda: self.list_mlapps(project_name))
        return app

    def create_application(
            self,
            project_name,
            mlapp_name,
            mlapptemplate_name,
            inference_regions,
            api_keys=None,
            components_configuration=None,
            dry_run=None):
        """
        Create a new inference workload using an existing volume and image from registry.

        :param project_name: str, name of the project
        :param mlapp_name: str, name for the new application
        :param mlapptemplate_name: str, identifier of application from MLApp catalog
        :param flavor_name: str, name of the flavor to use
        :param inference_regions: list[str], box regions where to deploy
        :param api_keys: list[str], API keys for access control
        :param exposed: bool, exposed component will obtain public address
        :param min_scale: int, Min is the minimum number of replicas the component can be scaled down to
        :param max_scale: int, Max is the maximum number of replicas the container can be scaled up to
        :param parameter_overrides: dict(str, str): Value is the new value assigned to the overridden parameter
        :param dry_run: bool, whether to perform a dry run
        :return: created Application object
        """
        application_data = {
          "name": mlapp_name,
          "application_name": mlapptemplate_name,
          "api_keys": api_keys or [],
          "components_configuration": components_configuration or {},
          "regions": inference_regions,
        }
        app = self.create_application_from_data(project_name, application_data, dry_run)
        return app

    def list_mlapps(self, project_name):
        """List MLAppDeployment objects"""
        apps = self.box_api_client.apps.v1_list_deployed_apps(project_name=project_name)
        LOG.debug(f"MLAppDeployments: {apps}")
        return apps.results

    def is_mlapp_exists(self, project_name, mlapp_name):
        """Check if the MLAppDeployment with the specified name exists"""
        objs = self.list_mlapps(project_name)
        return any(obj.name == mlapp_name for obj in objs)

    def delete_mlapp(self, project_name, mlapp_name, timeout=1200, interval=15):
        """Delete the specified MLAppDeployment and wait until it is removed from the apps list"""
        def _is_mlapp_deleted():
            mlapp_exists = self.is_mlapp_exists(project_name, mlapp_name)
            self.show_mlapps(project_name)

            inferences = self.get_inferences_by_mlapp(
                project_name=project_name,
                mlapp_name=mlapp_name)
            if inferences:
                LOG.warning(f"There are still some Inference objects related to the MLAppDeployment '{mlapp_name}':")
                self.show_inferences(project_name)

            return not mlapp_exists and not inferences

        LOG.info(f"Delete MLAppDeployment '{mlapp_name}' from the project '{project_name}'")
        self.box_api_client.apps.v1_delete_deployed_app(project_name=project_name, deploy_name=mlapp_name)
        waiters.wait(_is_mlapp_deleted,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout deleting MLAppDeployment '{mlapp_name}' after {timeout} sec.")
        LOG.info(f"MLAppDeployment '{mlapp_name}' is successfully deleted from the project '{project_name}'")

    def show_mlapps(self, project_name):
        """Show MLAppDeployments in the specified project"""
        mlapps = self.list_mlapps(project_name)
        headers = ["App name", "MLAppTemplate name", "Regions", "APIKeys", "Status"]
        mlapps_data = [[data.name,
                        data.application_name,
                        data.regions,
                        data.api_keys,
                        data.status,
                        ]
                       for data in mlapps]
        status_msg = tabulate(mlapps_data, tablefmt="presto", headers=headers)
        LOG.info(f"MLAppDeployments:\n{status_msg}\n")

    # Inference API
    def create_inference_request(self, inference_data):
        """Prepare a special object V1CreateInferenceRequest to create a new inference"""
        try:
            inference_request = self.box_api_client.inferences.V1CreateInferenceRequest.from_dict(inference_data)
        except Exception as e:
            # Swagger generated client can miss some attributes in the '.from_dict()' method.
            # Catch such errors and show details.
            if hasattr(e, 'errors'):
                raise Exception(f"Error creating 'CreateInferenceRequest' object: {e.errors()}")
            else:
                raise e
        return inference_request

    def create_inference(self, project_name, inference_name, flavor_name,
                         volume_name, mount_path, registry_user_name,
                         registry_name, registry_image_name, registry_image_tag,
                         inference_port, inference_regions, api_keys=None,
                         pull_secret=None, dry_run=None):
        """
        Create a new inference workload using an existing volume and image from registry.

        :param project_name: str, name of the project
        :param inference_name: str, name of the inference
        :param flavor_name: str, name of the flavor to use
        :param volume_name: str, name of the existing volume to mount
        :param mount_path: str, path where volume is mounted in the container
        :param registry_user_name: str, name of the registry user
        :param registry_name: str, name of the registry
        :param registry_image_name: str, name of the image in the registry
        :param registry_image_tag: str, tag of the image in the registry
        :param inference_port: int, port that inference listens on
        :param inference_regions: list[str], regions where to deploy
        :param api_keys: list[str], API keys for access control
        :param pull_secret: str, name of the pull secret (if needed)
        :param dry_run: bool, whether to perform a dry run
        :return: created inference object
        """
        # Create volume reference
        volume_ref = self.box_api_client.inferences.ApiServicesInferenceV1Volume(
            name=volume_name,
            mount_path=mount_path
        )

        # Create registry reference
        registry_ref = self.box_api_client.inferences.V1RegistryRef(
            name=registry_name
        )

        # Create registry image reference
        inference_registry_image_name = f"{registry_name}.{registry_image_name}".replace('/', '.')
        registry_image_ref = self.box_api_client.inferences.V1RegistryImageRef(
            name=inference_registry_image_name,
            tag=registry_image_tag
        )

        # Create registry user reference
        registry_user_ref = self.box_api_client.inferences.V1RegistryUserRef(
            name=registry_user_name
        )

        inference_data = {
          "api_keys": api_keys or [],
          "command": None,
          "description": None,
          "envs": None,
          "envs_per_region": None,
          "expose": None,
          "flavor": {
              "name": flavor_name
          },
          "host_header": None,
          "image": None,  # We'll use registry_image instead
          "ingress_opts": {
              "read_timeout": None
          },
          "is_disabled": False,
          "is_ingress_disabled": False,
          "listening_port": inference_port,
          "logging": None,
          "metrics_port": None,
          "name": inference_name,
          "probes": {
              "liveness_probe": {
                  "enabled": False,
                  "probe": None
              },
              "readiness_probe": {
                  "enabled": False,
                  "probe": None
              },
              "startup_probe": {
                  "enabled": False,
                  "probe": None
              }
          },
          "pull_secret": pull_secret,
          "regions": inference_regions,
          "registry": registry_ref,  # We're using registry_user + registry_image directly
          "registry_image": registry_image_ref,
          "registry_user": registry_user_ref,
          "timeout": 180,
          "tls_secret": None,
          "volumes": [volume_ref],
          "scale_per_region": {}
        }

        # Set scale per region
        for inference_region in inference_regions:
            inference_data['scale_per_region'][inference_region] = {
                "min": 1,
                "max": 1,
                "cooldown_period": 60,
                "triggers": {
                    "cpu": {
                        "threshold": 80
                    },
                    "gpu_memory": None,
                    "gpu_utilization": {
                        "threshold": 80
                    },
                    "http": None,
                    "memory": None
                }
            }

        return self.create_inference_from_data(project_name, inference_data, dry_run)

    def create_inference_from_data(self, project_name, inference_data, dry_run):
        """Create a new inference workload
        project_name - str, name of the project where to create the inference workload
        inference_data - dict, all the parameters for inference object
        """
        LOG.info(f"Create inference '{inference_data.get('name')}' in project '{project_name}'")
        inference_request = self.create_inference_request(inference_data)
        inference = self.box_api_client.inferences.v1_create_inference(
            project_name, inference_request, dry_run=dry_run)
        return inference

    def update_inference(self,
                         project_name,
                         inference_name,
                         inference_update_data):

        inference_update_request = self.box_api_client.inferences.V1UpdateInferenceRequest.from_dict(
            inference_update_data)
        inference = self.box_api_client.inferences.v1_update_inference(
            project_name, inference_name, inference_update_request)
        return inference

    def disable_inference(self, project_name, inference_name, timeout=300, interval=5):
        """Disable Inference instance. After disabling, it's ingress will return HTTP 404"""

        def _check_ingress_disabled():
            for region, address in addresses.items():
                try:
                    response = requests.get(address, verify=False)
                except Exception as e:
                    LOG.warning(f"Request to {address} failed: {e}")
                    return False
                LOG.info(f"{region}/{address}: {response.status_code}")
                if response.status_code != 404:
                    return False
            return True

        addresses = self.get_inference_addresses(
            project_name=project_name,
            inference_name=inference_name)

        inference_update_data = {
            'is_disabled': True
        }
        self.update_inference(
            project_name=project_name,
            inference_name=inference_name,
            inference_update_data=inference_update_data)

        LOG.info("Wait until Inference is disabled, and it's ingress returns 404 code")
        waiters.wait(_check_ingress_disabled,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting inference '{inference_name}' is disabled after {timeout} sec.")
        LOG.info(f"Inference '{inference_name}' is successfully disabled")

    def enable_inference(self, project_name, inference_name, timeout=300, interval=5):
        def _check_ingress_enabled():
            for region, address in addresses.items():
                try:
                    response = requests.get(address, verify=False)
                except Exception as e:
                    LOG.warning(f"Request to {address} failed: {e}")
                    return False
                LOG.info(f"{region}/{address}: {response.status_code}")
                if response.status_code == 404:
                    return False
            return True

        addresses = self.get_inference_addresses(
            project_name=project_name,
            inference_name=inference_name)

        inference_update_data = {
            'is_disabled': False
        }
        self.update_inference(
            project_name=project_name,
            inference_name=inference_name,
            inference_update_data=inference_update_data)

        LOG.info("Wait for Inference to be ready after enabling")
        self.wait_inferences_ready(
            project_name=project_name,
            inference_names=[inference_name]
        )

        LOG.info("Wait until Inference is enabled, and it's ingress returns non-404 code")
        waiters.wait(_check_ingress_enabled,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting inference '{inference_name}' is enabled after {timeout} sec.")
        LOG.info(f"Inference '{inference_name}' is successfully enabled")

    def list_inferences(self, project_name):
        """List inference objects"""
        inferences = self.box_api_client.inferences.v1_list_inferences(project_name=project_name)
        LOG.debug(f"Inferences: {inferences}")
        return inferences.results

    def get_inferences_by_mlapp(self, project_name, mlapp_name):
        """Get inference objects created by the MLAppDeployment"""
        inferences = self.list_inferences(project_name)
        inferences_with_mlapp = [
            inference for inference in inferences
            if (inference.object_references and
                inference.object_references[0].kind == 'AppDeployment' and
                inference.object_references[0].name == mlapp_name)
        ]
        LOG.debug(f"Inferences related to MLAppDeployment 'mlapp_name': {inferences_with_mlapp}")
        return inferences_with_mlapp

    def wait_inferences_with_mlapp(self, project_name, mlapp_name, min_count=2, timeout=300, interval=5):
        def _check_inferences_with_mlapp():
            inferences = self.get_inferences_by_mlapp(project_name, mlapp_name)
            LOG.info(f"Inferences related to MLAppDeployment '{mlapp_name}': {inferences}")
            return len(inferences) >= min_count

        waiters.wait(_check_inferences_with_mlapp,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting at least {min_count} inferences "
                                 f"with mlapp '{mlapp_name}' after {timeout} sec.")
        LOG.info(f"Found >='{min_count}' Inferences with mlapp '{mlapp_name}'")

    def get_inference(self, project_name, inference_name):
        """Get inference object from the list of inferences"""
        # Use v1_list_inferences instead of v1_get_inference because of bug in swagger client for 'statuses' field
        inferences = self.list_inferences(project_name)
        inference = [inference for inference in inferences if inference.name == inference_name]
        if not inference:
            raise Exception(f"Inference {inference_name} not found")
        LOG.debug(f"Inference: {inference[0]}")
        return inference[0]

    def is_inference_exists(self, project_name, inference_name):
        """Check if the inference with the specified name exists"""
        objs = self.list_inferences(project_name)
        return any(obj.name == inference_name for obj in objs)

    def delete_inference(self, project_name, inference_name, timeout=1200, interval=15):
        """Delete the specified inference and wait until it is removed from the inferences list"""
        def _is_inference_deleted():
            exists = self.is_inference_exists(project_name, inference_name)
            self.show_inferences(project_name)
            return not exists

        LOG.info(f"Delete inference '{inference_name}' from the project '{project_name}'")
        self.box_api_client.inferences.v1_delete_inference(project_name=project_name, inference_name=inference_name)
        waiters.wait(_is_inference_deleted,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout deleting inference '{inference_name}' after {timeout} sec.")
        LOG.info(f"Inference '{inference_name}' is successfully deleted from the project '{project_name}'")

    def get_inference_addresses(self, project_name, inference_name):
        """Get the specified inferece addresses per region
        return: dict(region1=address1, region2=address2, ...)
        """
        inference = self.get_inference(project_name, inference_name)
        addresses = {region: status.address for region, status in (inference.statuses or {}).items()}
        return addresses

    def show_inferences(self, project_name):
        """Show inferences in the specified project"""
        inferences = self.list_inferences(project_name)
        headers = ["Inference name", "Image", "Regions", "Listening port", "Flavor", "Status", "Addresses"]
        inferences_data = [[data.name,
                            data.image,
                            data.regions,
                            data.listening_port,
                            data.flavor.name,
                            data.status,
                            '\n'.join([f"[{region}] {status.address} - {status.status}"
                                       for region, status in (data.statuses or {}).items()]),
                            ]
                           for data in inferences]
        status_msg = tabulate(inferences_data, tablefmt="presto", headers=headers)
        LOG.info(f"Inferences:\n{status_msg}\n")

    def _show_inference_pods_state(self, project_name, inference_name, k8sclient):
        if k8sclient is not None:
            LOG.info("Pods containers status:")
            try:
                pods = k8sclient.pods.list(namespace=project_name)
                inference_pods = [
                    pod for pod in pods
                    if pod.data['metadata']['labels'].get('gcore.com/inference-name') == inference_name]
                for pod in inference_pods:
                    container_statuses = (pod.data['status'] or {}).get('container_statuses') or []
                    for cstat in container_statuses:
                        LOG.info(f"Container state {pod.namespace}/{pod.name} '{cstat.get('name')}':\n"
                                 f"{yaml.dump(cstat.get('state'))}")
            except Exception as e:
                LOG.error(e)

    def wait_inferences_ready(self, project_name, inference_names, k8sclient=None, timeout=1800, interval=15):
        """Wait until the specified inferences statuses becomes 'Active' and all it's addresses become 'Ready'"""
        def _get_inferences_readiness():
            self.show_inferences(project_name)
            endpoints_ready = True
            inferences_status = True
            for inference_name in inference_names:
                self._show_inference_pods_state(project_name, inference_name, k8sclient)
                inference = self.get_inference(project_name, inference_name)
                inferences_status &= (inference.status == 'Active')
                endpoints_ready &= all(status.status == 'Ready'
                                       for region, status in (inference.statuses or {}).items())
            return inferences_status and endpoints_ready

        waiters.wait(_get_inferences_readiness,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting inferences '{inference_names}' readiness after {timeout} sec.")
        LOG.info(f"Inferences '{inference_names}' are 'Active' and all it's addresses are 'Ready'")

    # OpenWeb UI helpers
    def get_openwebui_base_url(self, project_name, mlapp_name):
        """Get openwebui base url"""
        inferences = self.get_inferences_by_mlapp(
            project_name=project_name,
            mlapp_name=mlapp_name)
        ui_base_url = ''
        for inference in inferences:
            if 'open-webui' in inference.name:
                for region, status in (inference.statuses or {}).items():
                    if status.address:
                        ui_base_url = status.address
        assert ui_base_url, f"OpenWeb UI base address not found for Inference '{inference.name}': {inference.statuses}"
        return ui_base_url.rstrip('/')

    def get_openwebui_manager(self, project_name, mlapp_name, email, password):
        openwebui_base_url = self.get_openwebui_base_url(project_name, mlapp_name)
        return OpenWebUIManager(openwebui_base_url, email, password)

    # Common
    def wait_resource_readiness(self, resource_name, list_func, readiness_attr=None, readiness_value=None,
                                timeout=300, interval=5):
        """Wait until the specified resource name is appeared in the resources list with the specified readiness value

        :param resource_name:   str, name of the resource to wait
        :param list_func:       obj, callback function to get list of the resources
        :param readiness_attr:  str, name of the attribute in the resource to check readiness.
                                     If None - skip the readiness check
        :param readiness_value: str, value to wait in the resource's readiness_attr.
                                     If None - wait for any non-empty value
        :param timeout:         int, timeout in seconds of waiting the resource readiness
        :param interval:        int, interval in seconds between resource checks
        """

        def _check_resource_presence():
            resources = list_func()
            match_resources = [r for r in resources if resource_name == r.name]
            if not match_resources:
                LOG.warning(f"Resource name '{resource_name}' not found ...")
                return False

            resource = match_resources[0]
            if readiness_attr:
                if not hasattr(resource, readiness_attr):
                    LOG.warning(f"Resource '{resource}' doesn't have the attribute "
                                f"'{readiness_attr}' to check readiness value")
                    return False
                value = getattr(resource, readiness_attr)
                if readiness_value:
                    is_ready = (value == readiness_value)
                else:
                    is_ready = bool(value)
                LOG.info(f">>> Resource '{resource}' ready: {is_ready}")
                return is_ready
            else:
                LOG.info(f">>> Resource '{resource}' is present")
                return True

        waiters.wait(_check_resource_presence,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting resource name '{resource_name}' readiness after {timeout} sec.")
        LOG.info(f"Resource name '{resource_name}' is '{readiness_value}'")

    def wait_resource_absense(self, resource_name, list_func, timeout=300, interval=5):
        """Wait until the specified resource name is disappeared in the resources list

        :param resource_name:   str, name of the resource to wait
        :param list_func:       obj, callback function to get list of the resources
        :param timeout:         int, timeout in seconds of waiting the resource absense
        :param interval:        int, interval in seconds between resource checks
        """

        def _check_resource_absense():
            resources = list_func()
            match_resources = [r for r in resources if resource_name == r.name]
            if match_resources:
                LOG.warning(f"Resource name '{resource_name}' still exists ...")
                return False

            LOG.info(f">>> Resource '{resource_name}' is absent")
            return True

        waiters.wait(_check_resource_absense,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting resource name '{resource_name}' absense after {timeout} sec.")
        LOG.info(f"Resource name '{resource_name}' is disappeared")


class OpenWebUIManager(object):
    """OpenWebUI manager"""

    def __init__(self, openwebui_base_url, email, password):
        self.openwebui_base_url = openwebui_base_url
        self.email = email
        self.password = password
        self.token = self.get_openwebui_token()

    def get_openwebui_token(self):
        """Get openwebui token"""
        data = {"email": self.email, "password": self.password}
        r = self.request(url="/api/v1/auths/signin", method='POST', body=data, raise_on_error=False)
        if r.status_code == 400 or r.status_code == 401:
            self.create_admin_user()
            r = self.request(url="/api/v1/auths/signin", method='POST', body=data, raise_on_error=True)
        return r.json()["token"]

    @property
    def openwebui_headers(self):
        auth_headers = {"Authorization": f"Bearer {self.token}"}
        return auth_headers

    def create_admin_user(self):
        data = {
            "name": "Admin",
            "email": self.email,
            "password": self.password,
            "role": "admin"  # Assuming the API supports setting role during registration
        }
        try:
            response = self.request(url="/api/v1/auths/signup", method='POST', body=data)

            print(f"Admin user 'self.email' created successfully: {response.json()}")
            return response.json()

        except requests.exceptions.RequestException as e:
            print(f"Error creating admin user: {e}")
            if hasattr(e, 'response') and e.response is not None:
                print(f"Response content: {e.response.text}")
            raise e

    def get_model(self):
        r = self.request(url="/api/models", method='GET', headers=self.openwebui_headers)
        models = [m for m in r.json()['data'] if 'root' in m]  # Use 'root' key to select the uploaded model
        assert models, "No available models found"
        return models[0]

    def get_model_response(self, model, request_content):
        data = {
            "model": model['id'],
            "messages": [{"role": "user", "content": request_content}],
            "temperature": 0,
        }
        r = self.request(url="/api/chat/completions", method='POST', headers=self.openwebui_headers, body=data)
        response_content = r.json()["choices"][0]["message"]["content"]
        return response_content

    @retry(Exception, delay=2, tries=3, logger=LOG)
    def request(self, url, method, headers=None, body=None,
                raise_on_error=True, **kwargs):
        LOG.debug(
            "Sending request to: {}, body: {}, headers: {}, kwargs: {}".format(
                url, body, headers, kwargs))
        r = requests.request(method, self.openwebui_base_url + url,
                             headers=headers, data=json.dumps(body), **kwargs)
        LOG.debug(f"Response: {r.json()}")
        if not r.ok:
            LOG.error(
                "Response code: {}, content: {}, headers: {}".format(
                    r.status_code,
                    r.content,
                    r.headers
                )
            )
            if raise_on_error:
                r.raise_for_status()
                raise requests.HTTPError(r.content, response=r)
        return r
