from si_tests.utils import packaging_version as version
from si_tests import settings

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.kcm_manager import Manager


class ConditionManagerBase(object):
    """Common methods useful for condition checks"""

    # TODO(va4st): Enhance with cluster-related versions support in case if needed

    def __init__(self, kcm_manager: "Manager"):
        self._kcm_manager: "Manager" = kcm_manager

    @property
    def kcm_manager(self) -> "Manager":
        return self._kcm_manager

    @property
    def k8sclient(self):
        return self.kcm_manager.api

    @property
    def k0rdent_version(self):
        """Active k0rdent release version"""
        return self.kcm_manager.release.version

    @property
    def is_enterprise(self):
        return self.kcm_manager.release.is_enterprise


class ConditionManager(ConditionManagerBase):
    """Condition check manager that originally named as 'WorkaroundManager' in si-tests

    Purpose of this class is to keep all the checks for conditions/workarounds in a single place.
    - Each method should return 'True' if the related condition should be applied.
      Please don't add any waiters/actions here. Only checks for versions/providers/cluster objects/etc...
    - Please clear the code from outdated conditions/workarounds after the related tasks are resolved
    """

    def base_lower_oss_1_4_0(self):
        # https://github.com/k0rdent/ksi/issues/543
        # Assuming legacy behavior for any k0rdent version <= 1.1.1 (<1.2.0) for EP and <= 1.3.1(< 1.4.0) for OSS
        ep_condition = self.is_enterprise and version.parse(self.k0rdent_version) <= version.parse('1.1.1')
        # Temporarily set 1.3.1 as the <= limit for OSS to include main branch before 1.4.0 release due
        # to the absence of proper tags
        oss_condition = not self.is_enterprise and version.parse(self.k0rdent_version) <= version.parse('1.3.1')
        # True if any of conditions match
        return oss_condition or ep_condition

    def oss_1_4_0_and_above(self):
        # https://github.com/k0rdent/ksi/issues/516
        # determine api availability due https://mirantis.jira.com/browse/OSPO-15
        # Expect api available for 1.2.0 and later enterprise releases and 1.4.0 and later oss releases
        oss_condition = not self.is_enterprise and version.parse(self.k0rdent_version) >= version.parse('1.4.0')
        # Temporal set strict > 1.1.1 for enterprise to include 1.2.0-alpha1 etc..
        ep_condition = self.is_enterprise and version.parse(self.k0rdent_version) > version.parse('1.1.1')
        return oss_condition or ep_condition


class BootstrapConditionManager(object):
    """Condition checks related to the bootstrap process
    This class is intended to be used during the bootstrap phase using static version gathering through settings due
    kcm releases and management objects not available at that moment.
    """

    @property
    def k0rdent_version(self):
        """Active k0rdent release version from settings"""
        return settings.KCM_CHART_VERSION

    @property
    def is_enterprise(self):
        return True if settings.KCM_SOURCE in ['enterprise', 'custom-enterprise'] else False

    def enterprise_1_2_0_and_above(self):
        return self.is_enterprise and version.parse(self.k0rdent_version) >= version.parse('1.1.1')
