#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.


from kubernetes.client.rest import ApiException

from retry import retry
from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters, exceptions, k8s_utils
from si_tests.managers import kcm_manager

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.bootstrap_manager import KCMBootstrapManager

LOG = logger.logger


class KCMBootstrapCheckManager(object):
    """KCM bootstrap check manager"""

    EXCLUDED_PODS = []
    EXCLUDED_JOBS = []

    def __init__(self, bootstrapmanager: "KCMBootstrapManager"):
        self.bootstrapmanager: "KCMBootstrapManager" = bootstrapmanager
        self._kcm_mgr = None
        self._kcm_namespace = None
        self._kof_namespace = None
        self._mgmt = None

    @property
    def k8sclient(self):
        return self.bootstrapmanager.api

    @property
    def kcm_mgr(self) -> "kcm_manager.Manager":
        if not self._kcm_mgr:
            self._kcm_mgr = kcm_manager.Manager(self.bootstrapmanager.kubeconfig)
        return self._kcm_mgr

    @property
    def mgmt(self):
        if not self._mgmt:
            self._mgmt = self.k8sclient.k0rdent_managements.get(name=settings.KCM_MANAGEMENT_NAME)
        return self._mgmt

    @property
    def kcm_namespace(self) -> "kcm_manager.Namespace":
        if not self._kcm_namespace:
            self._kcm_namespace = self.kcm_mgr.get_namespace(namespace=settings.KCM_NAMESPACE)
        return self._kcm_namespace

    @property
    def kof_namespace(self) -> "kcm_manager.Namespace":
        if not self._kof_namespace:
            self._kof_namespace = self.kcm_mgr.get_namespace(namespace=settings.KSI_KOF_NAMESPACE)
        return self._kof_namespace

    def wait_for_helm_crd_available(self, timeout=1200, interval=10):
        timeout_msg = 'Timeout waiting for helm crd become available'
        try:
            waiters.wait(lambda: self.k8sclient.helmreleases.available,
                         timeout=timeout,
                         interval=interval,
                         timeout_msg=timeout_msg)
        except exceptions.TimeoutError as e:
            raise e

        LOG.debug('Helm crds available in cluster')

    @retry(ApiException, delay=30, tries=10, logger=LOG)
    def wait_for_helmrelease_readiness(self, chart_name, chart_namespace,
                                       timeout=1800, interval=15):
        _hr = self.k8sclient.helmreleases.get(name=chart_name,
                                              namespace=chart_namespace)
        timeout_msg = f'Timeout waiting for helmrelease {chart_namespace}/{chart_name} become ready'
        waiters.wait(lambda: _hr.ready,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    @retry(ApiException, delay=30, tries=10, logger=LOG)
    def wait_for_kcm_helmrelease_readiness(self, timeout=1800, interval=15):
        self.wait_for_helmrelease_readiness(chart_name=settings.KCM_CHART_NAME,
                                            chart_namespace=settings.KCM_NAMESPACE,
                                            timeout=timeout, interval=interval)

    def wait_for_mgmt_object_readiness(self, timeout=1800, interval=15):
        timeout_msg = 'Timeout waiting for KCM mgmt object become ready'
        waiters.wait(lambda: self.mgmt.ready,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    def wait_for_mgmt_object_not_ready(self, timeout=300, interval=5):
        timeout_msg = 'Timeout waiting for KCM mgmt object to become not ready'
        waiters.wait(lambda: not self.mgmt.ready,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    def delete_mgmt_object_and_wait(self, timeout=900, interval=15):
        timeout_msg = f"Management object {self.mgmt.name} still exist"
        if not self.mgmt.exists():
            LOG.info('No management object found. Skipping its removal')
            return
        self.mgmt.delete()
        LOG.info(f"Request to delete Management object {self.mgmt.name} was sent")
        waiters.wait(lambda: not self.mgmt.exists(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info("Mgmt object was removed")

    def uninstall_kcm_and_wait(self, timeout=900, interval=15):
        self.bootstrapmanager.uninstall_kcm()
        timeout_msg = "KCM helm release still present in cluster"
        waiters.wait(lambda: not self.bootstrapmanager.is_kcm_hr_present(),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info('KCM HelmRelease uninstalled successfully')

    def cleanup_crds(self, timeout=300, interval=15):
        def is_crds_present(verbose=True):
            crds_list = self.k8sclient.crds.list_all()
            present = True if crds_list else False
            if verbose and present:
                LOG.info(f"Crds exist in cluster:\n{crds_list}")
            elif verbose and not present:
                LOG.info('No crds found in cluster')
            return present

        self.k8sclient.crds.delete_collection()
        timeout_msg = "Crds still present in cluster after cleanup"
        waiters.wait(lambda: not is_crds_present(),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    def cleanup_grafana_entities(self, timeout=150, interval=10):
        LOG.info('Cleaning up grafana entities (datasources)')
        kof_ns = settings.KSI_KOF_NAMESPACE
        if (not self.k8sclient.grafanadatasources.list(namespace=kof_ns) and
                not self.k8sclient.grafanadashboards.list(namespace=kof_ns)):
            LOG.info("No grafana entities found. Nothing to cleanup")
            return
        self.k8sclient.grafanadatasources.delete_collection(namespace=kof_ns)
        timeout_msg = (f"Grafanadatasources in namespace {kof_ns} "
                       "failed to delete in {timeout} seconds")
        waiters.wait(lambda: not self.k8sclient.grafanadatasources.list(namespace=kof_ns),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        self.k8sclient.grafanadashboards.delete_collection(namespace=kof_ns)
        timeout_msg = (f"Grafanadatasources in namespace {kof_ns} "
                       "failed to delete in {timeout} seconds")
        waiters.wait(lambda: not self.k8sclient.grafanadashboards.list(namespace=kof_ns),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)

    def check_k8s_pods(self, phases=('Running', 'Succeeded'),
                       target_namespaces=None,
                       timeout=settings.WAIT_PODS_READY_TIMEOUT,
                       interval=30, pods_prefix='', check_all_namespaces=False):
        """Check that all pods in target namespaces are in expected phases."""

        if check_all_namespaces:
            # Check all even if expected pods template persists
            target_namespaces = self.k8sclient.namespaces.list_names()
        elif not target_namespaces:
            # Check only expected namespaces based on render of expected pods or all nss if no expected pods defined
            target_namespaces = self.kcm_mgr.get_expected_namespaces()
        self.k8sclient.pods.check_k8s_pods(
            phases=phases,
            target_namespaces=target_namespaces,
            excluded_pods=self.EXCLUDED_PODS,
            excluded_jobs=self.EXCLUDED_JOBS,
            timeout=timeout,
            interval=interval,
            pods_prefix=pods_prefix)

    def check_actual_expected_pods(self,
                                   expected_pods=None,
                                   exclude_pods=None):
        """Compare expected list of pods (which is fetched automatically,
           unless explicitly provided) with actual list of pods in this
           cluster. Comparison is conducted for all namespaces by default"""

        if settings.SKIP_EXPECTED_POD_CHECK:
            LOG.warning("Skipping expected pods checking")
            return

        LOG.info("Checking that all pods and their replicas are in place")

        if not expected_pods:
            if exclude_pods:
                expected_pods, do = self.kcm_mgr.get_expected_objects(
                    exclude_pods=exclude_pods)
            else:
                expected_pods = self.kcm_mgr.expected_pods

        k8s_utils.wait_expected_pods(self.k8sclient, expected_pods=expected_pods)

    def wait_kof_op_ready(self, timeout=600, interval=10):
        timeout_msg = f"{settings.KSI_KOF_OP_CHART_NAME} does not reached status <deployed> for the {timeout} seconds"
        waiters.wait(lambda: self.kcm_mgr.kof.get_kof_operators_chart_status(verbose=True) == 'deployed',
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        assert self.kcm_mgr.kof.is_kof_operators_installed, ("Opentelemetry CRD was not found in the cluster. Please "
                                                             "check installation of kof-operators")
        LOG.info(f"{settings.KSI_KOF_OP_CHART_NAME} successfully installed")

    def uninstall_kof_op_and_wait(self, timeout=900, interval=15):
        LOG.info(f"Starting uninstall process for {settings.KSI_KOF_OP_CHART_NAME}")
        self.kcm_mgr.kof.uninstall_kof_operators()
        timeout_msg = "kof-operators helm release still present in cluster"
        waiters.wait(lambda: not self.kcm_mgr.kof.is_kof_operators_present(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_OP_CHART_NAME} HelmRelease uninstalled successfully")

    def wait_kof_mothership_ready(self, timeout=600, interval=10):
        timeout_msg = f"{settings.KSI_KOF_MS_CHART_NAME} does not reached status <deployed> for the {timeout} seconds"
        waiters.wait(lambda: self.kcm_mgr.kof.get_kof_mothership_chart_status(verbose=True) == 'deployed',
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_MS_CHART_NAME} successfully installed")

    def uninstall_kof_mothership_and_wait(self, timeout=900, interval=15):
        LOG.info(f"Starting uninstall process for {settings.KSI_KOF_MS_CHART_NAME}")
        self.kcm_mgr.kof.uninstall_kof_mothership()
        timeout_msg = "kof-mothership helm release still present in cluster"
        waiters.wait(lambda: not self.kcm_mgr.kof.is_kof_mothership_present(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_MS_CHART_NAME} HelmRelease uninstalled successfully")

    def wait_kof_regional_ready(self, timeout=600, interval=10):
        timeout_msg = f"{settings.KSI_KOF_RE_CHART_NAME} does not reached status <deployed> for the {timeout} seconds"
        waiters.wait(lambda: self.kcm_mgr.kof.get_kof_regional_chart_status(verbose=True) == 'deployed',
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_RE_CHART_NAME} successfully installed")

    def uninstall_kof_regional_and_wait(self, timeout=900, interval=15):
        LOG.info(f"Starting uninstall process for {settings.KSI_KOF_RE_CHART_NAME}")
        self.kcm_mgr.kof.uninstall_kof_regional()
        timeout_msg = "kof-mothership helm release still present in cluster"
        waiters.wait(lambda: not self.kcm_mgr.kof.is_kof_regional_present(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_RE_CHART_NAME} HelmRelease uninstalled successfully")

    def wait_kof_child_ready(self, timeout=600, interval=10):
        timeout_msg = f"{settings.KSI_KOF_CH_CHART_NAME} does not reached status <deployed> for the {timeout} seconds"
        waiters.wait(lambda: self.kcm_mgr.kof.get_kof_child_chart_status(verbose=True) == 'deployed',
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_CH_CHART_NAME} successfully installed")

    def uninstall_kof_child_and_wait(self, timeout=900, interval=15):
        LOG.info(f"Starting uninstall process for {settings.KSI_KOF_CH_CHART_NAME}")
        self.kcm_mgr.kof.uninstall_kof_child()
        timeout_msg = "kof-mothership helm release still present in cluster"
        waiters.wait(lambda: not self.kcm_mgr.kof.is_kof_child_present(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_CH_CHART_NAME} HelmRelease uninstalled successfully")

    def wait_kof_istio_ready(self, timeout=600, interval=10):
        timeout_msg = f"{settings.KSI_KOF_IS_CHART_NAME} does not reached status <deployed> for the {timeout} seconds"
        waiters.wait(lambda: self.kcm_mgr.kof.get_kof_istio_chart_status(verbose=True) == 'deployed',
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_IS_CHART_NAME} successfully installed")

    def uninstall_kof_istio_and_wait(self, timeout=900, interval=15):
        LOG.info(f"Starting uninstall process for {settings.KSI_KOF_IS_CHART_NAME}")
        self.kcm_mgr.kof.uninstall_kof_istio()
        timeout_msg = "kof-mothership helm release still present in cluster"
        waiters.wait(lambda: not self.kcm_mgr.kof.is_kof_istio_present(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_IS_CHART_NAME} HelmRelease uninstalled successfully")

    def wait_kof_collectors_ready(self, timeout=600, interval=10):
        timeout_msg = f"{settings.KSI_KOF_CO_CHART_NAME} does not reached status <deployed> for the {timeout} seconds"
        waiters.wait(lambda: self.kcm_mgr.kof.get_kof_collectors_chart_status(verbose=True) == 'deployed',
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_CO_CHART_NAME} successfully installed")

    def uninstall_kof_collectors_and_wait(self, timeout=900, interval=15):
        LOG.info(f"Starting uninstall process for {settings.KSI_KOF_CO_CHART_NAME}")
        self.kcm_mgr.kof.uninstall_kof_collectors()
        timeout_msg = "kof-mothership helm release still present in cluster"
        waiters.wait(lambda: not self.kcm_mgr.kof.is_kof_collectors_present(verbose=True),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"{settings.KSI_KOF_CO_CHART_NAME} HelmRelease uninstalled successfully")

    def wait_management_backups_sync(self, timeout=600, interval=10):
        timeout_msg = f"Management backups does not exists for the {timeout} seconds"
        waiters.wait(lambda: len(self.k8sclient.k0rdent_management_backups.list()) != 0,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info("Management backups sync completed successfully")

    def wait_velero_backups_sync(self, backup_name, timeout=120, interval=10):
        """Wait for Velero backups to be synced from storage location

        Args:
            backup_name: Name of the specific backup to wait for
            timeout: Maximum time to wait in seconds
            interval: Check interval in seconds
        """
        timeout_msg = f"Velero backup {backup_name} not found after {timeout} seconds"

        def check_backup_exists():
            backups = self.k8sclient.velero_backups.list(namespace="kcm-system")
            backup_names = [b.name for b in backups]
            if backup_name in backup_names:
                LOG.info(f"Found backup {backup_name} in list: {backup_names}")
                return True
            LOG.debug(f"Backup {backup_name} not yet found. Available backups: {backup_names}")
            return False

        waiters.wait(check_backup_exists,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)
        LOG.info(f"Velero backup {backup_name} sync completed successfully")

    @staticmethod
    def check_region_removed(region, timeout=600, interval=15):
        """Check that region is removed"""
        LOG.info(f"Checking region: {region.name}")
        waiters.wait(lambda: not region.exists(verbose=True),
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting for region deletion')
        LOG.info(f"Cluster {region.name} has been deleted")

    def wait_for_default_storageclass(self):
        """Wait for the default storage class to be created if required
        """
        if not settings.KSI_CHECK_FOR_DEFAULT_STORAGECLASS:
            LOG.info("Skip check for default storageclass")
            return

        sc = None
        # return if there's already a default storageclass
        scs = self.k8sclient.storageclass.list_all()
        for sc in scs:
            if sc.is_default:
                LOG.info(f"Found a default storageclass: {sc.name}")
                return

        if scs:
            # existing sc isn't the default, so make the first one so
            sc = scs[0]
        else:
            # install the provisioner
            sc = self.kcm_mgr.wait_for_provisioner()

            # Ino SC exists → create one and make it default
            sc = self.kcm_mgr.create_local_storageclass(
                name=settings.KSI_STORAGECLASS_NAME,
                default_storageclass=settings.KSI_DEFAULT_STORAGECLASS,
                provisioner=settings.KSI_PROVISIONER_NAME
            )

        sc.make_default()
        return
