#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import os
import tempfile

import yaml

from si_tests import settings
from si_tests import logger
from si_tests.clients.k8s.cluster import K8sCluster
from si_tests.managers.helm3_manager import Helm3Manager
from si_tests.managers.bootstrapcheck_manager import KCMBootstrapCheckManager
from si_tests.managers.kcm_manager import Manager
from si_tests.managers.condition_manager import BootstrapConditionManager
from si_tests.utils import utils

LOG = logger.logger


class KCMBootstrapManager(object):
    """
    Manager to control the installation of kcm chart.
    Requires only KUBECONFIG value of ANY working cluster (KinD, Provisioned AWS/Azure/OS/vSphere/etc clds)
    """
    kubeconfig = None

    def __init__(self, kubeconfig=settings.KUBECONFIG_PATH):
        self._api = None
        self.kubeconfig = kubeconfig
        self._helmmanager = None
        self.check = KCMBootstrapCheckManager(self)
        self._kcm_mgr = None
        self.condition = BootstrapConditionManager()

    @property
    def api(self) -> K8sCluster:
        """
            :rtype: cluster.K8sCluster
        """
        if self._api is None:
            if not os.path.isfile(self.kubeconfig):
                raise FileNotFoundError(
                    'kubeconfig file={} not found!'.format(self.kubeconfig))
            self._api = K8sCluster(kubeconfig=self.kubeconfig)
        return self._api

    @property
    def kcm_mgr(self) -> Manager:
        if not self._kcm_mgr:
            self._kcm_mgr = Manager(kubeconfig=self.kubeconfig)
        return self._kcm_mgr

    # NOTE (va4st): use multiple instances of helmmanager. one instance per namespace (we not expect ns customisations)
    # to make possible reusing the common code inside this manager
    @property
    def helmmgr(self) -> "Helm3Manager":
        if self._helmmanager is None:
            self._helmmanager = Helm3Manager(self.kubeconfig, namespace=settings.KCM_NAMESPACE)
        return self._helmmanager

    @staticmethod
    def is_kcm_version_dev(version=settings.KCM_CHART_VERSION):
        return utils.is_version_dev(version)

    def get_kcm_chart_version(self) -> str:
        """
        reutrn example: 'kcm-0.3.0'
        :return:
        """
        if self.is_kcm_hr_present(verbose=True):
            list_hrs = self.helmmgr.list(args=['-A'])
            _required = [x for x in list_hrs if x['name'] == settings.KCM_CHART_NAME]
            if len(_required) > 1:
                LOG.warning(f"Found more than one kcm <{settings.KCM_CHART_NAME}> hr release:"
                            f"\n{_required}")
            return _required[0]['chart']
        return ''

    def is_kcm_installed(self):
        return True if self.api.k0rdent_cluster_deployments.available else False

    @staticmethod
    def get_kcm_chart_values(registry=None,
                             ctl_image_name=settings.KSI_KCM_CTL_NAME_DICT['oss'],
                             telemetry_image_name=settings.KSI_KCM_TELEMETRY_NAME_DICT['oss'],
                             charts_chunk='charts',
                             create_management=True,
                             create_access_management=True,
                             create_release=True,
                             create_templates=True,
                             enable_telemetry=True,
                             use_global_registry=False,
                             registry_insecure=settings.KSI_KCM_CUSTOM_INSECURE_REGISTRY,
                             helm_release_default_timeout=settings.KSI_KCM_HELM_RELEASE_TIMEOUT,
                             registry_creds_secret=None,
                             registry_cert_secret=None,
                             enable_velero_plugins=False,
                             velero_provider=settings.KSI_VELERO_PLUGIN_PROVIDER_NAME,
                             velero_plugin_tag=settings.KSI_VELERO_PLUGIN_PROVIDER_TAG):
        values = {
            'controller': {
                'createManagement': create_management,
                'createAccessManagement': create_access_management,
                'createRelease': create_release,
                'createTemplates': create_templates,
                'enableTelemetry': enable_telemetry,
                'insecureRegistry': registry_insecure
            },
            'regional': {}
        }
        if registry:
            if ctl_image_name:
                image_repo = {
                    'repository': f"{registry}/{ctl_image_name}"
                }
                values['image'] = image_repo
                telemetry_chunk = {
                    'telemetry': {
                        'controller': {
                            'image': {
                                'repository': f"{registry}/{telemetry_image_name}"
                            }
                        }
                    }
                }
                values['regional'] = telemetry_chunk
            values['controller']['templatesRepoURL'] = f"oci://{registry}/{charts_chunk}"
        if helm_release_default_timeout:
            values['controller']['defaultHelmTimeout'] = helm_release_default_timeout
        if use_global_registry:
            values['controller']['globalRegistry'] = registry
        if registry_creds_secret:
            values['controller']['registryCredsSecret'] = registry_creds_secret
        if registry_cert_secret:
            values['controller']['registryCertSecret'] = registry_cert_secret
        if enable_velero_plugins:
            velero_chunk = {
                'initContainers': [
                    {
                        'name': f'velero-plugin-for-{velero_provider}',
                        'image': f'velero/velero-plugin-for-{velero_provider}:{velero_plugin_tag}',
                        'volumeMounts': [
                            {
                                'mountPath': '/target',
                                'name': 'plugins'
                            }
                        ]
                    }
                ]
            }
            values['velero'] = velero_chunk
        return values

    def get_enterprise_config_values(self, registry, k0s_base_url=settings.KSI_K0S_URL):
        """Get values that should be used only in case of custom-enterprise deployment to override repositories.

        :param registry:
        :param k0s_base_url:
        :return:
        """
        # Common enterprise values
        values = {
            'flux2': {
                'helmController': {
                    'image': f"{registry}/fluxcd/helm-controller"
                },
                'sourceController': {
                    'image': f"{registry}/fluxcd/source-controller"
                },
                'cli': {
                    'image': f"{registry}/fluxcd/flux-cli"
                },
            },
            'k0rdent-ui': {
                'image': {
                    'repository': f"{registry}/k0rdent-ui"
                }
            },
        }

        if k0s_base_url:
            k0s = {
                'controller': {
                    'globalK0sURL': k0s_base_url
                }
            }
            utils.merge_dicts(values, k0s)

        # Values, which position depends on k0rdent version
        compatibility_chunk = {
            'cert-manager': {
                'image': {
                    'repository': f"{registry}/jetstack/cert-manager-controller"
                },
                'webhook': {
                    'image': {
                        'repository': f"{registry}/jetstack/cert-manager-webhook"
                    }
                },
                'cainjector': {
                    'image': {
                        'repository': f"{registry}/jetstack/cert-manager-cainjector"
                    }
                },
                'startupapicheck': {
                    'image': {
                        'repository': f"{registry}/jetstack/cert-manager-startupapicheck"
                    }
                }
            },
            'cluster-api-operator': {
                'image': {
                    'manager': {
                        'repository': f"{registry}/capi-operator/cluster-api-operator"
                    }
                }
            },
            'velero': {
                'image': {
                    'repository': f"{registry}/velero/velero"
                }
            }
        }

        if self.condition.enterprise_1_2_0_and_above():
            values['regional'] = compatibility_chunk
        else:
            utils.merge_dicts(values, compatibility_chunk)

        # Values added in 1.2.0
        chunk_120 = {
            "rbac-manager": {
                "image": {
                    "repository": f"{registry}/reactiveops/rbac-manager"
                }
            },
            "datasourceController": {
                "image": {
                    "repository": f"{registry}/datasource-controller"
                }
            }
        }
        if self.condition.enterprise_1_2_0_and_above():
            utils.merge_dicts(values, chunk_120)

        return values

    def install_kcm(self, mode='default'):
        """

        :param mode: default or restore. restore mode sets create flags to false and configures Velero plugin.
        :return:
        """
        values = None
        charts_chunk = 'charts'

        if self.is_kcm_installed():
            LOG.info('KCM chart already present in cluster')
            return
        if settings.KCM_SOURCE == 'enterprise':
            registry = settings.KCM_REPO_DICT['enterprise-repo']
            if mode == 'restore':
                # Do not set registry here to not override chart defaults. It's expected to override only parameters
                values = self.get_kcm_chart_values(create_release=False, create_templates=False,
                                                   create_access_management=False, create_management=False,
                                                   enable_velero_plugins=True)
            elif settings.KSI_OVERRIDE_ENTERPRISE_VALUES:
                values = self.get_kcm_chart_values()
        elif settings.KCM_SOURCE == 'custom-enterprise':
            assert settings.KCM_CUSTOM_REGISTRY, ('KCM_CUSTOM_REGISTRY can not be empty with '
                                                  'KCM_SOURCE=custom-enterprise')
            registry_creds_secret_name = None
            registry_cert_secret_name = None
            registry = settings.KCM_CUSTOM_REGISTRY
            ctl_name = settings.KSI_KCM_CTL_NAME_DICT['enterprise']
            tel_name = settings.KSI_KCM_TELEMETRY_NAME_DICT['enterprise']
            if (settings.KSI_KCM_CUSTOM_REGISTRY_USERNAME is None) \
                    != (settings.KSI_KCM_CUSTOM_REGISTRY_PASSWORD is None):
                if not settings.KSI_KCM_CUSTOM_REGISTRY_USERNAME:
                    raise ValueError("Env KSI_KCM_CUSTOM_REGISTRY_USERNAME is empty, "
                                     "but KSI_KCM_CUSTOM_REGISTRY_PASSWORD filled")
                else:
                    raise ValueError("Env KSI_KCM_CUSTOM_REGISTRY_PASSWORD is empty, "
                                     "but KSI_KCM_CUSTOM_REGISTRY_USERNAME filled")
            cred_condition = (settings.KSI_KCM_CUSTOM_REGISTRY_USERNAME
                              and settings.KSI_KCM_CUSTOM_REGISTRY_PASSWORD)
            cert_condition = (settings.KSI_CUSTOM_REGISTRY_CERT_PATH or
                              settings.KSI_CUSTOM_REGISTRY_CERT_TEXT)
            if settings.KSI_CUSTOM_REGISTRY_CERT_PATH and settings.KSI_CUSTOM_REGISTRY_CERT_TEXT:
                raise ValueError("Variables conflict. KSI_CUSTOM_REGISTRY_CERT_PATH "
                                 "and KSI_CUSTOM_REGISTRY_CERT_TEXT can not have a value at one time")
            # NOTE(both cases require the kcm-system namespace pre-deployed)
            if cred_condition or cert_condition:
                kcm_ns = self.kcm_mgr.get_or_create_namespace(settings.KCM_NAMESPACE)
                if cred_condition:
                    registry_creds_secret_name = f"ksi-registry-cred-{utils.gen_random_string(4)}"
                    kcm_ns.create_registry_auth_secret(registry_creds_secret_name)
                if cert_condition:
                    registry_cert_secret_name = f"ksi-registry-cert-{utils.gen_random_string(4)}"
                    kcm_ns.create_registry_cert_secret(registry_cert_secret_name)
            values = self.get_kcm_chart_values(registry=registry, ctl_image_name=ctl_name,
                                               telemetry_image_name=tel_name, use_global_registry=True,
                                               registry_creds_secret=registry_creds_secret_name,
                                               registry_cert_secret=registry_cert_secret_name)
            if mode == 'restore':
                values = self.get_kcm_chart_values(registry=registry, ctl_image_name=ctl_name,
                                                   telemetry_image_name=tel_name, use_global_registry=True,
                                                   registry_creds_secret=registry_creds_secret_name,
                                                   registry_cert_secret=registry_cert_secret_name,
                                                   create_release=False, create_templates=False,
                                                   create_access_management=False, create_management=False,
                                                   enable_velero_plugins=True)
            ep_values = self.get_enterprise_config_values(registry)
            utils.merge_dicts(values, ep_values)
        elif self.is_kcm_version_dev():
            registry = settings.KCM_REPO_DICT['oss-repo']
            ctl_name = settings.KSI_KCM_CTL_NAME_DICT['oss-ci']
            tel_name = settings.KSI_KCM_TELEMETRY_NAME_DICT['oss-ci']
            charts_chunk = 'charts-ci'
            values = self.get_kcm_chart_values(registry=registry, ctl_image_name=ctl_name,
                                               telemetry_image_name=tel_name, charts_chunk=charts_chunk)
            if mode == 'restore':
                values = self.get_kcm_chart_values(registry=registry, ctl_image_name=ctl_name,
                                                   charts_chunk=charts_chunk, create_release=False,
                                                   create_templates=False, create_access_management=False,
                                                   create_management=False, enable_velero_plugins=True)
        else:
            registry = settings.KCM_REPO_DICT['oss-repo']
            # Do not set registry here to not override chart defaults. It's expected to override only parameters
            if mode == 'restore':
                values = self.get_kcm_chart_values(create_release=False, create_templates=False,
                                                   create_access_management=False, create_management=False,
                                                   enable_velero_plugins=True)

        if values:
            LOG.info(f"Values for kcm installation: \n{yaml.dump(values)}")
        else:
            LOG.info('No values for kcm provided. Using defaults.')

        if settings.KCM_SOURCE in ['enterprise', 'custom-enterprise']:
            chart_path = f"oci://{registry}/{charts_chunk}/{settings.KSI_ENTERPRISE_NAME}"
        else:
            chart_path = f"oci://{registry}/{charts_chunk}/{settings.KCM_CHART_NAME}"

        if values:
            with tempfile.NamedTemporaryFile(mode="w") as values_path:
                yaml.dump(values, values_path)
                self.helmmgr.install_chart(chart_name=settings.KCM_CHART_NAME,
                                           chart_path=chart_path,
                                           version=settings.KCM_CHART_VERSION,
                                           timeout=settings.KSI_KCM_CHART_INSTALL_TIMEOUT,
                                           values_path=[values_path.name])
        else:
            self.helmmgr.install_chart(chart_name=settings.KCM_CHART_NAME,
                                       chart_path=chart_path,
                                       timeout=settings.KSI_KCM_CHART_INSTALL_TIMEOUT,
                                       version=settings.KCM_CHART_VERSION)

    def uninstall_kcm(self):
        if not self.is_kcm_installed():
            LOG.info('KCM chart not present in cluster')
            return

        self.helmmgr.delete_chart(chart_name=settings.KCM_CHART_NAME)

    def is_kcm_hr_present(self, verbose=False) -> bool:
        """Verify that kcm HR persists in cluster"""
        return self.helmmgr.is_release_present(settings.KCM_CHART_NAME, verbose)
