#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s import base
from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


class V1VeleroRestore(base.BaseModel):
    pass


class V1VeleroRestoreList(base.BaseModelList):
    pass


class VeleroRestore(base.K8sNamespacedResource):
    resource_type = 'restore'
    model = V1VeleroRestore

    @property
    def ready(self):
        """Check if the restore is ready/completed"""
        # Read fresh data from the cluster
        obj = self.read(cached=False)
        if obj and hasattr(obj, 'status') and obj.status:
            # status is a dict, so access it as such
            if isinstance(obj.status, dict):
                phase = obj.status.get('phase', '')
            else:
                phase = getattr(obj.status, 'phase', '')
            return phase.lower() == 'completed'
        return False

    @property
    def phase(self):
        """Get the current phase of the restore status"""
        # Read fresh data from the cluster
        obj = self.read(cached=False)
        if obj and hasattr(obj, 'status') and obj.status:
            # status is a dict, so access it as such
            if isinstance(obj.status, dict):
                return obj.status.get('phase', 'Unknown')
            else:
                return getattr(obj.status, 'phase', 'Unknown')
        return 'Unknown'

    @property
    def backup_name(self):
        """Get the backup name this restore is based on"""
        return self.data.get('spec', {}).get('backupName', '')

    @property
    def existing_resource_policy(self):
        """Get the existing resource policy"""
        return self.data.get('spec', {}).get('existingResourcePolicy', '')

    @property
    def included_namespaces(self):
        """Get the list of included namespaces"""
        return self.data.get('spec', {}).get('includedNamespaces', [])

    @property
    def start_timestamp(self):
        """Get the start timestamp of the restore"""
        return self.data.get('status', {}).get('startTimestamp', None)

    @property
    def completion_timestamp(self):
        """Get the completion timestamp of the restore"""
        return self.data.get('status', {}).get('completionTimestamp', None)

    @property
    def warnings(self):
        """Get the number of warnings during restore"""
        return self.data.get('status', {}).get('warnings', 0)

    @property
    def errors(self):
        """Get the number of errors during restore"""
        return self.data.get('status', {}).get('errors', 0)

    @property
    def failure_reason(self):
        """Get the failure reason if restore failed"""
        obj = self.read(cached=False)
        if obj and hasattr(obj, 'status') and obj.status:
            if isinstance(obj.status, dict):
                return obj.status.get('failureReason', '')
            else:
                return getattr(obj.status, 'failureReason', '')
        return ''

    def wait_for_completion(self, timeout=300, interval=10):
        timeout_msg = f"Timeout waiting for Restore {self.namespace}/{self.name} to become completed"

        def _check_ready():
            if self.ready:
                LOG.info(f"Restore {self.namespace}/{self.name} is completed.")
                return True
            elif self.phase in ['Failed', 'PartiallyFailed']:
                e = f"Restore {self.namespace}/{self.name} has failed with reason: {self.failure_reason}"
                LOG.error(e)
                raise Exception(e)
            else:
                LOG.info(f"Restore {self.namespace}/{self.name} is not completed yet. Current phase: {self.phase}")
                return False

        waiters.wait(lambda: _check_ready(), interval=interval, timeout=timeout,
                     timeout_msg=timeout_msg)


class VeleroRestoreManager(base.K8sBaseManager):
    model = V1VeleroRestoreList
    resource_class = VeleroRestore
    resource_group = 'velero.io'
    resource_version = 'v1'
    resource_plural = 'restores'
