#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s import base
from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


class V1VeleroBackup(base.BaseModel):
    pass


class V1VeleroBackupList(base.BaseModelList):
    pass


class VeleroBackup(base.K8sNamespacedResource):
    resource_type = 'backup'
    model = V1VeleroBackup

    @property
    def phase(self):
        # Read fresh data from the cluster
        obj = self.read(cached=False)
        if obj and hasattr(obj, 'status') and obj.status:
            # status is a dict, so access it as such
            if isinstance(obj.status, dict):
                return obj.status.get('phase', 'Unknown')
            else:
                return getattr(obj.status, 'phase', 'Unknown')
        return 'Unknown'

    def completed(self, verbose=True):
        res = self.phase == 'Completed'
        if not res and verbose:
            LOG.warning(f"Backup {self.namespace}/{self.name} is not completed yet. "
                        f"Current phase: {self.phase}")
        return res

    def wait_for_completion(self, timeout=600, interval=10):
        timeout_msg = f"Timeout waiting for Backup {self.namespace}/{self.name} to complete"
        waiters.wait(lambda: self.completed(verbose=True), interval=interval, timeout=timeout,
                     timeout_msg=timeout_msg)


class VeleroBackupManager(base.K8sBaseManager):
    model = V1VeleroBackupList
    resource_class = VeleroBackup
    resource_group = 'velero.io'
    resource_version = 'v1'
    resource_plural = 'backups'
